/*
   pointsys.cxx

   Mark Sibenac
   97-2-25
   Atacama Desert Trek
   Field Robotics Center

   This module is responsible for handling the two pointing motors.

*/

#include "pointsys.h"
#include "point_param.h"
#include "components/amps/amps.h"
#include "components/compass/compass.h"
#include "taskLib.h"
#include "math.h"

#define DEGTORAD(x) ((x)/180.0*PI)

int PointPIDFun (int pPointsys, int, int, int, int, int, int, int, int, int)
{
  return (((Cpointsys *)(pPointsys))->UpdateServos());
}

Cpointsys::Cpointsys()
{
  Dprintf(("Cpointsys::Cpointsys() being constructed\n"));

  m_antPos[0] = 0.0;
  m_antPos[1] = 0.0;
  m_antPos[2] = 0.0;
  
  m_roverPos[0] = 1000.0;
  m_roverPos[1] = 0.0;
  m_roverPos[2] = 0.0;
  
  m_stationPos[0] = 0.0;
  m_stationPos[1] = 0.0;
  m_stationPos[2] = 0.0;

  m_roverAng[0] = 0.0;
  m_roverAng[1] = 0.0;
  m_roverAng[2] = 0.0;

  m_antAng[0] = 0.0;
  m_antAng[1] = 0.0;
  m_antAng[2] = 0.0;

  m_point_time_preset = 1; // update every servo update by default
  m_point_timer = m_point_time_preset; // start timer at preset value

  /* create the amp objects for the two axes */
  AZ_AMP = new Camps(1, AZ_TICKS_REV); // 1 means angular
  EL_AMP = new Camps(1, EL_TICKS_REV); // 1 means angular

  /* Initialize each amp, 0 means no temperature signal */
  AZ_AMP->Init(&AZ_ENCODER, &AZ_CURRENT, &AZ_COMMAND, &AZ_FAULT, 0,
	       &AZ_ENABLE);
  EL_AMP->Init(&EL_ENCODER, &EL_CURRENT, &EL_COMMAND, &EL_FAULT, 0,
	       &EL_ENABLE);

  /* create the point objects for the two axes */
  pAzimuth   = new Cpoint (MOTOR_POINT_AZIMUTH, AZ_AMP);
  pElevation = new Cpoint (MOTOR_POINT_ELEVATION, EL_AMP);
}

Cpointsys::~Cpointsys()
{
  char pString[20];
  int tid;
  
  semFlush (m_semServo);
  semDelete (m_semServo);

  sprintf(pString, "tPSPID");
  if ((tid = taskNameToId(pString)) != -1) taskDelete (tid);

  delete EL_AMP;
  delete AZ_AMP;
  delete pAzimuth;
  delete pElevation;
}

int Cpointsys::Init ()
{
  char pString[20];
  int tid;
  
  Dprintf(("Cpointsys::Init()\n"));

  if (!(m_semServo = semBCreate (SEM_Q_FIFO, SEM_EMPTY)))
    {
      Dprintf(("Cpointsys::Cpointsys() semBCreate() was not able to allocate "
	       "memory.\n"));
      return -1;
    }

  sprintf(pString, "tPSPID");
  if ((tid = taskNameToId(pString)) != -1) taskDelete (tid);
  taskSpawn (pString, PRIORITY_PSPID, VX_SUPERVISOR_MODE | VX_DEALLOC_STACK | 
	     VX_FP_TASK | VX_STDIO, 20000, (FUNCPTR)PointPIDFun, 
	     (int)this, 0, 0, 0, 0, 0, 0, 0, 0, 0);

  pAzimuth->Init (1.0, .005, 0.0, 30000.0, 1000.0, 0.9, 11691);
  pElevation->Init (1.0, .005, 0.0, 30000.0, 1000.0, 0.9, 11691);
  
  return 0;
}

/*
   This routine runs at the pid_frequency. The new transform is calculated at
   a sub_frequency which is an even fraction of the pid_freq. */
int Cpointsys::UpdateServos ()
{
  while (1)
    {
      semTake (m_semServo, WAIT_FOREVER);

      if (--m_point_timer <= 0) // time to calculate the new transform
	{
	  m_point_timer = m_point_time_preset; // reset timer

	  m_roverAng[0] = DEGTORAD(double(compass->getAzimuth())/10.0);
	  m_roverAng[1] = DEGTORAD(double(compass->getPitch())/10.0);
	  m_roverAng[2] = DEGTORAD(double(compass->getRoll())/10.0);

	  Transform(); // calculate the new Antenna position

	  pAzimuth->SetTarget(m_antAng[0]);
	  pElevation->SetTarget(m_antAng[1]);
	}

      pAzimuth->UpdateServo();
      pElevation->UpdateServo();
    }
}

/*
   pid_freq   = servo PID updates
   traj_freq  = servo trajectory updates
   point_freq = New Angle calculations

   This relationship must be followed:
     (pid_freq >= traj_freq) && (pid_freq >= point_freq)

   pid_freq is the main clock source, traj_freq and point_freq are derived
     from it. So even divisible numbers should be used.
     EG:    pid_freq  traj_freq  point_freq
            200       50         25
	    100       25         10
	    100       50         50  (eliminates traj_gen)
*/
int Cpointsys::SetUpdateFreq (int pid_freq, int traj_freq, int point_freq)
{
  if (pid_freq < traj_freq)
    {
      Dprintf(("Cpointsys::SetUpdateFreq() illegal pid_freq=%d and traj_freq="
	       "%d\n", pid_freq, traj_freq));
      return -1;
    }

  if (pid_freq < point_freq)
    {
      Dprintf(("Cpointsys::SetUpdateFreq() illegal pid_freq=%d and point_freq="
	       "%d\n", pid_freq, point_freq));
      return -1;
    }

  if (point_freq <= 0)
    {
      Dprintf(("Cpointsys::SetUpdateFreq() illegal point_freq=%d\n", 
	       point_freq));
      return -1;
    }

  m_point_time_preset = pid_freq / point_freq;
  m_point_timer = m_point_time_preset;

  pAzimuth->SetUpdateFreq   (pid_freq, traj_freq);
  pElevation->SetUpdateFreq (pid_freq, traj_freq);
  return 0;
}

Cpoint *Cpointsys::GetPointPtr (int point_num)
{
  switch (point_num) {

    case MOTOR_POINT_AZIMUTH: return pAzimuth; break;
    case MOTOR_POINT_ELEVATION: return pElevation; break;
    default: Dprintf(("Cpointsys::GetPointPtr() illegal point_num=%d\n",
		      point_num));
  }
  return 0;
}

int Cpointsys::Transform ()
{
  // Declare temporal variables for all the sines and cosines
  double deltaXsv,deltaYsv,deltaZsv; /* delta pos vehicle and station in {G} */
  double xA,yA,zA; /* position of the antenna in {V} */
  double cYaw,sYaw; /* cosines and sine of yaw */
  double cPitch,sPitch;
  double cRoll,sRoll;
  double antYaw,antPitch;
  double xSb,ySb,zSb; /* Coordinates of station in {B} */

  // Asigns values to the antenna position on the rover
  xA = m_antPos[0];
  yA = m_antPos[1];
  zA = m_antPos[2];

  // Asigns actual values to the delta variables
  deltaXsv = m_roverPos[0] - m_stationPos[0];
  deltaYsv = m_roverPos[1] - m_stationPos[1];
  deltaZsv = m_roverPos[2] - m_stationPos[2];

  // Asigns actual values to sines and cosines
  cYaw   = cos( m_roverAng[0] );
  cPitch = cos( m_roverAng[1] );
  cRoll  = cos( m_roverAng[2] );

  sYaw   = sin( m_roverAng[0] );
  sPitch = sin( m_roverAng[1] );
  sRoll  = sin( m_roverAng[2] );

  // Gets station coordinates in {B} frame
  xSb = deltaXsv*(cYaw*cPitch)+
        deltaYsv*(sYaw*cPitch)-
	deltaZsv*sPitch-xA;

  ySb = deltaXsv*(cYaw*sPitch*sRoll-sYaw*cRoll)+
        deltaYsv*(sYaw*sPitch*sRoll+cYaw*cRoll)+
	deltaZsv*(cPitch*sRoll)-yA;

  zSb = deltaXsv*(cYaw*sPitch*cRoll+sYaw*sRoll)+
        deltaYsv*(sYaw*sPitch*cRoll-cYaw*sRoll)+
	deltaZsv*(cPitch*cRoll)-zA;

  // Calculates antenna yaw and pitch
  antYaw   = -atan2(xSb,ySb);
  antPitch = atan2(zSb,sqrt(xSb*xSb+ySb*ySb));

  // Asigns the value to the actual output matrix
  m_antAng[0] = antYaw;
  m_antAng[1] = antPitch;
  

    // If Error, return non-zero
    return (0);
}



