/*
  device_hardware.cxx

  97-1-17
  Mark Sibenac
  Atacama Desert Trek
  Field Robotics Center

  This is code that should be run imediately after loading the devices. It
  allocates memory for each device and initializes them.

  This file must be modified if the physical hardware changes.
*/

#include "common/nomad_types.h"
#include "device_hardware.h"
#include "interfaces/encoder/encoder.h"
#include "interfaces/dac/dac.h"
#include "interfaces/adc/adc.h"
#include "interfaces/di/di.h"
#include "interfaces/do/do.h"
#include "interfaces/relay/relay.h"

// These are the declarations of the board pointers
Cvmio28 *vmio28[MAX_VMIO28_BOARDS];
Cvmio14 *vmio14[MAX_VMIO14_BOARDS];
Cvadc   *vadc[MAX_VADC_BOARDS];
Cvadm   *vadm[MAX_VADM_BOARDS];
//Cvcpu40 *vcpu40[MAX_VCPU40_BOARDS];

// Device pointer mappings from board number
Cdevices *BoardNumToDevicePtr(Board_Type board_num)
{
  switch (board_num)
    {
      case VMIO28_BOARD_1 : return vmio28[0];
      case VMIO28_BOARD_2 : return vmio28[1];
      case VMIO14_BOARD_1 : return vmio14[0];
      case VMIO14_BOARD_2 : return vmio14[1];
      case VADM_BOARD_1   : return vadm[0];
      case VADC_BOARD_1   : return vadc[0];
      case VADC_BOARD_2   : return vadc[1];
//      case VCPU40_BOARD_1 : return vcpu40[0];
      default : Dprintf(("BoardNumToDevicePtr() illegal board_num = %d\n",
			 board_num));
		return ((Cdevices *)0x0);
    }
}


// must be called before any accesses to hardware devices
int InitHardware()
{
  vmio28[0] = new Cvmio28(VMIO28_1_ADDRESS, VMIO28_1_BAY);
  vmio28[1] = new Cvmio28(VMIO28_2_ADDRESS, VMIO28_2_BAY);
  vmio14[0] = new Cvmio14(VMIO14_1_ADDRESS, VMIO14_1_BAY, VMIO14_1_PORT);
  vmio14[1] = new Cvmio14(VMIO14_2_ADDRESS, VMIO14_2_BAY, VMIO14_2_PORT);
  vmio14[2] = new Cvmio14(VMIO14_3_ADDRESS, VMIO14_3_BAY, VMIO14_3_PORT);
  vmio14[3] = new Cvmio14(VMIO14_4_ADDRESS, VMIO14_4_BAY, VMIO14_4_PORT);
  vadc[0]   = new Cvadc(VADC_1_ADDRESS, VADC_1_BAY, VADC_1_SINGLEDIFF);
  vadc[1]   = new Cvadc(VADC_2_ADDRESS, VADC_2_BAY, VADC_2_SINGLEDIFF);
//  vadm[0]   = new Cvadm(VADM_1_ADDRESS, VADM_1_SINGLEDIFF);
  vadm[0]   = new Cvadm(VADM_1_ADDRESS, 0);
//  vcpu40[0] = new Cvcpu40();

  return (0);
}

int KillHardware ()
{
  delete vmio28[0];
  delete vmio28[1];
  delete vmio14[0];
  delete vmio14[1];
  delete vmio14[2];
  delete vmio14[3];
  delete vadc[0];
  delete vadc[1];
  delete vadm[0];

  return 0;
}

// Must be called before any accesses to interfaces.
// Must be called after initializing interfaces, calling InitInterfaces()
// Must be called after InitHardware()
int RegisterHardware()
{
  Cencoder::Register(VMIO28_1_NUM_ENC, BoardNumToDevicePtr(VMIO28_BOARD_1));
  Cencoder::Register(VMIO28_2_NUM_ENC, BoardNumToDevicePtr(VMIO28_BOARD_2));

  Cdac::Register(VADM_NUM_DAC, BoardNumToDevicePtr(VADM_BOARD_1));

  Cadc::Register(VADM_NUM_ADC_DIFF, BoardNumToDevicePtr(VADM_BOARD_1));
  Cadc::Register(VADC_NUM_ADC_DIFF, BoardNumToDevicePtr(VADC_BOARD_2));
  Cadc::Register(VADC_NUM_ADC_SINGLE, BoardNumToDevicePtr(VADC_BOARD_1));

  Cdi::Register(VADM_NUM_DI, BoardNumToDevicePtr(VADM_BOARD_1));

  Cdo::Register(VADM_NUM_DO, BoardNumToDevicePtr(VADM_BOARD_1));

  Crelay::Register(VADM_NUM_RELAY, BoardNumToDevicePtr(VADM_BOARD_1));

  return 0;
}
