/*
   drive.cxx

   Mark Sibenac
   97-1-28
   Atacama Desert Trek
   Field Robotics Center

   This module controls the coordinated motion and velocity servo loops on the
   four motors. It handles the PID loops and the trajectory generator
   algorithms.

   Each motor can be in one of the following states: {INHIBITED, SERVOING,
   OVERCURRENT, OVERTEMP, SERVO_ERROR}

*/

#include "drive.h"
#include "taskLib.h"

Cdrive *drive; // pointer to array of drive objects

// not angular and 1024 ticks/rev
Cdrive::Cdrive (int axis_number, Camps *amp_in) : m_pid(amp_in, 0), m_trajectory(amp_in, 0)  // 0 = not angular
{
  Dprintf(("Cdrive::Cdrive() creating a new drive object %d\n", axis_number));

  if ((axis_number < 0) || (axis_number > 3))
    {
      Dprintf(("Cdrive::Cdrive() illegal axis_number=%d\n", axis_number));
      return;
    }

  if (!amp_in)
    {
      Dprintf(("Cdrive::Cdrive() amp=%x is not initialized!!!\n", 
	       (unsigned int)amp_in));
      return;
    }

  m_axis_number = axis_number;
  m_amp = amp_in;
  
  m_trajectory.SetParams(MAX_MOTOR_TICKS_SEC, DEFAULT_MOTOR_ACCEL);

  m_traject_time_preset = 1; // update every servo update by default
  m_traject_timer = m_traject_time_preset; // start timer at preset value
  m_traject_current_des_pos = m_trajectory.GetNewPos();
  m_traject_current_des_vel = int(m_trajectory.GetCurrentVel());

  m_fDesVel = 0.0;
  m_fCurrentVel = 0.0;
  m_fAccel = DEFAULT_DRIVE_ACCEL;
}

Cdrive::~Cdrive ()
{
  m_state = INHIBITED;
  m_amp->Disable();
  m_amp->SetCommand(0.0);
}

// Init
//  This initializes all of the drive member variables to a known state
//   motor_state = INHIBITED
//   
int Cdrive::Init (double Kp, double Ki, double Kd, double I_limit,
		  double MaxPosErr, double MaxAmp, double Accel)
{
  Dprintf(("Cdrive::Init()\n"));

  m_state = INHIBITED;
  m_amp->Reset();
  m_amp->SetMaxCommand(MaxAmp);
  m_trajectory.SetAccel(Accel);
  m_pid.Init(Kp, Ki, Kd, I_limit, MaxPosErr);
  return 0;
}

// UpdateServo
//  This routine is called from a mother process that runs at the servo update
//  rate. It updates the trajectory generator if the timer hits 0. It always
//  updates the PID controller.
int Cdrive::UpdateServo ()
{
  m_amp->UpdateEncoder();
  m_amp->UpdateCurrentMon();
  m_amp->UpdateFaults();

  if (--m_traject_timer <= 0)
    {
      m_traject_timer = m_traject_time_preset; // reset timer
      m_traject_current_des_pos = m_trajectory.CalcNewPos();
      m_traject_current_des_vel = int(m_trajectory.GetCurrentVel());
      m_pid.SetDesPos (m_traject_current_des_pos);
    }

  m_pid.ServoLaw(m_traject_current_des_vel);

  if (m_state != SERVOING) return 1;

  m_amp->SetCommand(m_pid.GetCommand());
  return 0;
}

int Cdrive::ChangeState (motor_state_t state)
{
  switch (state)
    {
      case INHIBITED:
           m_amp->Disable();
	   m_amp->SetCommand(0.0);
	   break;
      case SERVOING:
	   m_pid.Reset();
	   m_trajectory.Reset();
	   m_amp->SetCommand(0.0);
	   m_amp->Enable();
	   break;
      case OVERCURRENT:
      case OVERTEMP:
      case SERVO_ERROR:
	   m_amp->Disable();
	   m_amp->SetCommand(0.0);
	   break;
    }

  m_state = state;
  return 0;
}

/*
   Set's the velocity of the wheel in radians per second.
*/
int Cdrive::SetVelocity (double rads_sec)
{
  if (rads_sec < -MAX_DRIVE_VEL)
    { rads_sec = -MAX_DRIVE_VEL; }
  else if (rads_sec > MAX_DRIVE_VEL)
    { rads_sec =  MAX_DRIVE_VEL; }

  m_fDesVel = rads_sec;
  return 0;
}

/*
   Sets the PID and Trajectory Generator's update frequency variables.
   Makes sure amp is in the INHIBITED state before the change is made.
*/
int Cdrive::SetUpdateFreq (int pid_freq, int trajectory_freq)
{
  motor_state_t state = m_state;
  
  ChangeState (INHIBITED); // change to an inhibited state to avoid problems
  m_pid.SetUpdateFreq (pid_freq);
  m_trajectory.SetFreq (trajectory_freq);
  m_traject_time_preset = pid_freq/trajectory_freq;
  m_traject_timer = m_traject_time_preset;
  
  ChangeState (state); // go back to previous state
  return 0;
}
