/*****************************************************************************
 *       $Id: serialRS232.c,v 1.1 1996/02/10 03:08:35 jorgen Exp $
 * $Revision: 1.1 $
 *     $Date: 1996/02/10 03:08:35 $
 *   $Author: jorgen $
 *    $State: Exp $
 *   $Locker:  $
 *
 *                           ******************************
 *                           *      Copyright (C) 1995    *   
 *                           * Carnegie Mellon University *
 *                           *      All Rights Reserved   *
 *                           ******************************
 *
 * PROJECT:	Automated Loading System (ALS)
 * 
 * FILE:	serialRS232.c
 *
 * DESCRIPTION:	Routines for interfacing to an RS232 serial line.
 *
 * HISTORY:
 *
 * $Log: serialRS232.c,v $
 * Revision 1.1  1996/02/10  03:08:35  jorgen
 * Initial release.
 *
 *****************************************************************************/

/*****************************************************************************
 * INCLUDES
 *****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include "serialRS232.h"

/*****************************************************************************
 * DEFINES
 *****************************************************************************/
#define DEBUG 0

/*****************************************************************************
 * STRUCTS & TYPEDEFS
 *****************************************************************************/

/*****************************************************************************
 * PUBLIC VARIABLES
 *****************************************************************************/

/*****************************************************************************
 * GLOBAL VARIABLES
 *****************************************************************************/

/*****************************************************************************
 *  FUNCTION:    serialLookupSpeed
 *  DESCRIPTION: Convert an integer speed into the right bit-mask 
 *  INPUTS:
 *  OUTPUTS: 
 *  EXCEPTIONS: 
 *  DESIGN: 
 *  NOTES: 
 *****************************************************************************/
#ifndef VXWORKS
static int serialLookupSpeed(int speed)
{
    int s;

    switch (speed) {
	case 1200: s = B1200; break;
	case 1800: s = B1800; break;
	case 2400: s = B2400; break;
	case 4800: s = B4800; break;
	case 9600: s = B9600; break;
	case 19200: s = B19200; break;
	case 38400: s = B38400; break;
	default:
	    fprintf(stderr,
		"Illegal baud rate %d specified, setting to 9600\n", speed);
	    s = B9600;
	    break;
    }
    return(s);
}
#endif

/*****************************************************************************
 *  FUNCTION:    serialInit
 *  DESCRIPTION: Initialize serial communications to a particular port. 
 *  INPUTS:
 *  OUTPUTS:     Returns the file descriptor, or -1 for failure.
 *  EXCEPTIONS: 
 *  DESIGN: 
 *  NOTES: 
 *****************************************************************************/
int serialInit(char *portName, int speed, PARITY_ENUM parity)
{
#ifndef VXWORKS
   struct sgttyb sg;
#endif
   int fd;

   /* Open the port */
   if (DEBUG) printf("Opening port\n");
   if ((fd = open(portName, O_RDWR, 0)) < 0) {
      fprintf(stderr, "Could not open serial port %s!\n", portName);
      return(-1);
   }
   if (DEBUG) printf("Port opened\n");

#ifndef VXWORKS
   if (ioctl(fd, TIOCEXCL, NULL) < 0) {
      fprintf(stderr, "Can't set exclusive access for %s!\n", portName);
      close(fd);
      return(-1);
   }
#endif


#ifndef VXWORKS
    /* Set the parameters for transparent operation */
    ioctl(fd, TIOCGETP, &sg);
    sg.sg_flags |= O_RAW;
    sg.sg_flags &= ~O_ECHO;
    if (parity == parity_ODD) {
       printf("Setting 7-O-1...\n");
       sg.sg_flags |= O_ODDP;
    }
    if (parity == parity_EVEN) {
       printf("Setting 7-E-1...\n");
       sg.sg_flags |= O_EVENP;
    }
    if (parity == parity_NONE) {
       printf("Setting 8-N-1...\n");
    }
    if (speed != 0) sg.sg_ispeed = sg.sg_ospeed = serialLookupSpeed(speed);
    ioctl(fd, TIOCSETP, &sg);
#else
    if (DEBUG) printf("IOCTL call 1\n");
    ioctl(fd, FIOSETOPTIONS, OPT_RAW);
    if (DEBUG) printf("IOCTL call 2\n");
    ioctl(fd, FIOBAUDRATE, &speed);
    if (DEBUG) printf("Done IOCTL calls\n");
#endif

    return(fd);
}


/*****************************************************************************
 *  FUNCTION:    serialNumChars
 *  DESCRIPTION: Return the number of characters waiting to be read from the
 *               port.
 *  INPUTS:      fd - the file descriptor
 *  OUTPUTS:     the num chars to be read
 *  EXCEPTIONS: 
 *  DESIGN: 
 *  NOTES: 
 *****************************************************************************/
int serialNumChars(int fd)
{
    int count;

    ioctl(fd, FIONREAD, &count);
    return(count);
}


/*****************************************************************************
 *  FUNCTION:    serialFlushBuffer
 *  DESCRIPTION: Flush the port's input buffer
 *  INPUTS:      fd - the file descriptor
 *  OUTPUTS: 
 *  EXCEPTIONS: 
 *  DESIGN: 
 *  NOTES: 
 *****************************************************************************/
void serialFlushBuffer(int fd)
{
#ifndef VXWORKS
    tcflush(fd, TCIFLUSH);
#else
    ioctl(fd, FIORFLUSH, NULL);
#endif
}


/*****************************************************************************
 *  FUNCTION:    serialPutBuf
 *  DESCRIPTION: Send a buffer to the port
 *  INPUTS:      fd - the file descriptor
 *               buf - the buffer
 *		 count - the count
 *  OUTPUTS: 
 *  EXCEPTIONS: 
 *  DESIGN: 
 *  NOTES: 
 *****************************************************************************/
void serialPutBuf(int fd, char *buf, int count)
{
    write(fd, buf, count);
}

/*****************************************************************************
 *  FUNCTION:    
 *  DESCRIPTION: Read the specified number of characters from the port.
 *  INPUTS:      fd - the file desciptor
 *               buffer - the buffer
 *		 num_chars - the count
 *		 timeout - the timeout value
 *  OUTPUTS:     Returns the number of characters read.
 *  EXCEPTIONS: 
 *  DESIGN: 
 *  NOTES: 
 *  If timeout > 0, return after many milliseconds (or as soon as the	
 *		   num_chars characters have been read).		
 *  timeout = 0, return right away with as many characters as are	
 *		   available, but no more than num_chars.		
 *  If timeout < 0, block until num_chars characters have been read.	
 *****************************************************************************/
int serialGetBuf(int fd, char *buffer, int num_chars, int timeout)
{
    int count;

    if (num_chars <= 0) 
      return 0;

    if (timeout < 0) {
	count = read(fd, buffer, num_chars);
	return(count);
    }

    if (timeout == 0) {
	ioctl(fd, FIONREAD, &count);
	if (count > num_chars) count = num_chars;
	count = read(fd, buffer, count);
	return(count);
    }

    if (timeout > 0) {
       return (0);
    }

    return(0);
}


/*****************************************************************************
 *  FUNCTION:    raw
 *  DESCRIPTION: Put stdin in raw I/O mode 
 *  INPUTS:
 *  OUTPUTS: 
 *  EXCEPTIONS: 
 *  DESIGN: 
 *  NOTES:       
 *****************************************************************************/
#ifndef VXWORKS
void raw(void)
{
   struct termios ts;

   ioctl( 0, TCGETS, &ts);
   ts.c_lflag = ISIG | ECHO;
   ioctl( 0, TCSETS, &ts);
}
#endif
  
/*****************************************************************************
 *  FUNCTION:    kbhit
 *  DESCRIPTION: Determines if a keyboard key has been hit.
 *  INPUTS:
 *  OUTPUTS: 
 *  EXCEPTIONS: 
 *  DESIGN: 
 *  NOTES:       This only works if STDIN has ben placed in raw I/O mode
 *****************************************************************************/
int kbhit()
{
  long len;

  if (ioctl(0, FIONREAD, &len)) 
     printf("kbhit error\n");
  if (len > 0) 
     return 1;
  else 
     return 0;
}


/*****************************************************************************
 *  FUNCTION:    serialGetChar
 *  DESCRIPTION: Returns a charcter read from the serial line.
 *  INPUTS:
 *  OUTPUTS: 
 *  EXCEPTIONS: 
 *  DESIGN: 
 *  NOTES: 
 *****************************************************************************/
unsigned char serialGetchar(int fd)
{
  unsigned char n;

  serialGetBuf(fd, &n, 1, -1);
  return n;
}

