/*
  triwg.cxx
  
  Mark Sibenac
  96-6-7
  Field Robotics Center
  Lunar Rover Terrestrial Prototype
  TRIWG control of Ambler Amps.

  Using VADM10 with 64 Pin connector module and interfaced to a DB-15

  Left Command  - AO0, DB1, c9, P17, White
  Right Command - A02, DB13, c10, P19, Brown
  Command Ground- OGND, DB3, a7, P14, Orange
  Enable        - R0b, DB7, c2, P3, Black
  Digital Ground- R0c, DB8, c3, P5, Red
  
  call initTriwg() to initialize things to a known state.
  */

#include <vxWorks.h>
#include <shellLib.h>
#include <stdio.h>
#include <stdlib.h>
#include "triwg.h"
#include "vadm/vadm.h"
#include "joy.h"

Vadm vadm;
int nLeft_command, nRight_command;
int nTurn_gain, nFwd_gain;
int nJog;
int nArray_max;
int nArray_pos;

void initTriwg(void)
{
  vadm.init(DIFFERENTIAL, IN_10_10, TWOS_COMP);
  disableAmps();
  nLeft_command = 0; nRight_command = 0;
  nTurn_gain = 0; nFwd_gain = 0;
  updateSpeed();
  nJog = 10;
  setvbuf(stdin, 0, _IONBF, 0);
}

void enableAmps(void)
{
  vadm.setRelays(1);
}

void disableAmps(void)
{
  vadm.setRelays(0);
}
  
// -100 --> +100, backward-->forward
int forwardPercent(int nPercent)
{
  if ((nPercent < -100) || (nPercent > 100))
    {
      printf("TRIWG::forwardPercent Error... nPercent = %d\n", nPercent);
      return 0xffff;
    }
  
  nFwd_gain = nPercent;
  updateSpeed();

  return nFwd_gain;
}

// -100 --> +100, left-->right
int turnPercent(int nPercent)
{
  if ((nPercent < -100) || (nPercent > 100))
    {
      printf("TRIWG::turnPercent Error... nPercent = %d\n", nPercent);
      return 0xffff;
    }
  
  nTurn_gain = nPercent;
  updateSpeed();

  return nTurn_gain;
}

void updateSpeed(void)
{
  double dLeft_command_voltage, dRight_command_voltage;

  dLeft_command_voltage = ((double)nFwd_gain/(double)10.0) * ((double)(100-nTurn_gain)/(double)200.0);

  dRight_command_voltage = ((double)nFwd_gain/(double)10.0) * ((double)(nTurn_gain+100)/(double)200.0);

  nLeft_command = (int)(dLeft_command_voltage * 204.8);
  nRight_command = (int)(dRight_command_voltage * 204.8);

  if (nLeft_command >= 2047) nLeft_command = 2047;
  if (nLeft_command <= -2048) nLeft_command = -2048;
  if (nRight_command >= 2047) nRight_command = 2047;
  if (nRight_command <= -2048) nRight_command = -2048;

  printf("nFwd_gain = %d%%,\tnTurn_gain = %d%%\n", nFwd_gain,
	 nTurn_gain);
  printf("dLeft_command_voltage = %f,\tnLeft_command = %d\n", 
	 dLeft_command_voltage, nLeft_command);
  printf("dRight_command_voltage = %f,\tnRight_command = %d\n", 
	 dRight_command_voltage, nRight_command);

  vadm.setDAC(0,nLeft_command);
  vadm.setDAC(2,nRight_command);
}

void showMenu()
{
  puts("\n");
  puts("f - foward by jog amount");
  puts("b - backward by jog amount");
  puts("n - neutral speed");
  puts("l - left by jog amounr");
  puts("r - right by jog amount");
  puts("c - center steering");
  puts("j - change jog amount");
  puts("e - enable amps");
  puts("d - disable amps");
  puts("s - Use Serial JoyStick to control Amps");
  puts("t - Test Serial JoyStick without DAC");
  puts("m - show menu");
  puts("q - quit this menu crap");
  puts("choice? ");
}

void menu(void)
{
  char c;

  initTriwg();
  showMenu();

  c = 0;
  while (c != 'q')
    {
    GETCHARACTER:
      c = getchar();

      switch (c)
	{
	case 'f': forwardPercent(nFwd_gain+nJog); break;
	case 'b': forwardPercent(nFwd_gain-nJog); break;
	case 'n': forwardPercent(0); break;
	case 'l': turnPercent(nTurn_gain+nJog); break;
	case 'r': turnPercent(nTurn_gain-nJog); break;
	case 'c': turnPercent(0); break;
	case 'j': changeJog(); break;
	case 'e': enableAmps(); break;
	case 'd': disableAmps(); break;
	case 's': goJoy(); showMenu(); break;
	case 't': testSerialJoy(); break;
	case 'm': showMenu(); break;
	case 'q': break;
	case '\n': goto GETCHARACTER;
	default: puts("ERROR\n");
	}
    }
}

void changeJog(void)
{
  char str[80];

  getchar();
  printf("nJog = %d ===> ", nJog);
  gets(str);
  nJog = atoi(str);
  printf("nJog = %d\n", nJog);
}

// assumes connected to /tyCo/1
void testSerialJoy(void)
{
  joyMain("/tyCo/1");
  showMenu();
}

void clearArray(double *pArray)
{
  int n;

  for (n=0; n < nArray_max; n++)
    pArray[n] = 0.0;
}

void insertArray(double *pArray, double dVal)
{
  if (dVal < 0.0)
    {
      if (dVal >= -2.0)
	dVal = 0.0;
      else
	dVal = (dVal + 2.0) * 1.33333;
    }
  else
    {
      if (dVal <= 2.0)
	dVal = 0.0;
      else
	dVal = (dVal - 2.0) * 1.33333;
    }

  pArray[nArray_pos] = dVal;
}

void getAvg(double *pArray, double &pVal)
{
  int n;
  double dSum;

  dSum = 0.0;
  for (n=0; n<nArray_max; n++)
    dSum += pArray[n];

  pVal = (dSum/nArray_max);
}

// assumes connected to /tyCo/1
void goJoy(void)
{
  joystruct joy;
  double dLeft, dRight;
  double dLeft_array[30], dRight_array[30];
  int nLeft_trigger, nLast;

  fd = 0;
  nLeft_trigger = 0; nLast = 0;
  nArray_max = 30;
  nArray_pos = 0;
  clearArray(dLeft_array);
  clearArray(dRight_array);
  joy.left_trigger = 0; joy.right_trigger = 0;
  strcpy(pPort, "/tyCo/1");

  printf("To stop, press ctrl-c or left trigger only ten times in a row.\n");
  printf("To go, press both triggers.\n");
  printf("Okay, play with it...\n\n");

  while(nLeft_trigger < 10)
    {
      read_joystick(&joy);
      if (joy.left_trigger && !joy.right_trigger)
	{
	  if (!nLast)
	    {
	      nLeft_trigger++;
	      nLast = 1;
	    }
	}
      else 
	{
	  nLast = 0;
	}
      
      if (joy.left_trigger && joy.right_trigger)
	{
	  nLeft_trigger = 0;

	  if ((joy.left_y != 0.0) && ( joy.right_y != 0.0))
	    {	    
	      dLeft = (joy.left_y-50.0)/3.75;
	      dRight = (joy.right_y-50.0)/3.75;
	    }

	  if (++nArray_pos >= nArray_max)
	    nArray_pos = 0;
	  insertArray(dLeft_array, dLeft);
	  insertArray(dRight_array, dRight);
	  
	  getAvg(dLeft_array, dLeft);
	  getAvg(dRight_array, dRight);

	  nLeft_command = (int)(dLeft * 204.8);
	  nRight_command = (int)(dRight * 204.8);
	  
	  if (nLeft_command >= 2047) nLeft_command = 2047;
	  if (nLeft_command <= -2048) nLeft_command = -2048;
	  if (nRight_command >= 2047) nRight_command = 2047;
	  if (nRight_command <= -2048) nRight_command = -2048;

	  vadm.setDAC(0,nLeft_command);
	  vadm.setDAC(2,nRight_command);
	  if ((nLeft_command == 0) && (nRight_command == 0))
	    disableAmps();
	  else
	    {
	      enableAmps();
	      /*	      printf("%d %d\n", nLeft_command, nRight_command);*/
	    }
	}
      else
	{
	  if (++nArray_pos >= nArray_max)
	    nArray_pos = 0;
	  insertArray(dLeft_array, 0.0);
	  insertArray(dRight_array, 0.0);
	  
	  disableAmps();
	  vadm.setDAC(0,0);
	  vadm.setDAC(2,0);
	}
    }

  disableAmps();
  vadm.setDAC(0,0);
  vadm.setDAC(2,0);

  close_joystick();
}

extern "C" void testInit(void)
{

  while(1)
    {
      printf("It worked!\n");
    }
}
