/*
   test_point.cxx

   Mark Sibenac
   97-2-5
   Atacama Desert Trek
   Field Robotics Center

   This tests the pointing module.

*/

#include "test_point.h"
#include "scopeprobe.h"
#include "taskLib.h"
#include "sysLib.h"
#include "components/pid/pid.h"
#include "components/amps/amps.h"
#include "components/traject/traject.h"
#include "vxLib.h"
#include "usrLib.h"
#include "semLib.h"
#include "logLib.h"

#define AUX_CLK_RATE_MIN 30
#define AUX_CLK_RATE_MAX 3000

int g_freqp=200; // global frequency

int point_servoISR (int)
{
  semGive (pAzimuth->m_semUpdate);
  semGive (pElevation->m_semUpdate);

  return 0;
}

void point_startISR (int freq)
{
  if ((freq < AUX_CLK_RATE_MIN) || (freq > AUX_CLK_RATE_MAX))
    {
      printf("point_startISR() illegal freq=%d. %d<freq<%d\n", freq, 
	     AUX_CLK_RATE_MIN, AUX_CLK_RATE_MAX);
      return;
    }
  
  taskPrioritySet(taskNameToId("tShell"), 50);
  
  if (sysAuxClkConnect ((FUNCPTR)point_servoISR, 0))
    {
      printf("AuxClock could not connect\n");
      return;
    }

  if (!sysAuxClkRateSet (freq))
    Dprintf(("AuxClk Connected at freq=%d\n", freq));

  sysAuxClkEnable  ();
}

void point_stopISR ()
{
  sysClkDisable ();
}

int point_Init        (int freq)
{
  g_freqp = freq;

  printf("Initializing pointing modules\n");
  pAzimuth->Init();
  pElevation->Init();

  printf("Initializing trajectory generator\n");
  pAzimuth->m_trajectory.SetParams(40000.0, 30000.0, g_freqp);
  pElevation->m_trajectory.SetParams(40000.0, 30000.0, g_freqp);

  printf("Spawning Scope at 50 Hz\n");
  sp((FUNCPTR)point_Scope, 0, 50, 3, 4, 5, 6, 7, 8, 9); // Azimuth
  sp((FUNCPTR)point_Scope, 1, 50, 3, 4, 5, 6, 7, 8, 9); // Elevation

  printf("Starting ISR at %d Hz\n", freq);
  point_startISR (freq);

  return 0;
}

int point_ChangeState (int point_num, motor_state_t value)
{
  return (point[point_num]->ChangeState(value));
}

int point_GetState    (int point_num)
{
  register motor_state_t state=point[point_num]->GetState();

  printf("State = %d\n", state);
  return state;
}

int point_GetPID      (int point_num)
{
  register Cpid *pid = &point[point_num]->m_pid;

  printf("point%d Values:\n", point_num);
  printf("\tKp = %f\n", pid->GetKp());
  printf("\tKi = %f\n", pid->GetKi());
  printf("\tKd = %f\n", pid->GetKd());
  printf("\tIlimit = %f\n", pid->GetILimit());
  printf("\tMaxPosErr = %f\n", pid->GetMaxPosErr());
  printf("\tCommand = %f\n", pid->GetCommand());
  printf("\tDesPos = %f\n", pid->GetDesPos());
  printf("\tCurPos = %f\n", pid->GetCurPos());
  return 0;
}

int point_GetAmp      (int point_num)
{
  register Camps *Amp = point[point_num]->m_amp;

  printf("point%d Amp Status:\n", point_num);
  printf("\tPosition = %ld\n", Amp->GetPosition());
  printf("\tAngle = %f\n", Amp->GetAngle());
  printf("\tFault = %d\n", Amp->GetFault());
  printf("\tTemp = %d\n", Amp->GetTemp());
  printf("\tCurrentMon = %f\n", Amp->GetCurrentMon());
  printf("\tCommand = %f\n", Amp->GetCommand());
  printf("\tEnable = %d\n", Amp->GetEnable());
  printf("\tMaxCommand = %f\n", Amp->GetMaxCommand());
  return 0;
}

int point_SetPID      (int point_num)
{
  register Cpid *pid = &point[point_num]->m_pid;
  double fNum;

  printf("SetPID parameters for point%d:\n", point_num);
  printf("Kp [%f] = ", pid->GetKp());
  if (scanf("%lf", &fNum)) pid->SetKp(fNum);
  printf("Ki [%f] = ", pid->GetKi());
  if (scanf("%lf", &fNum)) pid->SetKi(fNum);
  printf("Kd [%f] = ", pid->GetKd());
  if (scanf("%lf", &fNum)) pid->SetKd(fNum);
  printf("ILimit [%f] = ", pid->GetILimit());
  if (scanf("%lf", &fNum)) pid->SetILimit(fNum);
  printf("MaxPosErr [%f] = ", pid->GetMaxPosErr());
  if (scanf("%lf", &fNum)) pid->SetMaxPosErr(fNum);
  printf("DesPos [%f] = ", pid->GetDesPos());
  if (scanf("%lf", &fNum)) pid->SetDesPos(fNum);
  return 0;
}

int point_SetDes      (int point_num, double des)
{
  return (point[point_num]->m_pid.SetDesPos(des));
}

int point_SetAmp      (int point_num)
{
  register Camps *Amp = point[point_num]->m_amp;
  double fNum;

  printf("SetAmp parameters for amp%d:\n", point_num);
  printf("Command [%f] = ", Amp->GetCommand());
  if (scanf("%lf", &fNum)) Amp->SetCommand(fNum);
  printf("MaxCommand [%f] = ", Amp->GetMaxCommand());
  if (scanf("%lf", &fNum)) Amp->SetMaxCommand(fNum);
  return 0;
}

int point_Disable     (int point_num)
{
  return (point[point_num]->m_amp->Disable());
}

int point_Enable      (int point_num)
{
  return (point[point_num]->m_amp->Enable());
}

char *ScopeNameP(char *name, int point_num)
{
  static char str[20];

  sprintf(str, "point%d.%s", point_num, name);
  return str;
}

int point_Scope(int point_num, int freq)
{
  double fCommand=0.0, fCurPos=0.0, fDesPos=0.0, fCurrent=0.0, fAngle=0.0, 
         fVelocity=0.0;
  int    nTargetPos=0, nSetPoint=0;

  if ((freq < 0) || (freq > sysClkRateGet()))
    {
      printf("freq = %d must be between 0 and %d\n", freq, sysClkRateGet());
      return -1;
    }
  
  ScopeInstallSignal(ScopeNameP("Command", point_num), "volt", &fCommand, 
		     "double", 0);
  ScopeInstallSignal(ScopeNameP("CurPos", point_num), "tick", &fCurPos, 
		     "double", 0);
  ScopeInstallSignal(ScopeNameP("DesPos", point_num), "tick", &fDesPos, 
		     "double", 0);
  ScopeInstallSignal(ScopeNameP("Current", point_num), "amp", &fCurrent, 
		     "double", 0);
  ScopeInstallSignal(ScopeNameP("Angle", point_num), "rad", &fAngle, 
		     "double", 0);
  ScopeInstallSignal(ScopeNameP("TargetPos", point_num), "tick", &nTargetPos, 
		     "int", 0);
  ScopeInstallSignal(ScopeNameP("SetPoint", point_num), "tick", &nSetPoint, 
		     "int", 0);
  ScopeInstallSignal(ScopeNameP("Velocity", point_num), "tick/s", &fVelocity, 
		     "double", 0);

  ScopeChangeSampleRate(double(freq), 0);

  while(1)
    {
      fCommand = point[point_num]->m_pid.GetCommand();
      fCurPos  = point[point_num]->m_pid.GetCurPos();
      fDesPos  = point[point_num]->m_pid.GetDesPos();
      fCurrent = point[point_num]->m_amp->GetCurrentMon();
      fAngle   = point[point_num]->m_amp->GetAngle();
      nTargetPos= point[point_num]->m_trajectory.GetTarget();
      nSetPoint=  point[point_num]->m_trajectory.GetNewPos();
      fVelocity=  point[point_num]->m_trajectory.GetCurrentVel();
      ScopeCollectSignals(0);
      taskDelay(sysClkRateGet()/freq);
    }

  return 0;
}

int point_SetTarget (int point_num)
{
  register Ctraject *traject = &point[point_num]->m_trajectory;
  int nNum;

  printf("SetTarget parameters for amp%d:\n", point_num);
  printf("Target Position [%ld] = ", traject->GetTarget());
  if (scanf("%d", &nNum)) traject->SetNewTarget(nNum);
  return 0;
}

int point_SetTargetAngle (int point_num, double fAngle)
{
  return (point[point_num]->SetTarget(fAngle));
}


int point_SetTraject (int point_num)
{
  register Ctraject *traject = &point[point_num]->m_trajectory;
  double fAccel, fMaxVel;

  printf("SetTarget parameters for amp%d:\n", point_num);
  printf("Accel  [%f] = ", traject->GetAccel());
  scanf("%lf", &fAccel);
  printf("MaxVel [%f] = ", traject->GetMaxVel());
  if (scanf("%lf", &fMaxVel)) traject->SetParams(fMaxVel, fAccel, g_freqp);
  return 0;
}

int point_GetTraject (int point_num)
{
  register Ctraject *traject = &point[point_num]->m_trajectory;

  printf("GetTraject parameters for amp%d:\n", point_num);
  printf("\tTrajectory Position = %d\n", traject->GetNewPos());
  printf("\tMax Velocity = %f\n", traject->GetMaxVel());
  printf("\tAcceleration = %f\n", traject->GetAccel());
  printf("\tFrequency = %d\n", traject->GetFreq());
  printf("\tTime = %d\n", traject->GetTime());
  printf("\tDirection = %d = %s\n", traject->GetDirection(), (traject->GetDirection() == 1) ? "Positve, Clockwise" : "Negative, Counter-Clockwise");
  printf("\tCurrent Velocity = %f\n", traject->GetCurrentVel());
  printf("\tTarget Position = %ld\n", traject->GetTarget());
  printf("\tStatus = %d\n", traject->GetStatus());
  printf("\tCase = %d\n", traject->GetCase());
  return 0;
}

int point_Reset (int point_num)
{
  point_ChangeState (point_num, INHIBITED);
  if (point_num == 0) 
    amp[6].Reset();
  if (point_num == 1)
    amp[7].Reset();
  point_SetTargetAngle (point_num, 0.0);
  return 0;
}

int point_Home ()
{
  point_SetTargetAngle (0, 0.0);
  point_SetTargetAngle (1, 0.0);
  return 0;
}

int point_Demo ()
{
  point_SetTargetAngle (0, PI);
  point_SetTargetAngle (1, DECTORAD(40.0));
  taskDelay(sysClkRateGet()*2); // wait 2 seconds
  point_SetTargetAngle (0, 0.0);
  point_SetTargetAngle (1, 0.0);
  taskDelay(2);
  point_SetTargetAngle (0,PI);
  point_SetTargetAngle (1,DECTORAD(-40.0));
  taskDelay(sysClkRateGet()*1); // wait 1 second
  point_SetTargetAngle (0,-PI/2.0);
  return 0;
}
