/*
   point.cxx

   Mark Sibenac
   97-2-4
   Atacama Desert Trek
   Field Robotics Center

   This module handles the two pointing system motors (azimuth, pitch).

   Each motor can be in one of the following states: {INHIBITED, SERVOING,
   OVERCURRENT, OVERTEMP, SERVO_ERROR}

*/

#include "point.h"
#include "taskLib.h"
#include "math.h"
#include "test/test_compass.h"

Cpoint **point; // pointer to array of point object pointers
Cpoint *pAzimuth;
Cpoint *pElevation;

#define DEGTORAD(x) ((x)/180.0*PI)

// Function wrapper to Cpoint->UpdateServo() for Azimuth
int UpdatePointServoFun (int axis)
{
  point[axis]->UpdateServo();
}

// Angular and 1000 ticks/rev (Azimuth=1000*150, Pitch=1000*88)
Cpoint::Cpoint (int axis_number, Camps *amp_in, int angular, int priority) : m_pid(amp_in, angular), m_trajectory(amp_in, angular)
{
  char pString[20];
  int tid;

  Dprintf(("Cpoint::Cpoint() creating a new point object %d\n", axis_number));

  if ((axis_number < 0) || (axis_number > 1))
    {
      Dprintf(("Cpoint::Cpoint() illegal axis_number=%d\n", axis_number));
      return;
    }

  if (!amp_in)
    {
      Dprintf(("Cpoint::Cpoint() amp=%x is not initialized!!!\n", (unsigned int)amp_in));
      return;
    }

  m_axis_number = axis_number;
  m_amp = amp_in;
  m_semUpdate = semBCreate (SEM_Q_FIFO, SEM_EMPTY);
  if (!m_semUpdate)
    {
      Dprintf(("Cpoint::Cpoint() semBCreate() was not able to allocate "
	       "memory.\n"));
      return;
    } else {
      Dprintf(("Cpoint::Cpoint() semUpdate = 0x%x\n", m_semUpdate));
    }

  sprintf(pString, "tPntServo%d", axis_number);
  if ((tid = taskNameToId(pString)) != -1)
    taskDelete (tid);
  taskSpawn (pString, priority, VX_SUPERVISOR_MODE | VX_DEALLOC_STACK | 
	     VX_FP_TASK | VX_STDIO, 20000, (FUNCPTR)UpdatePointServoFun,
	     axis_number, 0, 0, 0, 0, 0, 0, 0, 0, 0);
}

Cpoint::~Cpoint ()
{
  char pString[20];

  sprintf(pString, "tPntServo%d", m_axis_number);

  m_state = INHIBITED;
  m_amp->Disable();
  m_amp->SetCommand(0.0);

  taskDelete(taskNameToId(pString));
  semDelete(m_semUpdate);
  Dprintf(("Cpoint::~Cpoint() semUpdate is gone and %s is deleted\n",pString));
}

// Init
//  This initializes all of the point member variables to a known state
//   motor_state = INHIBITED
//   
int Cpoint::Init ()
{
  m_state = INHIBITED;
  m_amp->Reset();
  m_pid.Init(0.002, 0.0001, 0.0, 2000.0, 500.0); // Some close values
  m_amp->SelectEncoderMode (4); // encoder count synchronous with all edges

  _antPos[0] = 0.0;
  _antPos[1] = 0.0;
  _antPos[2] = 0.0;
  
  _roverPos[0] = 1000.0;
  _roverPos[1] = 0.0;
  _roverPos[2] = 0.0;
  
  _stationPos[0] = 0.0;
  _stationPos[1] = 0.0;
  _stationPos[2] = 0.0;

  _roverAng[0] = 0.0;
  _roverAng[1] = 0.0;
  _roverAng[2] = 0.0;

  _antAng[0] = 0.0;
  _antAng[1] = 0.0;
  _antAng[2] = 0.0;

  return 0;
}

// UpdateServos
//  This is set up to run as a separate control thread that waits on a 
//  semaphore to run the PID loop and trajectory calculation. 
int Cpoint::UpdateServo ()
{
  static int t=0;

  while (1)
    {
      semTake (m_semUpdate, WAIT_FOREVER);
      if (m_axis_number == 1) // Elevation
	{
	  t = 0;
	  _roverAng[0] = DEGTORAD(double(compass->getAzimuth())/10.0);
	  _roverAng[1] = DEGTORAD(double(compass->getPitch())/10.0);
	  _roverAng[2] = DEGTORAD(double(compass->getRoll())/10.0);
	  Transform();
	  SetTarget(_antAng[1]);
	} else {
	  SetTarget(point[1]->_antAng[0]);
	}

      m_amp->UpdateEncoder();
      m_amp->UpdateCurrentMon();
      m_amp->UpdateFaults();

      m_pid.SetDesPos (m_trajectory.CalcNewPos());
      m_pid.ServoLaw(m_trajectory.GetCurrentVelSlice());
      
      if (m_state != SERVOING) continue;
      
      m_amp->SetCommand(m_pid.GetCommand());
    }
}

int Cpoint::ChangeState (motor_state_t state)
{
  switch (state)
    {
      case INHIBITED:
           m_amp->Disable();
	   m_amp->SetCommand(0.0);
	   break;
      case SERVOING:
	   m_pid.Reset();
	   m_amp->SetCommand(0.0);
	   m_amp->Enable();
	   break;
      case OVERCURRENT:
      case OVERTEMP:
      case SERVO_ERROR:
	   m_amp->Disable();
	   m_amp->SetCommand(0.0);
	   break;
    }

  m_state = state;
  return 0;
}

int Cpoint::SetTarget (double fTargetAngle)
{
  m_fTargetAngle = fTargetAngle;
  m_trajectory.SetNewTarget(int(fTargetAngle / PI_2 * double(m_amp->GetEncoderTicksPerRev())));

  return 0;
}

int Cpoint::Transform ()
{
  // Declare temporal variables for all the sines and cosines
  double deltaXsv,deltaYsv,deltaZsv; /* delta pos vehicle and station in {G} */
  double xA,yA,zA; /* position of the antenna in {V} */
  double cYaw,sYaw; /* cosines and sine of yaw */
  double cPitch,sPitch;
  double cRoll,sRoll;
  double antYaw,antPitch;
  double xSb,ySb,zSb; /* Coordinates of station in {B} */

  // Asigns values to the antenna position on the rover
  xA = _antPos[0];
  yA = _antPos[1];
  zA = _antPos[2];

  // Asigns actual values to the delta variables
  deltaXsv = _roverPos[0] - _stationPos[0];
  deltaYsv = _roverPos[1] - _stationPos[1];
  deltaZsv = _roverPos[2] - _stationPos[2];

  // Asigns actual values to sines and cosines
  cYaw   = cos( _roverAng[0] );
  cPitch = cos( _roverAng[1] );
  cRoll  = cos( _roverAng[2] );

  sYaw   = sin( _roverAng[0] );
  sPitch = sin( _roverAng[1] );
  sRoll  = sin( _roverAng[2] );

  // Gets station coordinates in {B} frame
  xSb = deltaXsv*(cYaw*cPitch)+
        deltaYsv*(sYaw*cPitch)-
	deltaZsv*sPitch-xA;

  ySb = deltaXsv*(cYaw*sPitch*sRoll-sYaw*cRoll)+
        deltaYsv*(sYaw*sPitch*sRoll+cYaw*cRoll)+
	deltaZsv*(cPitch*sRoll)-yA;

  zSb = deltaXsv*(cYaw*sPitch*cRoll+sYaw*sRoll)+
        deltaYsv*(sYaw*sPitch*cRoll-cYaw*sRoll)+
	deltaZsv*(cPitch*cRoll)-zA;

  // Calculates antenna yaw and pitch
  antYaw   = -atan2(xSb,ySb);
  antPitch = atan2(zSb,sqrt(xSb*xSb+ySb*ySb));

  // Asigns the value to the actual output matrix
  _antAng[0] = antYaw;
  _antAng[1] = antPitch;
  

    // If Error, return non-zero
    return (0);
}



