/* File: gpsUtil.c
 * Created by: Nick Vallidis
 * Creation date: 970115
 *
 * Description: utility routines for communicating with gps receiver 
 */

#include <stdlib.h>
#include <stdio.h>
#include "gpsUtil.h"
#include "serialRS232.h"



int checkOperational(int serial_fd)
{
  unsigned char b;

  /* make sure that the gps receiver is hooked up and ready. */
  serialFlushBuffer(serial_fd);
  serialPutBuf(serial_fd, "\005", 1);  /* send the ENQ byte */
  sleep(1);
  if ((serialNumChars(serial_fd) == 0) ||
      (b = serialGetchar(serial_fd) != ACK)) {
    return -1;
  }
  return 0;
}


GpsMessagePtr getResponse(int serial_fd)
{
  char b;
  GpsMessagePtr pResponse;

  pResponse = (GpsMessagePtr)malloc(sizeof(GpsMessage));
  b = serialGetchar(serial_fd);
  if (b == ACK) {                        /* message received */
    pResponse->type = MESSAGE_OK;
    pResponse->length = 0;
    return pResponse;
  }
  if (b == NAK) {                        /* receiver busy... */
    printf("getResponse: Got NAK!\n");
    pResponse->type = MESSAGE_ERROR;
    pResponse->length = 0;
    return pResponse;
  }
  if (b == STX) {                        /* alright! a real response! */
    pResponse->status = serialGetchar(serial_fd);       /* get status byte */
    if (b != 0)
      printf("getResponse: GPS Receiver Status != 0\n");
    pResponse->type = serialGetchar(serial_fd); /* get message type */
    pResponse->length = serialGetchar(serial_fd); /* data length */
    pResponse->pData = malloc(pResponse->length);
    while(serialNumChars(serial_fd) < pResponse->length)
      ; /* wait for enough characters to be in buffer to read them */
    serialGetBuf(serial_fd, pResponse->pData, pResponse->length, -1);
    b = serialGetchar(serial_fd);       /* checksum */
    b = serialGetchar(serial_fd);       /* should be ETX */
    if (b == ETX)
      return pResponse;
    free(pResponse->pData);
  }

  printf("getResponse: Received a bad message (either didn't start with\n");
  printf("             STX or didn't end with ETX)\n");
  pResponse->type = MESSAGE_ERROR;
  pResponse->length = 0;
  serialFlushBuffer(serial_fd);
  return pResponse;
}


void sendCommand(int serial_fd, GpsMessagePtr pMessage)
{
  unsigned char buf[254]; /* longest possible message length is 254 */
  unsigned int i, sum=0;

  buf[0] = STX;              /* message always starts with STX */
  buf[1] = 0;                /* this is used for receiver to indicate status */
  buf[2] = pMessage->type;   /* what kind of message this is */
  buf[3] = pMessage->length; /* size of the data for this message */
  memcpy(&buf[4], pMessage->pData, pMessage->length); /* copy in the data */
  for (i=1; i<(4+pMessage->length); i++)
    sum += buf[i];
  buf[i++] = sum % 256;      /* calculate the checksum */
  buf[i++] = ETX;              /* always end with ETX */

  /* now send the buffer */
  serialPutBuf(serial_fd, buf, i);
}
