/*
 * 
 * gps.cxx
 *
 * Mike Krebs
 * Atacama Desert Trek
 * 96-2-27
 *
 */


#include "common/nomad_global.h"
#include <logLib.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include "gps.h"
// #include "interfaces/serial/serial.h"
#include "../../interfaces/serial/serial.h"


int  TserialNumChars    (int tag, int fd) {
//  printf("serialNumChars (tag=%d)\n", tag);
  return serialNumChars(fd);
}

void TserialFlushBuffer (int tag, int fd) {
//  logMsg("serialFlushBuffer (tag=%d)\n", tag);
  serialFlushBuffer(fd);
}

void TserialPutBuf      (int tag, int fd, char *buf, int count) {
//  logMsg("serialPutBuf (tag=%d)\n", tag, 2,3,4,5,6);
  serialPutBuf(fd, buf, count);
}

int  TserialGetBuf      (int tag, int fd, char *buffer, int num_chars,
			 int timeout) {
//  logMsg("serialGetBuf (tag=%d)\n", tag, 2,3,4,5,6);
  return serialGetBuf(fd, buffer, num_chars, timeout);
}

char TserialGetchar     (int tag, int fd) {
//  logMsg("serialGetchar (tag=%d)\n", tag, 2,3,4,5,6);
  return serialGetchar(fd);
}


void Cgps::updateState() {
//  GpsMessage command;
  
//  command.type = MESSAGE_GENOUT;
//  command.length = 0;
//  sendCommand(&command);
  while (handleResponse() != MESSAGE_GENOUT);
  return;
}


Cgps::Cgps() {
  m_pSVS = NULL;
  m_TxID = 0;
  
  m_dLatitude = 0.0;
  m_dLongitude = 0.0;
  m_dHeight = 0.0;

  m_dX = 0.0;
  m_dY = 0.0;
  m_dZ = 0.0;

  m_dDeltaX = 0.0;
  m_dDeltaY = 0.0;
  m_dDeltaZ = 0.0;
}


Cgps::~Cgps() {
  if (m_pSVS) free(m_pSVS);
  close(m_fd);     /* close serial port to be nice */
}


/* sets up the updateState function in the background and initializes
 *  the binary semaphore. (uh, right now there is no semaphore...) */
int Cgps::init(char *pSerialDeviceName)
{
  // open serial port at 9600 bps and no parity
  m_fd = serialInit(pSerialDeviceName, 9600);
  
  TserialFlushBuffer(1, m_fd);

  /* NOTE: following should return error since GPS is not ready */
  if (checkOperational()) {
    printf("GPS receiver not ready.\n");
    return 1;
  }
  
  // put port in line mode
//  ioctl(fd, FIOSETOPTIONS, OPT_LINE);
//  ioctl(fd, FIOFLUSH, 0);

  return 0;
}

void Cgps::flushSerial(void) {
  TserialFlushBuffer(2, m_fd);
  return;
}


void Cgps::showPanel(void) {
  GpsMessage command;

  command.type = MESSAGE_SCRDUMP;
  command.length = 0;
  sendCommand(&command);
  while (handleResponse() != MESSAGE_SCRDUMP);
  return;
}


void Cgps::panelMode(void) {
  GpsMessage command;
  char key;

  for(key=0; (key != 'q') && (key != 'Q'); key=getchar()) {
    if (key == 0x0A) continue;
    switch(key) {
     case ',':
     case '<':
      key = 0x1D;
      break;
     case '.':
     case '>':
      key = 0x1C;
      break;
     case ';':
      key = 0x7F;
      break;
     case '/':
      key = 0x0D;
      break;
     case 'z': 
     case 'Z':
      key = 0x1B;
      break;
     default:
      key = toupper(key);
      break;
    }
    command.type = MESSAGE_KEYSIM;
    command.length = 1;
    command.pData = (unsigned char *)&key;
    sendCommand(&command);
    handleResponse();
    command.type = MESSAGE_SCRDUMP;
    command.length = 0;
    sendCommand(&command);
    while (handleResponse() != MESSAGE_SCRDUMP);
  }
  return;
}


void Cgps::keySequence(char *seq) {
  GpsMessage command;

  for(int i = 0; i < strlen(seq); i++) {
    switch(seq[i]) {
     case ',':
     case '<':
      seq[i] = 0x1D;
      break;
     case '.':
     case '>':
      seq[i] = 0x1C;
      break;
     case ';':
      seq[i] = 0x7F;
      break;
     case '/':
      seq[i] = 0x0D;
      break;
     case 'z': 
     case 'Z':
      seq[i] = 0x1B;
      break;
     default:
      seq[i] = toupper(seq[i]);
      break;
    }
    command.type = MESSAGE_KEYSIM;
    command.length = 1;
    command.pData = (unsigned char *)&seq[i];
    sendCommand(&command);
    handleResponse();
  }
  return;
}


void Cgps::setMessage(unsigned char cPort, unsigned char cType,
		      unsigned char cFreq) {
  GpsMessage command;
  int type;

  
  command.type = MESSAGE_APPFILE;
  command.length = 14;
  /* NOTE: should probably check for out of memory error */
  command.pData = (unsigned char *)malloc(14);

  command.pData[0] = m_TxID++; /* get next transmission block identifier */
  command.pData[1] = 0;        /* set Page Index to 0 */
  command.pData[2] = 0;        /* set Max Page Index to 0 */
  
  /* File Control Information */
  command.pData[3] = 0;        /* set App File Version */
  command.pData[4] = 1;        /* set Device Type to 7400MSi */
  command.pData[5] = 1;        /* apply the appfile now */
  command.pData[6] = 0;        /* only alter these paremeters */

  /* Output Message record */
  command.pData[7] = 7;        /* set Output Message info (type=7) */
  command.pData[8] = 5;        /* set length of the sub-record */
  command.pData[9] = 10;       /* set Output Message type to GSOF (type=10) */
  command.pData[10] = cPort;   /* set port number for these Output Messages */
  command.pData[11] = cFreq;   /* set frequency of Output Message (page C-4) */
  command.pData[12] = 0;       /* set offset of frequencies */
  command.pData[13] = cType;   /* set GSOF message type (page C-5) */

  sendCommand(&command);
  while (((type = handleResponse()) != MESSAGE_OK) &&
	 (type != MESSAGE_ERROR));

  free(command.pData);
}


/* make sure that the gps receiver is hooked up and ready. */
int Cgps::checkOperational(void) {
  TserialFlushBuffer(3, m_fd);
  TserialPutBuf(1, m_fd, "\005", 1);  /* send the ENQ byte */
//  sleep(1);
  taskDelay(sysClkRateGet());
  if ((TserialNumChars(1, m_fd) == 0) || (TserialGetchar(1, m_fd) != ACK))
    return -1;
  return 0;
}


void Cgps::sendCommand(GpsMessagePtr pMessage)
{
  unsigned char buf[254]; /* longest possible message length is 254 */
  unsigned int i, sum=0;
  
  buf[0] = STX;              /* message always starts with STX */
  buf[1] = 0;                /* this is used for receiver to indicate status */
  buf[2] = pMessage->type;   /* what kind of message this is */
  buf[3] = pMessage->length; /* size of the data for this message */
  memcpy(&buf[4], pMessage->pData, pMessage->length); /* copy in the data */
  for (i=1; i<(4+pMessage->length); i++)
    sum += buf[i];
  buf[i++] = sum % 256;      /* calculate the checksum */
  buf[i++] = ETX;            /* always end with ETX */
  
  /* now send the buffer */
  TserialPutBuf(2, m_fd, (char *)buf, i);
}


void Cgps::procGENOUT(GpsMessagePtr pResponse) {
  /* NOTE: should probably deal with multiple pages here (pg A-50) */

  switch (pResponse->pData[3]) {     /* table found on pg C-5 */
   case 2:    /* LAT, LONG, HEIGHT */
    /* skip the length (should it be used?) */
    m_dLatitude = ptod(&pResponse->pData[5]) * (double)180.0 / PI;
    m_dLongitude = ptod(&pResponse->pData[13]) * (double)180.0 / PI;
    m_dHeight = ptod(&pResponse->pData[21]);
    break;
   case 3:    /* ECEF POSITION */
    /* skip the length (should it be used?) */
    m_dX = ptod(&pResponse->pData[5]);
    m_dY = ptod(&pResponse->pData[13]);
    m_dZ = ptod(&pResponse->pData[21]);
    break;
   case 6:    /* ECEF DELTA */
    /* skip the length (should it be used?) */
    m_dDeltaX = ptod(&pResponse->pData[5]);
    m_dDeltaY = ptod(&pResponse->pData[13]);
    m_dDeltaZ = ptod(&pResponse->pData[21]);
    break;
   case 13:   /* SV BRIEF INFO */
    /* get "NUMBER OF SVS" and do sanity check against sub-record length */
    if ((pResponse->pData[4]-3)/3 != (m_cSVtotal = pResponse->pData[5])) {
      printf("Invalid length for number of SVs\n");
      return;
    }
    free(m_pSVS);
    /* NOTE: should probably check for out of memory error */
    m_pSVS = (t_sSV *)calloc(m_cSVtotal, sizeof(t_sSV));
    for (int i = 0; i < m_cSVtotal; i++) {
      m_pSVS[i].cPrn = pResponse->pData[3*i + 6];
      m_pSVS[i].cSVflag1 = pResponse->pData[3*i + 7];
      m_pSVS[i].cSVflag2 = pResponse->pData[3*i + 8];
    }
    break;
  }
  return;
}

void fillpanel(char buf[], unsigned char frombuf[], int linenum) {
  if ((frombuf[160] - 1) / 40 == linenum) {
    int curpos = (frombuf[160] - 1) % 40 + 1;
    strncpy(buf, (char *)frombuf+linenum*40, curpos);
    buf[curpos] = '_';
    buf[curpos+1] = (frombuf+linenum*40)[curpos];
    buf[curpos+2] = '_';
    strncpy(buf+curpos+3, (char *)frombuf+linenum*40+curpos+1, 40-curpos-1);
    buf[42] = 0;
  }
  else {
    strncpy(buf, (char *)frombuf+linenum*40, 40);
    buf[40] = 0;
  }
  return;
}

void Cgps::procSCRDUMP(GpsMessagePtr pResponse) {
  char buf[43];

  if (pResponse->length != 161) {
    printf("Not enough data in screen dump\n");
    return;
  }
  
  fillpanel(buf, pResponse->pData, 0);
  printf("1: %s\n", buf);
  fillpanel(buf, pResponse->pData, 1);
  printf("2: %s\n", buf);
  fillpanel(buf, pResponse->pData, 2);
  printf("3: %s\n", buf);
  fillpanel(buf, pResponse->pData, 3);
  printf("4: %s\n", buf);
  return;
}

GpsMessagePtr Cgps::getResponse(void) {
  GpsMessagePtr pResponse;
  unsigned char sum = 0;
  
  /* NOTE: should probably check for out of memory error */
  pResponse = (GpsMessagePtr)malloc(sizeof(GpsMessage));
  pResponse->pData = NULL;   /* initialize to NULL */
  /* loop until a valid response is found */
  while (1) {
    unsigned char start;

    switch (start = TserialGetchar(2, m_fd)) {
     case ACK:       /* message received */
      pResponse->type = MESSAGE_OK;
      pResponse->length = 0;
      TserialFlushBuffer(4, m_fd);
      return pResponse;
     case NAK:       /* receiver busy... */
      printf("Got a NAK...\n");
      pResponse->type = MESSAGE_ERROR;
      pResponse->length = 0;
      TserialFlushBuffer(5, m_fd);
      return pResponse;
     case STX:       /* alright! a real response! */
      pResponse->status = TserialGetchar(3, m_fd);     /* get status byte */
      pResponse->type = TserialGetchar(4, m_fd);       /* get message type */
      pResponse->length = TserialGetchar(5, m_fd);     /* data length */
      /* NOTE: should probably check for out of memory error */
      pResponse->pData = (unsigned char *)malloc(pResponse->length);
      /* wait for enough characters to be in buffer to read them */
      while(TserialNumChars(2, m_fd) < pResponse->length);
      TserialGetBuf(1, m_fd, (char *)pResponse->pData, pResponse->length, -1);
      
      /* verify the checksum (page A-2) */
      for (int i = 0; i < pResponse->length; i++)
	sum += pResponse->pData[i];
      sum += pResponse->status + pResponse->type + pResponse->length;
      unsigned char b;
      if ((b = (unsigned char)TserialGetchar(6, m_fd)) != sum % 256) {
//	printf("Checksum\n");
//	printf("Checksum mismatch (type=%#x len=%d sub-type=%d "
//	       "sub-len=%d page=%d %d!=%d)\n", pResponse->type,
//	       pResponse->length, pResponse->pData[3], pResponse->pData[4], b,
//	       pResponse->pData[1], sum % 256);
//	for (int i = 0; i < 24; i++)
//	  printf("%d ", pResponse->pData[5+i]);
//	printf("\n");
//	pResponse->status = 0xff;
	goto response_err;
      }
      
      /* check for end of message */
      if (TserialGetchar(7, m_fd) != ETX) {
	printf("No End of Data found\n");
	goto response_err;
      }
      TserialFlushBuffer(6, m_fd);
      return pResponse;
    }
    printf("Invalid char found on serial line (char=%d)\n", start);
  }

response_err:
  /* Received a bad message */
//  printf("T: bad message\n");
  pResponse->type = MESSAGE_ERROR;
  pResponse->length = 0;
  TserialFlushBuffer(7, m_fd);
  return pResponse;
}

int Cgps::handleResponse(void) {
  GpsMessagePtr pResponse = getResponse();

  /* NOTE: should check status byte here (pg A-4) */

  /* NOTE: redundant... should be taken out */
  /* check if there was an error while getting data */
  if (pResponse->type == MESSAGE_ERROR) {
    printf("Error while getting message\n");
    goto gps_err;
  }

  switch (pResponse->type) {
   case MESSAGE_OK:
    printf("Command completed successfully\n");
    break;
   case MESSAGE_ERROR:
    printf("Command failed\n");
    break;
   case MESSAGE_GENOUT:
//    printf("General Output message found (length=%d sub-type=%d page=%d)\n",
//	   pResponse->length, pResponse->pData[3], pResponse->pData[1]);
//    for (int i = 0; i < 24; i++)
//      printf("%d ", pResponse->pData[5+i]);
//    printf("\n");
    procGENOUT(pResponse);
    break;
   case MESSAGE_SCRDUMP:
//    printf("Screen Dump found\n");
    procSCRDUMP(pResponse);
    break;
   default:
    printf("Message type not supported (%#x)\n", pResponse->type);
    goto gps_err;
  }
  
gps_err:
  unsigned char nType = pResponse->type;
  if (pResponse->pData) free(pResponse->pData);
  free(pResponse);
  return nType;
}


double Cgps::ptod(unsigned char *pData) {
  double tempnum;
  int i;
  
  for (i=0; i<8; i++)
    ((unsigned char *)&tempnum)[i] = pData[i];
  return(tempnum);
}
