/*
 * -------------------------------------------------------
 *
 *  Copyright (C) 1997 Jochen Karrer
 *
 *  Released under the terms of the GPL.
 *
 * -------------------------------------------------------
 */

#include <linux/config.h>
#ifdef MODULE
#include <linux/module.h>
#include <linux/version.h>
#endif
#include <linux/errno.h>
#include <linux/fs.h>
#include <linux/fcntl.h>
#include <linux/major.h>
#include <asm/segment.h>
#include <linux/kernel.h>
#include <linux/timer.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/malloc.h>
#include <linux/ioport.h>
#include <linux/proc_fs.h>
#include <asm/irq.h>
#include <asm/bitops.h>
#include <asm/io.h>
#include <asm/pgtable.h>

#define COMMAND_PORT(adc) ((adc)->iobase+4)
#define INPUT_PORT(adc) ((adc)->iobase+5)
#define DATA_PORT(adc) ((adc)->iobase+6)
#define C54_CONTROL_PORT(adc) ((adc)->iobase+3)
#define C54_COUNTER(counter,adc) ((adc)->iobase+(counter))

// 60 is MAJOR for experimental use 
#ifndef ADC_MAJOR
  #define ADC_MAJOR 61 
#endif
#define STATUS_MINOR 255
#define N_CCD_PIXELS (2048)
#define FIFOSIZE (8*N_CCD_PIXELS)
#define MAX_AQUISITIONS_PER_SEC 30
#define MAX_CMDLEN 40

#define EOF_SENDABLE   0
#define EOF_PENDING 1
#define EOF_SENT    2

struct Adc_500_s;
static int mod_use_count=0;

extern int s2000_procinfo (char *buf, char **start, off_t fpos, 
                                int length, int dummy);

static void s2000_set_integration_time(int time,struct Adc_500_s *adc);
static void s2000_trigger_aquisition(struct Adc_500_s *adc);
static int s2000_init(struct Adc_500_s *adc);

typedef struct {
	int (*init)(struct Adc_500_s *adc);
	void (*set_integration_time)(int time,struct Adc_500_s *adc);
	void (*trigger_aquisition)(struct Adc_500_s *adc);
	int  n_pixels;
} spectro_ops;

static spectro_ops 
s2000_spec_ops = {
	s2000_init,
	s2000_set_integration_time,
	s2000_trigger_aquisition,
	2048,
};
typedef struct  Adc_500_s {
	unsigned int iobase;	
	unsigned int irq;

	// statistics
	unsigned int n_irqs;
	unsigned int watchdog_time; // for watchdog

	// Software fifo
	unsigned int * fifo;
	unsigned int fifo_wp;	
	
	unsigned int message_wp;

	unsigned int continuous;
	unsigned int add_frames;
	unsigned int framecounter;
	unsigned int channel;
	unsigned int integration_time;

	struct wait_queue *wait_on_data;
	struct wait_queue *wait_on_message;
	struct timer_list trigger_timer;
	int timer_busy;

	// register backingstores
	unsigned char cmd;

	spectro_ops *spec_ops;
} Adc_500;

typedef struct {
	Adc_500  *adc;
	unsigned int message_rp;
	unsigned int fifo_rp;
	char str[25];
	unsigned int str_readpos;
	int (*readfunc)(struct inode *,struct file *,char *,int);
	int  eof;

	// interpreter
	char cmd[MAX_CMDLEN];
	unsigned int  cmd_wp;
	char *statuspage;
	int statuslen;
} adc_file_data;

#define LinuxVersionCode(v, p, s) (((v)<<16)+((p)<<8)+(s))
#if LINUX_VERSION_CODE < LinuxVersionCode(2,1,3)
        #define copy_to_user memcpy_tofs
        #define copy_from_user memcpy_fromfs
        #define put_user_noverify put_user
#else
        #define put_user_noverify __put_user
#endif

#define MAXBOARDS 1

static Adc_500 adcinfo[MAXBOARDS];
static unsigned int adc_iobase[MAXBOARDS]={0x360,};
static unsigned int adc_irq[MAXBOARDS]={0x05,};


static inline void 
adc_load_counter(unsigned int counter,int value,Adc_500 *adc) {
	int ctrl;
	if(counter > 2){
		printk("shit, trying to load nonexistent counter %d\n",counter);
		return;
	}
	ctrl = (counter << 6)+0x36;
	outb(ctrl,C54_CONTROL_PORT(adc));
	outb(value & 0xff,C54_COUNTER(counter,adc));
	outb((value>>8)&0xff,C54_COUNTER(counter,adc));
}
static void 
adc_sleep(unsigned int time){
	unsigned int timeout=jiffies+time+1;
	while(jiffies<timeout) {
		schedule();
	}
}

static inline void 
adc_reset_fifo(Adc_500 *adc){
	adc->cmd |= 0x20;
	outb(adc->cmd,COMMAND_PORT(adc));
	adc->cmd &= ~0x20;
	outb(adc->cmd,COMMAND_PORT(adc));
}
static void 
s2000_set_integration_time(int time,Adc_500 *adc) {
	if((time < 5) || (time>65535) ) 
		time = 5;
	adc->integration_time = time;
	adc_load_counter(2,time,adc);
}
static void 
s2000_trigger_aquisition(Adc_500 *adc) {
	adc->watchdog_time = jiffies;
	adc->cmd &= ~0x98; 	// mask channel bits
	adc->cmd |= ((adc->channel&3)<<3) + ((adc->channel&4)<<5);
	adc->cmd &= ~0x41; 	// disable interrupt, disable trigger
	adc->cmd |= 0x20; 	// reset fifo
	outb(adc->cmd,COMMAND_PORT(adc));
	adc->cmd &= ~0x20; 	// stop reset 
	outb(adc->cmd,COMMAND_PORT(adc));
	adc->cmd |= 0x41; 	//  enable interrupt, enable trigger 
	adc->cmd |= 0x2; 	//  not dark, not external 
	outb(adc->cmd,COMMAND_PORT(adc));
	adc->cmd &= ~0x2;
}
static inline void 
adc_watchdog(Adc_500 *adc) {
	if(adc->continuous && (adc->watchdog_time+2*adc->integration_time<jiffies)) {
		adc->watchdog_time = jiffies;
		adc->spec_ops->trigger_aquisition(adc);
	}
}
static inline void
align_read_fifo(Adc_500 *adc) {
        u_int i = adc->fifo_wp  % N_CCD_PIXELS;
        adc->fifo_wp = ( adc->fifo_wp - i ) % FIFOSIZE;
}
static void
adc_timer_func(unsigned long data) {
	Adc_500 *adc = (Adc_500 *)data;
	adc->spec_ops->trigger_aquisition(adc);
	adc->timer_busy = 0;
}
/*
 * -------------------------------------------------------------------
 *
 *  adc_interrupt
 * 
 *	read fifo and reset it, restart aquisition if continuous
 * 
 * -------------------------------------------------------------------
 */
static void 
adc_interrupt(int irq,void *dev,struct pt_regs *regs) {
	int i;
	Adc_500 *adc = (Adc_500 *)dev;
	static int intwatch; 
	static u_int last_jiffies;

	adc->n_irqs++;

	adc->cmd &= ~0x41; // turn interrupts and triggers off
	outb(adc->cmd,COMMAND_PORT(adc));
	sti();
	for(i=0;i<N_CCD_PIXELS;i++){
		unsigned int value = inw(DATA_PORT(adc));
		if(adc->framecounter == adc->add_frames) 
			adc->fifo[(adc->fifo_wp+i)%FIFOSIZE] = value;
		else 
			adc->fifo[(adc->fifo_wp+i)%FIFOSIZE] += value;
	}	
	adc_reset_fifo(adc);
	if(jiffies > last_jiffies)
		intwatch = 0;
	intwatch++;
	if(adc->continuous) {
		if(intwatch > 300 / HZ) {       // to much interrupts
			adc->continuous = 0;    // disable one board
		}
		else {
			cli();
			if(adc->integration_time>1000/MAX_AQUISITIONS_PER_SEC){ 
				sti();
				adc->spec_ops->trigger_aquisition(adc);	
			} else if(!adc->timer_busy) {
				init_timer(&adc->trigger_timer);
				adc->trigger_timer.expires = jiffies+HZ/MAX_AQUISITIONS_PER_SEC;
				add_timer(&adc->trigger_timer);
				adc->timer_busy = 1;
			}
			sti();
		}
	} else if(intwatch > 300/HZ) {	// still interrupts ?
		free_irq(irq,dev);  	// ( wrong ioport, correct irq for example)
		adc->irq = 0;
	}
	adc->framecounter--;
	if(adc->framecounter == 0){
		adc->framecounter = adc->add_frames;
		adc->fifo_wp = (adc->fifo_wp+N_CCD_PIXELS)%FIFOSIZE;
		align_read_fifo(adc);
		wake_up_interruptible(&adc->wait_on_data);
	}
	return;
}
/*
 * --------------------------------------------------------------------------
 *
 * 	adc_lseek -
 *
 *		SEEK_END: skip data in read-buffer
 *
 * -------------------------------------------------------------------------
 */
static int 
adc_lseek(struct inode * inode, struct file * file, off_t offset, int orig){
	adc_file_data *pd = (adc_file_data *) file->private_data;

	if(MINOR(inode->i_rdev) == STATUS_MINOR) 
		return file->f_pos;
	switch (orig) {
                case 0: // SEEK_SET
                case 1: // SEEK_CUR
			return file->f_pos;
                case 2: // SEEK_END
			pd->fifo_rp = pd->adc->fifo_wp;
                        return file->f_pos;
                default:
                        return -EINVAL;
	}
}
/*
 * ---------------------------------------------------------------------------
 *
 * 	adc_binary_read -
 *
 *		puts the data to user space in binary (unsigned int) format 
 *
 * ---------------------------------------------------------------------------
 */
static int 
adc_binary_read(struct inode *inode,struct file *file,char *buf,int count){
	adc_file_data *pd = (adc_file_data *) file->private_data;
	Adc_500 *adc = pd->adc;
	int i;
	if(pd->eof == EOF_PENDING ) {
		pd->eof = EOF_SENT;
		return 0;
	}
	count = count - (count % sizeof(*adc->fifo));  // align
	for(i=0;i<count;i+=sizeof(*adc->fifo)) {
		if((pd->eof !=EOF_SENT) && ((pd->fifo_rp % N_CCD_PIXELS)==0))  {
			if(i) 
				pd->eof = EOF_PENDING;
			else 	
				pd->eof = EOF_SENT;
			return i;
		}
		while(adc->fifo_wp == pd->fifo_rp) {
			if(i)
				return i;
			if(file->f_flags & O_NONBLOCK) 
				return -EWOULDBLOCK;
			interruptible_sleep_on(&adc->wait_on_data);
			if(current->signal & ~current->blocked) {
				return -ERESTARTSYS;
			}
		}	
		put_user(adc->fifo[pd->fifo_rp],(typeof(adc->fifo))(buf+i));
		pd->fifo_rp = (pd->fifo_rp+1) % FIFOSIZE;	
		pd->eof = EOF_SENDABLE;
		file->f_pos += sizeof(*adc->fifo);
	}
	return i;	
}
/*
 * ----------------------------------------------------------------------
 *
 * 	adc_ascii_read
 *
 *		put the data in ascii format to user space
 *
 * ----------------------------------------------------------------------
 */
static int
adc_ascii_read(struct inode *inode,struct file *file,char *buf,int count) {
	adc_file_data *pd = (adc_file_data *) file->private_data;
	Adc_500 *adc = pd->adc;
	int i;
	if(pd->eof == EOF_PENDING ) {
		pd->eof = EOF_SENT;
		return 0;
	}
	for(i=0;i<count;i++) {
		if(pd->str[pd->str_readpos] == 0) {
			if((pd->eof !=EOF_SENT) && ((pd->fifo_rp % N_CCD_PIXELS)==0))  {
				if(i) 
					pd->eof = EOF_PENDING;
				else 	
					pd->eof = EOF_SENT;
				return i;
			}
			while(adc->fifo_wp == pd->fifo_rp) {
				if(i)
					return i;
				if(file->f_flags & O_NONBLOCK) 
					return -EWOULDBLOCK;
				interruptible_sleep_on(&adc->wait_on_data);
				if(current->signal & ~current->blocked) 
					return -ERESTARTSYS;
			}
			pd->str_readpos=0;
			sprintf(pd->str,"%u\n",adc->fifo[pd->fifo_rp]);
			pd->fifo_rp = (pd->fifo_rp+1) % FIFOSIZE;	
			pd->eof = EOF_SENDABLE;
			if(need_resched)
				schedule();
		}
		put_user(pd->str[pd->str_readpos],buf+i);
		pd->str_readpos++;
		file->f_pos++;			
	}
	return i;
}
/*
 * ------------------------------------------------------------------------
 *
 *  adc_read_status --
 *
 * 	print the state of all boards into a string, and put
 *	the string to userspace
 *
 * ------------------------------------------------------------------------
 */
static int
adc_status_read(struct inode * inode,struct file * file,char *buf,int count){
	adc_file_data *pd = (adc_file_data *) file->private_data;
	Adc_500 *adc;
	int i;
	if(file->f_pos == 0) {
		char *str = pd->statuspage;
		for(i=0;i<MAXBOARDS;i++) {
			adc=adcinfo+i;
			str+=sprintf(str,"\nS2000/ADC-500 Board %d Status:\n",i);
			str+=sprintf(str,"---------------------------------\n");
			if(adc->continuous == 0) 
				str+=sprintf(str,"Aquisition Is Stopped\n");	
			str+=sprintf(str,"IO-Baseaddress:      0x%x\n",adc->iobase);
			str+=sprintf(str,"Interrupt:           %d\n",adc->irq);
			str+=sprintf(str,"received Interrupts: %d\n",adc->n_irqs);
			str+=sprintf(str,"active channel:      %d\n",adc->channel);
			str+=sprintf(str,"Integration Time:    %d ms\n",adc->integration_time);	
			str+=sprintf(str,"Frames to add:       %d\n" , adc->add_frames);
			str+=sprintf(str,"Number Of Users:     %d\n" , mod_use_count);
		}
		str[0]=0;	
		pd->statuslen = str-pd->statuspage;
	}
	if(file->f_pos>=pd->statuslen) {
		return 0;
	}
	if((file->f_pos+count)>pd->statuslen) {
		count = pd->statuslen - file->f_pos;   
	}
	copy_to_user(buf,pd->statuspage+file->f_pos,count);
	file->f_pos+=count;
	return count;
}
static int
adc_read(struct inode *inode,struct file *file,char *buf,int count) {
	adc_file_data *pd = (adc_file_data *) file->private_data;
	int error = verify_area(VERIFY_WRITE,buf,count);
        if (error)
                return error;
	return pd->readfunc(inode,file,buf,count);
}
/*
 * --------------------------------------
 * 
 * sscan an integer
 *
 * --------------------------------------
 */
static void 
adc_isscan(char *str,u_int *value) {
	int i=0;
	*value = 0;
	while(str[i]==' ')
		i++;
	while((str[i]>='0') && (str[i]<='9')) {
		*value = *value * 10 + str[i] - '0';
		i++;
	}	
}
/*
 * -----------------------------------------------------------------
 *
 *  adc_interpreter -
 *
 *	interprete the ascii strings sent to /dev/adc500 
 *
 * -----------------------------------------------------------------
 */
static void
adc_interpreter(char *cmd,adc_file_data *pd) {
	u_int time;
	Adc_500 *adc = pd->adc;
	adc_isscan(cmd,&time);
	if(time) {
		adc->spec_ops->set_integration_time(time,adc);
		return;
	}
	switch (cmd[0]) {
		case 'a':			
			if(cmd[1] == 'd') {	 // add 
				adc_isscan(cmd+3,&adc->add_frames);
				if(adc->add_frames <=0) 
					adc->add_frames = 1;
				adc->framecounter=1;
			} else if (cmd[1]=='s') {   // ascii
				pd->readfunc = adc_ascii_read;
			}
			break;

		case 'b':
			pd->readfunc = adc_binary_read;
			break;
			
		case 'c':
			adc_isscan(cmd+7,&adc->channel);
			adc->channel &=7;
			break;

		case 's':
			if(cmd[2] == 'a') {	 // start
				adc->continuous = 1;
				adc->spec_ops->trigger_aquisition(adc); 
			}
			else if (cmd[2] == 'o')  // stop
				adc->continuous = 0;
			break;
	} 
}
static int
adc_write(struct inode *inode, struct file *file,const char *buf,int count) {
	adc_file_data *pd = (adc_file_data *) file->private_data;
	int i;
	int error = verify_area(VERIFY_READ,buf,count);
	if(error)
		return error;

	if(MINOR(inode->i_rdev) == STATUS_MINOR) 
		return -ENODEV;

	for(i=0;i<count;i++) {
		if ( pd->cmd_wp < MAX_CMDLEN-1  ) {
			pd->cmd[pd->cmd_wp] = get_user(&buf[i]);
			pd->cmd[pd->cmd_wp+1]=0;
		}
		else {
			pd->cmd_wp = 0;
			continue;
		}
		if( pd->cmd[pd->cmd_wp] == '\n') {
			adc_interpreter(pd->cmd,pd);
			pd->cmd_wp = 0;
			continue;
		}
		pd->cmd_wp++;
	}
	file->f_pos+=i;
	return i;
	
}
static int
adc_select(struct inode *inode, struct file *file, int sel_type,select_table *wait) {
	adc_file_data *pd = (adc_file_data *) file->private_data;
	Adc_500 *adc = pd->adc;
	switch (MINOR(inode->i_rdev)) {
		u_long flags;
		case STATUS_MINOR:
			if(sel_type == SEL_IN) {
				save_flags(flags);
				cli();
				if(pd->message_rp != adc->message_wp) {
					restore_flags(flags);
					return 1;
				}
				select_wait(&adc->wait_on_message,wait);
				restore_flags(flags);
				return 0;
	
			}
			break;
		default:
			if(sel_type == SEL_IN) {
				save_flags(flags);
				cli();
				if(pd->fifo_rp != adc->fifo_wp) {
					restore_flags(flags);
					return 1;
				}
				select_wait(&adc->wait_on_data,wait);
				restore_flags(flags);
				return 0;
			}
	}
	return 0;
}
static void 
adc_close(struct inode *inode,struct file *file) {
	adc_file_data *pd = (adc_file_data *) file->private_data;
	if(pd->statuspage)
		vfree(pd->statuspage);
	kfree(pd);
	MOD_DEC_USE_COUNT;
	mod_use_count--;
}

static int
adc_open( struct inode* inode, struct file* file)
{
        adc_file_data *pd;
	int minor =  MINOR(inode->i_rdev);
	if((minor >= MAXBOARDS) && (minor != STATUS_MINOR)) {
		return -ENODEV;
	}
        if(!(file->private_data =
            kmalloc(sizeof(adc_file_data),GFP_KERNEL)))
                return -ENOMEM;
	pd = (adc_file_data *) file->private_data;
	pd->statuspage = NULL;
	if(minor == STATUS_MINOR) {
		pd->statuspage = vmalloc(PAGE_SIZE);		
		if(pd->statuspage == NULL) {
			kfree(pd);
			return -ENOMEM;
		}
		pd->adc = adcinfo;
		pd->message_rp = pd->adc->message_wp;
		pd->statuslen = 0;
		pd->readfunc = adc_status_read;
	}
	else { // NOT STATUS_MINOR
		pd->adc = adcinfo+minor;
		pd->readfunc = adc_ascii_read;
		pd->str[0]=0;
		pd->str_readpos=0;
		pd->cmd_wp=0;
		pd->fifo_rp = pd->adc->fifo_wp;
		pd->eof = EOF_SENT;
		adc_watchdog(pd->adc);
	}
	MOD_INC_USE_COUNT;
	mod_use_count++;
	return 0;
}

static struct file_operations adc_fops = {
	adc_lseek,  // lseek
	adc_read,  
	adc_write, 
	NULL,  // readdir
	adc_select,  
	NULL,  // ioctl 
	NULL,  // mmap
	adc_open, 
	adc_close,
	NULL,  // fsync
};

/*
 * ----------------------------------------------------------------
 *
 * Proc dir entry for the s2000 
 *
 *-----------------------------------------------------------------
 */

static struct proc_dir_entry s2000_proc_entry = {
0, 5, "s2000", S_IFREG | S_IRUGO, 1, 0, 0, 0, 0, s2000_procinfo };

void register_procinfo(void) {
	s2000_proc_entry.name    = "s2000";
	s2000_proc_entry.namelen = strlen("s2000");
	proc_register_dynamic(&proc_root, &s2000_proc_entry);
}
int s2000_procinfo (char *buf, char **start, off_t fpos, 
                                int length, int dummy) {
	Adc_500 *adc;
	char * str = buf;
	int i;
	for(i=0;i<MAXBOARDS;i++) {
		adc=adcinfo+i;
		str+=sprintf(str,"\nS2000/ADC-500 Board %d Status:\n",i);
		str+=sprintf(str,"---------------------------------\n");
		if(adc->continuous == 0)
			str+=sprintf(str,"Aquisition Is Stopped\n");
		str+=sprintf(str,"IO-Baseaddress:      0x%x\n",adc->iobase);
		str+=sprintf(str,"Interrupt:           %d\n",adc->irq);
			str+=sprintf(str,"received Interrupts: %d\n",adc->n_irqs);
		str+=sprintf(str,"active channel:      %d\n",adc->channel);
		str+=sprintf(str,"Integration Time:    %d ms\n",adc->integration_time);
		str+=sprintf(str,"Frames to add:       %d\n" , adc->add_frames);
		str+=sprintf(str,"Number Of Users:     %d\n" , mod_use_count);
	}
	return str - buf;
}
void unregister_procinfo(void) {
	proc_unregister(&proc_root, s2000_proc_entry.low_ino );

}
/*
 * -------------------------------------------------------------
 *
 *  cleanup_module()
 *
 *		free resources
 *
 * -------------------------------------------------------------
 */
void 
cleanup_module(void) {
	int i;
	if(unregister_chrdev(ADC_MAJOR,"ADC-500") != 0) {
		printk("cleanup module failed\n");
	}
	for(i=0;i<MAXBOARDS;i++) {
		unsigned long flags;
		Adc_500 *adc = adcinfo+i;
		if(adc->irq) { 
			// this has to be done first !
			free_irq(adc->irq,adc);
		}
		save_flags(flags);
		cli();
		if(adc->timer_busy) {
			del_timer(&adc->trigger_timer);
		}
		restore_flags(flags);
		if(adc->iobase) {
			release_region(adc->iobase,0x8);
		}
		if(adc->fifo != NULL ) {
			vfree(adc->fifo);
		}
	}
	unregister_procinfo();
}
static int
s2000_init(Adc_500 *adc) {

	// Counter 0: Master clock for the S2000 (2 * conversion frequency)
	//            programmed to 1 MHz
 	//
	// Counter 1: Connected to input of counter 2 which gives 
	//            the integration time and is set to 1ms 

	adc_load_counter(0,8,adc);
	adc_load_counter(1,8000,adc);
	// probe if really a S2000
	// ...................
	return 1;
}
/*
 * ------------------------------------------------------------------
 *
 *  init_module --
 *
 *	allocate resources and reset the card
 *
 * -------------------------------------------------------------------
 */
int 
init_module(void) {
	int i;
	memset(adcinfo,0,sizeof(Adc_500)*MAXBOARDS);
	if(register_chrdev(ADC_MAJOR,"ADC-500",&adc_fops)) {
		printk("register chardevice failed\n");
		return -EBUSY;
	}
	for(i=0;i<MAXBOARDS;i++) {
		Adc_500 *adc = adcinfo+i;
		// init adc struct just for fun ( see memset above )
		adc->wait_on_data = NULL;
		adc->channel = 0;
		adc->cmd = 0x0 ;
		adc->timer_busy = 0;
		adc->spec_ops = &s2000_spec_ops;

		if(check_region(adc_iobase[i],0x08)<0) {
			printk("ADC-500: can't get IO-address 0x%x\n",adc_iobase[i]);
			cleanup_module();
			return -EBUSY;
		}
		request_region(adc->iobase,0x8,"ADC-500");
		adc->iobase = adc_iobase[i];

		// reset the hardware fifo before interrupts are enabled

		adc_reset_fifo(adc);

		// setup software fifo
		adc->fifo = (typeof(adc->fifo))vmalloc(FIFOSIZE*sizeof(*adc->fifo));
		if(adc->fifo==NULL) {
			cleanup_module();
			return -ENOMEM;
		}
		if(request_irq(adc_irq[i], adc_interrupt,SA_INTERRUPT,"ADC-500",adc)){
			printk("can't get interrupt %d\n",adc_irq[i]);	
			cleanup_module();
			return -EBUSY;
		}
		adc->irq = adc_irq[i];

		if(adc->spec_ops->init(adc)) {
			printk("initialized ADC-500 (S2000) number %d\n",i);
		} else  {
			// try S1000
		}

		// test if present with readback of control register
		inb(C54_COUNTER(0,adc));
		outb(0x80|0x40|0x20|0x02,C54_CONTROL_PORT(adc));
		if((inb(C54_COUNTER(0,adc))&0xf)!=0x6){
			printk("can't initialize counter! maybe ioport 0x%x is wrong\n",
				adc->iobase);
		}
		else 
			printk("found 8254, card seems to be present at 0x%x\n",
				adc->iobase);

		adc->trigger_timer.data = (unsigned long) adcinfo+i;
		adc->trigger_timer.function = adc_timer_func;
		
		// try to trigger an interrupt:
		adc->spec_ops->set_integration_time(5,adc);
		adc->spec_ops->trigger_aquisition(adc); 
		adc_sleep(HZ/10);
		if(adc->n_irqs<1) 
			printk("can't trigger interrupt %d\n",adc->irq);
		else 
			printk("triggering interrupt %d succeeded\n",adc->irq);
		adc->continuous = 1;
		adc->add_frames = 1;
		adc->framecounter = adc->add_frames;
		adc->spec_ops->set_integration_time(333,adc);
		adc->spec_ops->trigger_aquisition(adc); 
	}
	register_procinfo();
	return 0;
}
