#!/usr/local/pxc/pxc200/pxc_ui
#
# Copyright (c) 1998,1999  Alessandro Rubini  (rubini@linux.it)
# Copyright (c) 1998       Daniel Scharstein  (schar@panther.middlebury.edu)
# Copyright (c) 1999       Carnegie Mellon University (mwagner@frc.ri.cmu.edu)
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#

# Use the environmente for configuration (not actually used in this version)
set donorm 0;  if [info exists env(ACQUIRE_NORM)]   {set donorm 1}
set dograb 1;  if [info exists env(ACQUIRE_NOGRAB)] {set dograb 0}

# Commandline parsing
set devicefile /dev/pxc0pgm
set imageSavePath "/usr/nomad/data/HiRes/"
if {([llength $argv] == 2) && [string match /* $argv]} {
    set devicefile [lindex $argv 0]
    set imageSavePath [lindex $argv 1]
} elseif {([llength $argv] == 1) && [string match /* $argv]} {
    set devicefile $argv 
} elseif [info exists env(PXCDEVICE)] {
    set devicefile $env(PXCDEVICE)
}

# Check whether the device is there
if ![file exists $devicefile] {
    puts stderr "$argv0: $devicefile: No such file or directory"
    exit 1
}

### Ok, we are up and running 
puts "$argv0: using $devicefile"

# Set up some of my global variables
if {$devicefile == "/dev/pxc0ppm" || $devicefile == "/dev/pxc0pgm"} {
    set imgWidth 320;
    set imgHeight 240;
} else {
    set imgWidth 640;
    set imgHeight 480;
}

# I use to "source" this file to interactively test new versions
catch {eval destroy [winfo children .]}

# Keep the file open, to maximize perfomance
set keepopen [open $devicefile r]

# Use an external image, as some versions of wish are not able to load from
# a device node.
set imgfile /tmp/bla.pgm
exec cp $devicefile $imgfile

# Set global camera settings
set MIDCOL 320
set MIDROW 240
set CCD_WIDTH 0.0059862387
set NUM_COL 640
set NUM_ROW 480
set CCD_X 0
set CCD_Y -0.04
set CCD_Z 0.11
set DGPS_SENSOR_Z 1.84
set PTU_X -0.68
set PTU_Y 0.79
set PTU_Z [expr $DGPS_SENSOR_Z + 0.225 - 0.078]; # 0.078 is the height of the
                                                 # now removed PTU spacers 

set highResMuxSetting 1
set armMuxSetting 2
set lastInsertedRecord -1

### The procedures

proc min {a b} {
    if {$a < $b} {
	return $a;
    } else {
	return $b;
    }
}

proc max {a b} {
    if {$a > $b} {
	return $a;
    } else {
	return $b;
    }
}

proc LiveGrab {} {
    global LiveVideo devicefile imgfile keepopen
    if !$LiveVideo return
    exec cp $devicefile $imgfile
    grabimg config -file $imgfile
    update
    after 10 LiveGrab
}

proc Run {} {
    # Keep the file open, to maximize perfomance

    global LiveVideo keepopen devicefile imageSrc highResMuxSetting armMuxSetting 
	
    set LiveVideo 1
    .c configure -background cyan
    .b.run configure -state disabled
    .b.stop configure -state normal
    .b.save configure -state disabled
    if {$keepopen == ""} {
	set keepopen [open $devicefile r]
    }

    if {$imageSrc == "HIGH_RES"} {
	CtrlSet setmux $highResMuxSetting
    } elseif {$imageSrc == "ARM"} {
	CtrlSet setmux $armMuxSetting
    }

    after 10 LiveGrab
}
    
proc Stop {} {
    global LiveVideo devicefile imgfile keepopen
    set LiveVideo 0
    .c configure -background #aeb2c3
    .b.run configure -state normal
    .b.stop configure -state disabled
    .b.save configure -state normal

    if {$keepopen != ""} {
	close $keepopen
	set keepopen ""
	#exec rm -f $imgfile
    }
}

proc Save {} {
    global LiveVideo imgfile Saving lastInsertedRecord imageSavePath
    set fname "/usr/nomad/data/HiRes/"
    set targetID $lastInsertedRecord
    if $LiveVideo return
    set Saving 1
    toplevel .w -class Dialog
    wm title .w "Save File"
    frame .w.c
    label .w.c.lab -text "File name:"
    label .w.c.id -text "Target ID:"
    entry .w.c.ent -width 30 -relief sunken -bd 2 -textvariable fname
    entry .w.c.ident -width 30 -relief sunken -bd 2 -textvariable targetID

    bind .w.c.ident <Button-1> {
	.w.c.ident select range 0 end
    }

    .w.c.ent insert 0 $imageSavePath 
    if {[string index [.w.c.ent get] [expr [.w.c.ent index end] - 1]] != "/"} {
	.w.c.ent insert end "/"
    }
    if {$lastInsertedRecord != -1} {
	.w.c.ident insert 0 $lastInsertedRecord
	.w.c.ident select range 0 end
    }

    pack .w.c.lab .w.c.ent .w.c.id .w.c.ident -in .w.c -side left
    button .w.sav -text "Save" -command  {DoSave $fname $targetID 0}
    button .w.can -text "Cancel" -command {DoSave "" $targetID 0}
    pack .w.c -side top
    pack .w.sav .w.can -side left -expand yes

    bind .w <Return> {
	DoSave $fname $targetID 1
    }
    set oldFocus [focus]
    grab set .w
    focus .w.c.ent
    tkwait window .w
    focus $oldFocus
    set Saving 0
}

proc DoSave {fname targetID flash} {
    global imgfile

    set msecs 50
    if $flash {
	set msecs 500
    }
    if {$fname == ""} {
	.w.can configure -state active
	set msecs 500
    } else {
	.w.sav configure -state active
	exec cp $imgfile $fname
	puts "Database putting image $fname into target $targetID, returned: [putHiResDbImage $targetID $fname]"
    }
    .w.c.ent delete 0 end
    .w.c.ident delete 0 end
    after $msecs destroy .w
}
    

proc Quit {} {
    global imgfile
    exec rm -f $imgfile
    exit 0
}


### And the initial setup

if ![file size $imgfile] {
    puts "error in grabbing file"
    Quit
}

# The image canvas
canvas .c -height $imgHeight -width $imgWidth
pack .c -side top -expand no


image create photo grabimg  -file $imgfile
.c create image 0 0 -image grabimg -anchor nw -tags currentImage;

bind .c <Button-3> {
    global zoomBox upperLeftZoomCorner imageSrc

    if {$imageSrc == "HIGH_RES"} {
	set zoomBox(anchor) [list %x %y]
	set upperLeftZoomCorner(x) %x
	set upperLeftZoomCorner(y) %y
	catch {unset zoomBox(last)}
    } else {
    }
} 

bind .c <B3-ButtonRelease> {
    global zoomBox upperLeftZoomCorner imageSrc

    if {$imageSrc == "HIGH_RES"} {
	set zoomWidth [expr abs($upperLeftZoomCorner(x) - %x)]
	set zoomHeight [expr abs($upperLeftZoomCorner(y) - %y)]
	set zoomCenter(x) [expr $zoomWidth / 2 + [min %x $upperLeftZoomCorner(x)]]
	set zoomCenter(y) [expr $zoomHeight / 2 + [min %y $upperLeftZoomCorner(y)]]
	
	puts "Centering PTU to $zoomCenter(x), $zoomCenter(y)"
	
	set pan [lindex [getHiResDerivedState] 0];
	set tilt [lindex [getHiResDerivedState] 1];
	set FOV [lindex [getHiResDerivedState] 2];
	set pixelAngularWidth  [expr $FOV / $imgWidth];
	set pixelAngularHeight [expr $FOV / $imgHeight];
	set newFOV [max [expr $pixelAngularWidth * $zoomWidth] [expr $pixelAngularHeight * $zoomHeight]]
	if {$newFOV > $FOV} {
	    set newFOV $FOV;
	}
	
	puts "Current FOV is $FOV rad"
	puts "New FOV will be $newFOV rad"
	
	set function 0; # TELEOP_HI_RES_DEPLOY
	set newPan  [expr $pan + ($pixelAngularWidth * ($imgWidth / 2 - $zoomCenter(x)))];
	set newTilt [expr $tilt + ($pixelAngularHeight * ($imgHeight /2 - $zoomCenter(y)))];
	
	puts "newPan = $newPan, newTilt = $newTilt"
	
	Stop

	catch {
	    %W delete $zoomBox(last)
	}

	puts "Zooming and autofocusing (this may take a while...)"
	puts "Returned: [sendHiResRequest $function $newPan $newTilt $newFOV]"

	Run
	
    } else {
    }
}

bind .c <B3-Motion> {
    global zoomBox imageSrc

    if {$imageSrc == "HIGH_RES"} {
	catch {%W delete $zoomBox(last)}
	set zoomBox(last) [eval {%W create rect} $zoomBox(anchor) {%x %y -tag box -outline green}]
    }
}

bind .c <Button-1> {
    global imageSrc;

    if {$imageSrc == "HIGH_RES"} {
	puts "(R G B) = ([grabimg get %x %y])"
	set clickedCol [expr 640 * %x / $imgWidth];
	set clickedRow [expr 480 * %y / $imgHeight];
	
	set robotX [expr [lindex [getPosDerivedState] 0] / -100.0];
	set robotY [expr [lindex [getPosDerivedState] 1] / 100.0];
	set robotZ [expr [lindex [getPosDerivedState] 2] / 100.0];
	set pitch [expr [lindex [getPosDerivedState] 3] / 100.0];
	set roll [expr [lindex [getPosDerivedState] 4] / 100.0];
	set yaw [expr 2 * 3.1415927 - [lindex [getPosDerivedState] 4] / 100.0];
	set pan [lindex [getHiResDerivedState] 0];
	set tilt [lindex [getHiResDerivedState] 1];
	set FOV [lindex [getHiResDerivedState] 2];
	set focalLength [expr [lindex [getHiResDerivedState] 3] / 1000.0];
	
	puts "*******"
	puts "Robot is at ($robotX, $robotY, $robotZ), its pose is (P = $pitch, R = $roll, Y = $yaw)"
	puts "Camera is pointed at (pan = $pan, tilt = $tilt)"
	
	set stilt [expr sin($tilt)];
	set span [expr sin($pan)]; 
	set ctilt [expr cos($tilt)]; 
	set cpan [expr cos($pan)];
	set spitch [expr sin($pitch)]; 
	set sroll [expr sin($roll)]; 
	set syaw [expr sin($yaw)];
	set cpitch [expr cos($pitch)]; 
	set croll [expr cos($roll)]; 
	set cyaw [expr cos($yaw)];
	
	set x_prime [expr ($clickedCol - $MIDCOL)*($CCD_WIDTH / $NUM_COL)];
	set z_prime [expr ($MIDROW - $clickedRow)*($CCD_WIDTH / $NUM_ROW)];
	
	if {$focalLength == 0} {
	    set focalLength [expr 10.5 / 1000]; # this is so "divide by zero" errors don't occur
	}

	set p_3_Y [expr -($stilt * ($CCD_Y - $focalLength) + $ctilt * $CCD_Z + $PTU_Z) / \
		($stilt + $ctilt * $z_prime / $focalLength)];
	set p_3_X [expr $p_3_Y * $x_prime / $focalLength];
	set p_3_Z [expr $p_3_Y * $z_prime / $focalLength];
	
	set p_1_X [expr $cpan * $p_3_X - $span * $ctilt * $p_3_Y + $span * $stilt * $p_3_Z + $cpan * $CCD_X - \
		$span * $ctilt * ($CCD_Y - $focalLength) + $span * $stilt * $CCD_Z + $PTU_X];
	set p_1_Y [expr $span * $p_3_X + $cpan * $ctilt * $p_3_Y - $cpan * $stilt * $p_3_Z + $span * $CCD_X + \
		$cpan * $ctilt * ($CCD_Y - $focalLength) - $cpan * $stilt * $CCD_Z + $PTU_Y];
	set p_1_Z 0;
	
	set dgpsX [expr $cyaw * $croll * $p_1_X + (-1.0 * $syaw * $cpitch + $cyaw * $sroll * $spitch) * $p_1_Y + \
		($syaw * $spitch + $cyaw * $sroll * $cpitch) * $p_1_Z + $robotX];
	set dgpsY [expr $syaw * $croll * $p_1_X + ($cyaw * $cpitch + $syaw * $sroll * $spitch) * $p_1_Y + \
		($syaw * $sroll * $cpitch - $cyaw * $spitch) * $p_1_Z + $robotY];
	set dgpsZ [expr -1.0 * $sroll * $p_1_X + $croll * $spitch * $p_1_Y + $croll * $cpitch* $p_1_Z + $robotZ];
	
	set distance [expr sqrt($p_1_X * $p_1_X + $p_1_Y * $p_1_Y + ($p_1_Z - $DGPS_SENSOR_Z) * ($p_1_Z - $DGPS_SENSOR_Z))];
	
	puts "Mouse click at col = $clickedCol, row = $clickedRow";
	puts "Flat plane DGPS estimate = ($dgpsX, $dgpsY, $dgpsZ)";
	puts "Distance estimate = $distance meters";
	puts "Not inserting target into database (click with button-2 to insert target)";
	puts "*******"
    } else {
    }
}

bind .c <Button-2> {
    global imageSrc lastInsertedRecord;

    if {$imageSrc == "HIGH_RES"} {
	set clickedCol [expr 640 * %x / $imgWidth];
	set clickedRow [expr 480 * %y / $imgHeight];
	
	set robotX [expr [lindex [getPosDerivedState] 0] / -100.0];
	set robotY [expr [lindex [getPosDerivedState] 1] / 100.0];
	set robotZ [expr [lindex [getPosDerivedState] 2] / 100.0];
	set pitch [expr [lindex [getPosDerivedState] 3] / 100.0];
	set roll [expr [lindex [getPosDerivedState] 4] / 100.0];
	set yaw [expr 2 * 3.1415927 - [lindex [getPosDerivedState] 4] / 100.0];
	set pan [lindex [getHiResDerivedState] 0];
	set tilt [lindex [getHiResDerivedState] 1];
	set FOV [lindex [getHiResDerivedState] 2];
	set focalLength [expr [lindex [getHiResDerivedState] 3] / 1000.0];
	
	puts "*******"
	puts "Robot is at ($robotX, $robotY, $robotZ), its pose is (P = $pitch, R = $roll, Y = $yaw)"
	puts "Camera is pointed at (pan = $pan, tilt = $tilt)"
	
	set stilt [expr sin($tilt)];
	set span [expr sin($pan)]; 
	set ctilt [expr cos($tilt)]; 
	set cpan [expr cos($pan)];
	set spitch [expr sin($pitch)]; 
	set sroll [expr sin($roll)]; 
	set syaw [expr sin($yaw)];
	set cpitch [expr cos($pitch)]; 
	set croll [expr cos($roll)]; 
	set cyaw [expr cos($yaw)];
	
	set x_prime [expr ($clickedCol - $MIDCOL)*($CCD_WIDTH / $NUM_COL)];
	set z_prime [expr ($MIDROW - $clickedRow)*($CCD_WIDTH / $NUM_ROW)];
	
	if {$focalLength == 0} {
	    set focalLength [expr 10.5 / 1000]; # this is so "divide by zero" errors don't occur
	}

	set p_3_Y [expr -($stilt * ($CCD_Y - $focalLength) + $ctilt * $CCD_Z + $PTU_Z) / \
		($stilt + $ctilt * $z_prime / $focalLength)];
	set p_3_X [expr $p_3_Y * $x_prime / $focalLength];
	set p_3_Z [expr $p_3_Y * $z_prime / $focalLength];
	
	set p_1_X [expr $cpan * $p_3_X - $span * $ctilt * $p_3_Y + $span * $stilt * $p_3_Z + $cpan * $CCD_X - \
		$span * $ctilt * ($CCD_Y - $focalLength) + $span * $stilt * $CCD_Z + $PTU_X];
	set p_1_Y [expr $span * $p_3_X + $cpan * $ctilt * $p_3_Y - $cpan * $stilt * $p_3_Z + $span * $CCD_X + \
		$cpan * $ctilt * ($CCD_Y - $focalLength) - $cpan * $stilt * $CCD_Z + $PTU_Y];
	set p_1_Z 0;
	
	set dgpsX [expr $cyaw * $croll * $p_1_X + (-1.0 * $syaw * $cpitch + $cyaw * $sroll * $spitch) * $p_1_Y + \
		($syaw * $spitch + $cyaw * $sroll * $cpitch) * $p_1_Z + $robotX];
	set dgpsY [expr $syaw * $croll * $p_1_X + ($cyaw * $cpitch + $syaw * $sroll * $spitch) * $p_1_Y + \
		($syaw * $sroll * $cpitch - $cyaw * $spitch) * $p_1_Z + $robotY];
	set dgpsZ [expr -1.0 * $sroll * $p_1_X + $croll * $spitch * $p_1_Y + $croll * $cpitch* $p_1_Z + $robotZ];
	
	set distance [expr sqrt($p_1_X * $p_1_X + $p_1_Y * $p_1_Y + $p_1_Z * $p_1_Z)];
	
	puts "Mouse click at col = $clickedCol, row = $clickedRow";
	puts "Flat plane DGPS estimate = ($dgpsX, $dgpsY, $dgpsZ)";
	puts "Distance estimate = $distance meters";
	set lastInsertedRecord [insertDbRecord $dgpsX $dgpsY $dgpsZ];
	puts "Inserted record #$lastInsertedRecord into the database";

	puts "*******"
    } else {
	# Here we want to allow the user to click on a pixel and have the arm move directly over that pixel.
	puts "*******"
	puts "Mouse click at row = %y, col = %x (pixel coordinates)"
	puts "*******"

	moveArm %y %x; # moveArm row col
    }
}


### The control scales
proc More {} {
    global imageSrc;

    toplevel .more
    pack [button .more.close -text "Close" -command "destroy .more"] \
            -side bottom -expand y -fill x

    if {$imageSrc == "HIGH_RES"} {
	pack [button .more.zoomOut -text "Zoom Out" -command "zoomOut"] \
		[button .more.resetPtu -text "Reset PTU" -command "resetPTU"] \
		-side bottom -expand y -fill x
    } else {
	pack [button .more.stow -text "Stow Arm" -command "stowArm"]
    }

    proc __makeone {varname ctrlget ctrlset text min max} {
	global $varname devicefile
	pack [frame .more.$varname -bd 2 -relief ridge] -padx 2 -pady 2 \
		-expand y -fill both -side left
	pack [label .more.$varname.l -text $text] -side top
	pack [scale .more.$varname.s -orient v -from $min -to $max \
		-command "CtrlSet $ctrlset" -variable $varname] \
		-expand y -fill both
	# Init the $varname to a sane value, by parsing pxc_control output
	catch "exec pxc_control $devicefile $ctrlget" result
	set value [expr int([lindex [lindex [split $result {\"}] 3] 0])]
	# fix 0..255 to appear as -128..127
	if {$min<0 && $value>127} {set value [expr $value-256]}
	set $varname $value
    }

    __makeone bright getbright setbright Brightness -128 127
    __makeone contrast getcontrast setcontrast Contrast 0 511
    __makeone hue gethue sethue Hue -128 127
    __makeone satu getsatu setsatu "Saturation-U" 0 511
    __makeone satv getsatv setsatv "Saturation-V" 0 511
}

proc CtrlSet {cmd value} {
    global devicefile

    # turn -128..-1 to 128..255
    if {$value<0} {set value [expr $value + 256]}
    exec pxc_control $devicefile $cmd $value 2> /dev/null
}

proc CtrlGet {cmd} {
    global devicefile

    # turn -128..-1 to 128..255
    catch "exec pxc_control $devicefile $cmd" result
    return $result
}

proc zoomOut {} {
    puts "Zooming out and autofocusing (this may take a while)..."
    set function 0; # TELEOP_HI_RES_RESET
    set newPan [lindex [getHiResDerivedState] 0];
    set newTilt [lindex [getHiResDerivedState] 1];
    set newFOV [expr 4 * 3.1415927]; # 4 PI is an impossible FOV, so it should zoom out the camera as much as possible
    puts "zoomOut returned: [sendHiResRequest $function $newPan $newTilt $newFOV]"
}

proc resetPTU {} {
    puts "Resetting PTU (this may take a while)..."
    set function 3; # TELEOP_HI_RES_RESET
    set newPan 0;   # The following don't matter for a reset
    set newTilt 0;
    set newFOV 0;
    puts "resetPTU returned: [sendHiResRequest $function $newPan $newTilt $newFOV]"
}

proc stowArm {} {
    puts "Stowing arm..."
    puts "stowArm returned: [stowArm]"
}

# Radio buttons 
pack [canvas .rb] -fill both
radiobutton .rb.hiRes -command "CtrlSet setmux $highResMuxSetting" -text "High-Res Camera" -variable imageSrc -value HIGH_RES
radiobutton .rb.arm   -command "CtrlSet setmux $armMuxSetting"   -text "Arm Camera"      -variable imageSrc -value ARM
pack .rb.hiRes .rb.arm -side left -expand yes -fill both

# Buttons
pack [canvas .b] -fill both
button .b.run  -command "Run"  -text "Run"     -disabledforeground #808080
button .b.stop -command "Stop" -text "Stop"    -disabledforeground #808080
button .b.save -command "Save" -text "Save"    -disabledforeground #808080
button .b.more -command "More" -text "More..." -disabledforeground #808080
button .b.quit -command "Quit" -text "Quit"
pack .b.run .b.stop .b.save .b.more .b.quit -side left -expand yes -fill both



bind all <space> {if !$Saving {if $LiveVideo Stop else Run}}
bind all <s> {if !$Saving Save}
bind all <q> {if !$Saving Quit}
bind all <Control-c> {Quit}

set Saving 0
update
set imageSrc HIGH_RES
Run

