/*
 * pxc_statux.c -- get status information from a pxc device
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/sysmacros.h>

#include "pxc200.h"

/* A structure to declare how data is decoded */
struct pxc_info {
    int offset;              /* the data item to retrieve */
    unsigned long bitmask;   /* mask to apply to register */
    int type;                /* 'b' or 'i' */
    int shift;               /* number of shifts to retrieve number */
    char *string;            /* printf string */
    char **strings;          /* extra specs */
    int odd;                 /* must do it twice? */
};

/* Some values are split across two registers -- type is always 'i' */
struct pxc_info2 {
    int offset;              /* the data item to retrieve */
    unsigned long bitmask;   /* mask to apply to register */
    int offset2;             /* the data item to retrieve */
    unsigned long bitmask2;  /* mask to apply to register */
    int shift, shift2;
    char *string;            /* printf string */
    int odd;                 /* must do it twice? */
};

#define ONEBYTE 0xFF
#define TWOBYTES 0xFFFF
#define FOURBYTES 0xFFFFFFFF

static char *strings_xtal[] =
  {"Reserved","X0","X1","Auto"};
static char *strings_video[] =
  {"Auto","NTSC-M","NTSC-Japan","PAL","PAL-M","PAL-N","SECAM","Reserved"};

struct pxc_info dumpinfo[] = {
    {BT848_DSTATUS, BT848_DSTATUS_COF, 'b', 0,
	 "Status: Chroma overflow: %c", NULL, 0},
    {BT848_DSTATUS, BT848_DSTATUS_LOF, 'b', 0,
	 "Status: Luma overflow: %c", NULL, 0},
    {BT848_DSTATUS, BT848_DSTATUS_CSEL, 'i', BT848_DSTATUS_CSEL_S,
	 "Status: Xtal used: %i", NULL, 0},
    {BT848_DSTATUS, BT848_DSTATUS_NUML, 'b', 0,
	 "Status: Num-lines matches Pal: %c", NULL, 0},
    {BT848_DSTATUS, BT848_DSTATUS_FIELD, 'i', BT848_DSTATUS_FIELD_S,
	 "Status: Current field: %i", NULL, 0},
    {BT848_DSTATUS, BT848_DSTATUS_PRES, 'b', 0,
	 "Status: Video Present: %c", NULL, 0},

    {BT848_IFORM, BT848_IFORM_NORM, 'i', BT848_IFORM_S,
	 "Format: Video type: %i: %s", strings_video, 0},
    {BT848_IFORM, BT848_IFORM_XTSEL, 'i', BT848_IFORM_XTSEL_S,
	 "Format: Xtal mode: %i: %s", strings_xtal, 0},

    {BT848_TDEC, BT848_TDEC_DEC_RAT, 'i', 0, /* low bits */
	 "Decimate: Dropped per second: %i", NULL, 0},
    {BT848_TDEC, BT848_TDEC_FLDALIGN, 'b', 0,
	 "Decimate: Start on even field: %c", NULL, 0},
    {BT848_TDEC, BT848_TDEC_DEC_FIELD, 'b', 0,
	 "Decimate: Decimate fields: %c", NULL, 0},

    {BT848_INT_STAT, BT848_INT_RISCS, 'i', BT848_INT_RISCS_S,
	 "IrqStatus: Risc Status: %i",NULL, 0},
    {BT848_INT_STAT, BT848_INT_RISC_EN, 'b', 0,
	 "IrqStatus: Risc Enable: %c",NULL, 0},
    {BT848_INT_STAT, BT848_INT_FIELD, 'b', 0,
	 "IrqStatus: Even Field: %c",NULL, 0},
    {BT848_INT_STAT, BT848_INT_OCERR, 'b', 0,
	 "IrqStatus: Instruction Error: %c",NULL, 0},
    {BT848_INT_STAT, BT848_INT_PABORT, 'b', 0,
	 "IrqStatus: Pci Abort Received: %c",NULL, 0},
    {BT848_INT_STAT, BT848_INT_OFLOW, 'b', 0,
	 "IrqStatus: Luma or Chroma Overflow: %c",NULL, 0},

    {BT848_RISC_COUNT, FOURBYTES, 'x', 0,
	 "RiscIP: 0x%08x", NULL, 0},

    {BT848_GPIO_DMA_CTL, BT848_GPIO_DMA_CTL_RISC_ENABLE, 'b', 0,
	 "DMA Control: Risc enabled: %c", NULL, 0},
    {BT848_GPIO_DMA_CTL, BT848_GPIO_DMA_CTL_FIFO_ENABLE, 'b', 0,
	 "DMA Control: Fifo enabled: %c", NULL, 0},

    /* more to come... */

    {0, 0, 0, 0, NULL, NULL, 0}
    };

/* FIXME: left/right */
/* here, the first one is also shifted 8 places to the left */

/* format: off, mask, off, mask, shift, shift */
struct pxc_info2 dumpinfo2[] = {
    {BT848_E_CROP, BT848_CROP_HACTIVE,
       BT848_E_HACTIVE_LO, ONEBYTE,
       BT848_CROP_HACTIVE_S, 0,
             "Counter: Horizontally Active: %i", BT848_ODD_OFFSET},
    {BT848_E_CROP, BT848_CROP_HDELAY,
       BT848_E_HDELAY_LO, ONEBYTE,
       BT848_CROP_HDELAY_S, 0,
             "Counter: Horizontal Delay: %i", BT848_ODD_OFFSET},
    {BT848_E_CROP, BT848_CROP_VACTIVE,
       BT848_E_VACTIVE_LO, ONEBYTE,
       BT848_CROP_VACTIVE_S, 0,
             "Counter: Vertically Active: %i", BT848_ODD_OFFSET},
    {BT848_E_CROP, BT848_CROP_VDELAY,
       BT848_E_VDELAY_LO, ONEBYTE,
       BT848_CROP_VDELAY_S, 0,
             "Counter: Vertical Delay: %i", BT848_ODD_OFFSET},
    {BT848_E_HSCALE_HI, ONEBYTE,
       BT848_E_HSCALE_LO, ONEBYTE,
       0, 0,
	     "Counter: Horizontal scale: %i", BT848_ODD_OFFSET},

    /* more to come... */
    {0, 0, 0, 0, 0, 0, NULL}
    };


unsigned char buffer[BT848_NR_REGISTERS];

int main(int argc, char **argv)
{
    int fd=0, len;
    char *prgname;
    char *devname = "/dev/pxc0ctl";
    struct pxc_info *ptr;
    struct pxc_info2 *ptr2;
    unsigned int datum, datum2;
    int o; /* odd offset */
    struct stat statbuf;

    prgname = argv[0];
    if (argc > 1 && argv[1][0] == '/') { /* first */
	devname = argv[1];
	argc--; argv++;
    } else if (argc > 1 && argv[argc-1][0] == '/') { /* last */
	devname = argv[argc-1];
	argc--;
    } else if (getenv("PXCDEVICE")) {
	devname = getenv("PXCDEVICE");
    }

    fd=open(devname,O_RDONLY);
    if (fd<0) {
	fprintf(stderr,"%s: %s: %s\n",prgname, devname, strerror(errno));
	exit(1);
    }
    if (fstat(fd, &statbuf)<0) {
	fprintf(stderr,"%s: %s: stat(): %s\n",prgname, devname,
		strerror(errno));
	exit(1);
    }
    /* This check should use TYPE() and PX_CTL, but they are kernel-only */
    if (minor(statbuf.st_rdev)>>4 != 2) {
	fprintf(stderr,"%s: %s is not a control file\n",prgname, devname);
	exit(1);
    }
    switch (len=read(fd,buffer,BT848_NR_REGISTERS)) {
      case 0:
        fprintf(stderr,"%s: %s: no data\n", prgname, devname);
        exit(1);
      case -1:
        fprintf(stderr,"%s: %s: %s\n", prgname,  devname, strerror(errno));
        exit(1);
      default: break;
    }
    if (len < BT848_NR_REGISTERS) {
        buffer[len]='\0';
        fprintf(stderr,"%s: %s: short read\n",prgname, devname);
        exit(1);
    }

#if 0
    for (len=0; len<BT848_NR_REGISTERS; ) {
	printf("0x%02x",buffer[len]&0xff);
	len++;
	putchar(len%16? ' ':'\n');
    }
#endif

    /* First of all, single-register values */

    for (ptr = dumpinfo; ptr->string; ptr++) {
	/*
	 * Perform a little-endian read of the item.
	 */
	datum = buffer[ptr->offset]
	    | (buffer[ptr->offset+1]<<8)
		| (buffer[ptr->offset+2]<<16)
		    | (buffer[ptr->offset+3]<<24);
	datum = (datum & ptr->bitmask) >> ptr->shift;
	if (ptr->type=='b')
	    printf(ptr->string, datum ? 'y' : 'n');
	else
	    printf(ptr->string, datum,
		   ptr->strings ? ptr->strings[datum] : NULL);

        /* If ptr->odd is not 0, repeat for other field */
        if (ptr->odd) {
            o=ptr->odd;
            datum = buffer[ptr->offset+o]
                | (buffer[ptr->offset+o+1]<<8)
                    | (buffer[ptr->offset+o+2]<<16)
                        | (buffer[ptr->offset+o+3]<<24);
            datum = (datum & ptr->bitmask) >> ptr->shift;
            if (ptr->type=='b')
                printf(" %c", datum ? 'y' : 'n');
            else
                printf(ptr->strings ? " -- %i: %s" : " %i", datum,
		   ptr->strings ? ptr->strings[datum] : NULL);
        }
        putchar('\n');
    }

    /* Then, two-register counters */
    for (ptr2 = dumpinfo2; ptr2->string; ptr2++) {
	/*
	 * Perform a little-endian read of the items
	 */
	datum = buffer[ptr2->offset]
	    | (buffer[ptr2->offset+1]<<8)
		| (buffer[ptr2->offset+2]<<16)
		    | (buffer[ptr2->offset+3]<<24);
	datum2 = buffer[ptr2->offset2]
	    | (buffer[ptr2->offset2+1]<<8)
		| (buffer[ptr2->offset2+2]<<16)
		    | (buffer[ptr2->offset2+3]<<24);
        datum = (((datum & ptr2->bitmask) >> ptr2->shift) << 8)
            | ((datum2 & ptr2->bitmask2) >> ptr2->shift2);
        printf(ptr2->string, datum);

        /* If ptr->odd is not 0, repeat for other field */
        if (ptr2->odd) {
            o=ptr2->odd;
            datum = buffer[ptr2->offset+o]
                | (buffer[ptr2->offset+o+1]<<8)
                    | (buffer[ptr2->offset+o+2]<<16)
                        | (buffer[ptr2->offset+o+3]<<24);
            datum2 = buffer[ptr2->offset2+o]
                | (buffer[ptr2->offset2+o+1]<<8)
                    | (buffer[ptr2->offset2+o+2]<<16)
                        | (buffer[ptr2->offset2+o+3]<<24);
            datum = (((datum & ptr2->bitmask) >> ptr2->shift) << 8)
                | ((datum2 & ptr2->bitmask2) >> ptr2->shift2);
            printf(" %i", datum);
        }
        putchar('\n');
    }

    return 0;
}

