/*
 * pxc_grab.c -- acquire a sequence of images with the pxc grabber
 *
 * Copyright (C) 1998 Alessandro Rubini (rubini@linux.it)
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/time.h>

#include "pxc200.h"

char *prgname;

/* variables to retain options, set to default values */
int      opt_verbose    = 0;
int      opt_step       = 1; /* Step is 1: every other field or every frame */
int      opt_count      = 1; /* Default: acquire once */
int      opt_mem        = 1; /* Mem is one buffer worth of data */
char    *opt_output     = NULL;
char    *opt_geo        = NULL;

/* Filtering options */
double   opt_gain       = 1.0;
double   opt_add        = 0.0;
char    *opt_bgfile     = NULL;

#define VERBOSE(args) if (opt_verbose) fprintf args /* FIXME -- not clean */
#define PERROR(s) fprintf(stderr,"%s: %s: %s\n",prgname,(s),strerror(errno))

/*===================================================================*/
static int usage(char *what)
{
    printf("%s: %s is bad.\n", prgname, what);

    printf("Usage: %s [options]  <device_name>\n",prgname);
    printf("  Valid options are:\n"
	   " -v                be verbose on stderr\n"
	   " -q                quiet -- not verbose (default)\n"
	   " -s <n>            step: repeat grabbing every that many frames\n"
	   " -n <int>          acquire that many images\n"
	   " -M <int>          memory: grab to a buffer of <int> frames\n"
	   " -o <dirname>      outputfile radix (default: stdout)\n"
	   " -r <widxhei+x+y>  resolution (geometry)\n"
	   " -b <pgm-file>     subtract <pgm-file> as background\n"
	   " -g <float>        multiply input image by <gain>\n"
	   " -a <float>        add <float> to image (1==white)\n"
	);
    return 1;
}

/*===================================================================*/
int main(int argc, char **argv)
{
    static char ofname[256];
    char options[]="vqs:n:M:o:r:g:a:b:";
    int i, opt, fd, x, y;
    unsigned long hwover1, hwover2, swover1, swover2;
    double pixel;
    FILE *f;
    struct stat stat_buf;
    int ofnamelen=0;
    FILE *fout=NULL;
    unsigned char *buffer, *bgbuffer;
    Px_Roi roi;
    Px_AcqControl ctrl;
    int wid, hei, imgsize;
    static char pgm_header[80];
    struct timeval tv;
    char *devname = "/dev/pxc0pgm";

    prgname=argv[0];
    if (argc > 1 && argv[1][0] == '/') { /* first */
	devname = argv[1];
	argc--; argv++;
	argv[0] = prgname; /* for getopt() */
    } else if (argc > 1 && argv[argc-1][0] == '/') { /* last */
	devname = argv[argc-1];
	argc--;
    } else if (getenv("PXCDEVICE")) {
	devname = getenv("PXCDEVICE");
    }
    memset(&roi, 0, sizeof(Px_Roi));

    if ((fd=open(devname,O_RDWR))<0) {
	PERROR(devname); exit(1);
    }
    VERBOSE((stderr, "devname: %s\n", devname));
  
    while ((opt = getopt(argc, argv, options)) != -1) {

	switch (opt) {
 	    case 'v': opt_verbose=1; continue;
	    case 'q': opt_verbose=0; continue;
      
	    case 's':
		if (sscanf(optarg,"%i%s", &opt_step, optarg)!=1)
		    exit(usage("\"-c\" argument"));
		continue;
	    case 'n':
		if (sscanf(optarg,"%i%s", &opt_count, optarg)!=1)
		    exit(usage("\"-t\" argument"));
		continue;
	    case 'M':
		if (sscanf(optarg,"%i%s", &opt_mem, optarg)!=1)
		    exit(usage("\"-M\" argument"));
		continue;

	    case 'o': opt_output = optarg; continue;
	    case 'r': opt_geo = optarg; continue;

	    case 'b': opt_bgfile = optarg; continue;

 	    case 'g':
		if (sscanf(optarg,"%lf%s", &opt_gain, optarg)!=1)
		    exit(usage("\"-g\" argument"));
		continue;
 	    case 'a':
		if (sscanf(optarg,"%lf%s", &opt_add, optarg)!=1)
		    exit(usage("\"-g\" argument"));
		continue;

	    default: exit(usage("commandline"));
	}
    }

    /*
     * Options done, go ahead
     */

    if (!(f=fopen(devname,"r"))) {
	fprintf(stderr, "%s: %s: %s\n", prgname, devname, strerror(errno));
	exit(1);
    }
    if (fscanf(f,"P5 %i %i", &wid, &hei)!=2) {
	fprintf(stderr, "%s: %s: Not a PGM file\n", prgname, devname);
	exit(1);
    }
    fclose(f); /* Note: fd remains open, so this is low-overhead */

    roi.wid = wid; roi.hei = hei; /* default (x0==y0==0) */
    if (opt_geo) {
	if (sscanf(opt_geo,"%dx%d+%d+%d", &roi.wid, &roi.hei,
		   &roi.x0, &roi.y0) != 4) {
	    fprintf(stderr,"%s: Wrong geometry: \"%s\"\n", prgname, opt_geo);
	    exit(1);
	}
    }

    if ((roi.hei+roi.y0 > hei) || (roi.wid+roi.x0 > wid)) {
	fprintf(stderr,"%s: The ROI \"%s\" is out of bounds\n",
		prgname, opt_geo);
	exit(1);
    }

    VERBOSE((stderr, "roi: %i x %i + %i +%i\n",
	     roi.wid, roi.hei, roi.x0, roi.y0));
    imgsize = wid * hei; /* whole image */
    buffer = malloc(imgsize);
    bgbuffer = malloc(imgsize);
    if (!buffer || !bgbuffer) {
	fprintf(stderr,"%s: malloc(): %s\n", prgname, opt_geo);
	exit(1);
    }
    memset(bgbuffer, 0, wid*hei);

    /* Read the background image, if one is there */
    if (opt_bgfile) {
	if (!(f=fopen(opt_bgfile,"r"))) {
	    fprintf(stderr, "%s: %s: %s\n", prgname, opt_bgfile,
		    strerror(errno));
	    exit(1);
	}
	if (fscanf(f,"P5 %i %i %i", &wid, &hei, &i /* unused */) != 3) {
	    fprintf(stderr, "%s: %s: Not a PGM file\n", prgname, opt_bgfile);
	    exit(1);
	}
	while (getc(f)!='\n')
	    /* Skip to the next newline */ ;
	if (fread(bgbuffer, 1, imgsize, f) != imgsize) {
	    fprintf(stderr, "%s: %s: Short read\n", prgname, opt_bgfile);
	    exit(1);
	}
	fclose(f);
    }

    if (opt_output)  { /* otherwise, use stdout */
	strcpy(ofname,opt_output);
	if (opt_count) { /* more than once, use a directory */
	    if (!stat(ofname, &stat_buf)) {
		fprintf(stderr, "%s: %s: Already exists\n",prgname, ofname);
		exit(1);
	    }
	    if (mkdir(ofname,0777)) {
		fprintf(stderr, "%s: mkdir %s: %s\n",prgname,
			ofname, strerror(errno));
		exit(1);
	    }
	    strcat(ofname,"/");
	    strcat(ofname,"sequence.");
	    ofnamelen=strlen(ofname); /* used to add numbers */
	}
    } else fout = stdout;

    /* Create your pgm header */
    sprintf(pgm_header,"P5 %i %i 255\n",roi.wid,roi.hei);

    /* Fill the descriptive structure */
    memset(&ctrl, 0, sizeof(Px_AcqControl));
    ctrl.buflen = opt_mem;
    ctrl.count = opt_count;
    ctrl.step = opt_step;

#if 0
    /* Record current overrun events */ 
    if (ioctl(fd, PX_IOCGHWOVERRUN, &hwover1) 
	|| ioctl(fd, PX_IOCGSWOVERRUN, &swover1)) {
	fprintf(stderr, "%s: %s: getoverrun: %s\n",prgname,
		devname, strerror(errno));
	exit(1);
    }
#endif

    VERBOSE((stderr,"hw o.r.: %li, sw o.r.: %li\n",hwover1,swover1));
    
    /* Tell the driver about */ 
    gettimeofday(&tv, NULL);
    VERBOSE((stderr, "ioctl at time %03li.%06li\n",
	     (long)(tv.tv_sec%1000), (long)tv.tv_usec));
   if (ioctl(fd, PX_IOCSEQUENCE, &ctrl)) {
	fprintf(stderr, "%s: %s: getsequence: %s\n",prgname,
		devname, strerror(errno));
	exit(1);
    }

    /* make your life easier by using a file pointer */
    f = fdopen(fd,"r");
    if (!f) {
	fprintf(stderr, "%s: fdopen(%s): %s\n",prgname,
		devname, strerror(errno));
	exit(1);
    }

    for (i=0; i<opt_count; i++) {
	
	/* Prepare next output file */
	if (opt_count) {
	    if (fout) fclose(fout); fout = NULL;
	    sprintf(ofname+ofnamelen,"%03d",i);
	    fout=fopen(ofname,"w");
	    if (!fout) { PERROR(ofname); continue; }
	    fprintf(fout, pgm_header);
	}
	else if (!fout) {
	    fout=fopen(ofname,"w");
	    if (!fout) { PERROR(ofname); continue; }
	}
	if (fread(buffer, 1, imgsize, f) != imgsize) {
	    PERROR("short read");
	    fclose(fout); fout=NULL;
	    continue; /* Aaargh!!! FIXME: the header is there */
	}

	gettimeofday(&tv, NULL);
	VERBOSE((stderr, "acquire to \"%s\" at time %03li.%06li\n",
		 ofname, (long)(tv.tv_sec%1000), (long)tv.tv_usec));
	/* Now write out your roi */
	for (y=roi.y0; y< roi.y0+roi.hei; y++)
	    for (x=roi.x0; x< roi.x0+roi.wid; x++) {
		/* Make the needed filtering, first */
		pixel =    (double)buffer[wid*y + x];
		pixel -= (double)bgbuffer[wid*y + x];
		pixel *= opt_gain;
		pixel += opt_add * 256.0;
		if (pixel < 0) pixel = 0; if (pixel > 255.0) pixel=255.0;
		putc((int)pixel, fout);
	    }

    }

    if (fout) fclose(fout);

#if 0
    /* Retrieve overrun counts and report any of them */
    if (ioctl(fd, PX_IOCGHWOVERRUN, &hwover2) 
	|| ioctl(fd, PX_IOCGSWOVERRUN, &swover2)) {
	fprintf(stderr, "%s: %s: getoverrun: %s (error ignored)\n",
		prgname, devname, strerror(errno));
    } else {
	if (hwover2 != hwover1) {
	    fprintf(stderr,"%s: detected %li hardware overruns\n",
		    prgname, hwover2-hwover1);
	}
	if (swover2 != swover1) {
	    fprintf(stderr,"%s: detected %li buffer overflows\n",
		    prgname, swover2-swover1);
	}
    }
#endif
    return 0;
}

