/* 
    view.c - gtk video viewing program, 
           for use with pxc-200 driver, Copyright (c) Alessandro Rubini 

    Copyright (C) 1999 Chanop Silpa-Anan (chanop.silpa-anan@faceng.anu.edu.au)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <sys/time.h>
#include <unistd.h>

#include <gtk/gtk.h>

#include "pxc200.h"
#include <sys/ioctl.h>

/* This Program & Driver use S-video and RGB 24 color format from NTSC video*/

//#define HIRES  /* Low res or Hi res */

#ifdef HIRES

#define WIDTH 640
#define HEIGHT 480
#define B (WIDTH * HEIGHT * 3) /* 640 * 480 * 3 -> RGB 24 */
#define HOWMANY (WIDTH * HEIGHT * 3) /* 640 * 480 * 3 -> RGB 24 */
#define PXC200_0 "/dev/pxc0Hrgb"
#define PXC200_1 "/dev/pxc1Hrgb"

#else

#define WIDTH 320
#define HEIGHT 240
#define B (WIDTH * HEIGHT * 3) /* 320 * 240 * 3 -> RGB 24 */
#define HOWMANY (WIDTH * HEIGHT * 3) /* 320 * 240 * 3 -> RGB 24 */
#define PXC200_0 "/dev/pxc0rgb"
#define PXC200_1 "/dev/pxc1rgb"

#endif

/* Data passing to callback function */
typedef struct _Info {

  GtkWidget * view_port; 
  GtkWidget * view_area;
  FILE *f1;
  gint fd1;
  unsigned char image [B];

  gboolean view_on;
  gint timer;

  gdouble start_time, total_time;
  gint frame;
} Info;

gdouble get_time (void) {
  struct timeval tv;
  struct timezone tz;

  gettimeofday (&tv, &tz);

  return tv.tv_sec + 1e-6 * tv.tv_usec;
}

/* quit function callback */
static void quit_application ( GtkWidget * widget, GdkEvent * event, gpointer * data ) {
  gtk_main_quit ();
}

/* initialize function */
static void init ( Info * data ) {
  data->view_on = FALSE;
  data->timer = 0;
  data->frame = 0;

  data->f1 = fopen (PXC200_0, "r+");
  if (!data->f1)
    {
      perror (PXC200_0);
      exit (1);
    }

  data->fd1 = fileno (data->f1);
  
}

/* callback fuction for capturing video */
static int view ( Info * data ) {
  //gint i,j;
  gint tot;

  //unsigned char * im_ptr, swapbr;

  /* read from binary stream */
  for (tot = 0; tot < HOWMANY; tot += read (data->fd1, data->image + tot, HOWMANY - tot))
    ;
  
  /* swap blue and red  */
  /*
  // reading from pxcxxrgb is faster!!!
  im_ptr = data->image;
  for ( j = 0; j < HEIGHT; j++ ) {
    for (i = 0; i < WIDTH; i++ ) {
      swapbr = *im_ptr;
      *im_ptr = *(im_ptr+2);
      *(im_ptr+2) = swapbr;
      im_ptr += 3;
    }
  }    
  */  
  /* use gdk_draw_rgb_image to draw the image */
  gdk_draw_rgb_image (data->view_area->window,
		      data->view_area->style->white_gc,
		      0, 0, WIDTH, HEIGHT,
		      GDK_RGB_DITHER_NONE,
		      data->image, WIDTH * 3);    
  
  data->frame++;

  return TRUE;
}

/* callback fucntion for toggling capturing video */
static void view_toggle ( GtkWidget * widget, Info * data ) {
  if (GTK_TOGGLE_BUTTON (widget)->active) 
    {
      data->view_on=TRUE;
      data->timer = gtk_timeout_add(10, GTK_SIGNAL_FUNC (view), data);
      data->start_time = get_time ();
    } else {
      if (data->view_on) {
	data->total_time = get_time () - data->start_time;
	data->view_on=FALSE;
	gtk_timeout_remove ( data->timer );
	data->timer = 0;
	g_print ("time elapsed: %.2fs, %.1f fps, %.2f ms/frame\n",
		 data->total_time,
		 data->frame / data->total_time,
		 data->total_time*1000/data->frame);
	data->frame=0;
      }
    }
}

static int bright_control ( GtkAdjustment * adj, Info * data ) {

  unsigned long nums[4];
  nums[0] = (int) adj->value;

  ioctl(data->fd1, PX_IOCSBRIGHT, nums);
  
  return TRUE;
}  

static int contrast_control ( GtkAdjustment * adj, Info * data ) {

  unsigned long nums[4];
  nums[0] = (int) adj->value;

  ioctl(data->fd1, PX_IOCSCONTRAST, nums);
  
  return TRUE;
}  

static int hue_control ( GtkAdjustment * adj, Info * data ) {

  unsigned long nums[4];
  nums[0] = (int) adj->value;

  ioctl(data->fd1, PX_IOCSHUE, nums);
  
  return TRUE;
}  

static int sat_u_control ( GtkAdjustment * adj, Info * data ) {

  unsigned long nums[4];
  nums[0] = (int) adj->value;

  ioctl(data->fd1, PX_IOCSSATU, nums);
  
  return TRUE;
}  

static int sat_v_control ( GtkAdjustment * adj, Info * data ) {

  unsigned long nums[4];
  nums[0] = (int) adj->value;

  ioctl(data->fd1, PX_IOCSSATV, nums);
  
  return TRUE;
}  

static int sat_control ( GtkAdjustment * adj, Info * data ) {

  unsigned long nums[4];

  nums[0] = (int) adj->value * 0xfe / 100;

  ioctl(data->fd1, PX_IOCSSATU, nums);

  nums[0] = (int) adj->value * 0xb4 / 100;

  ioctl(data->fd1, PX_IOCSSATV, nums);
  
  return TRUE;
}  

static int reset_pic_control ( GtkAdjustment * adj, Info * data ) {

  unsigned long nums[4];

  nums[0] = 0x20;
  ioctl(data->fd1, PX_IOCSBRIGHT, nums);

  nums[0] = 0xd8;
  ioctl(data->fd1, PX_IOCSCONTRAST, nums);

  nums[0] = 0;
  ioctl(data->fd1, PX_IOCSHUE, nums);

  nums[0] = 0xfe;


  ioctl(data->fd1, PX_IOCSSATU, nums);
  
  nums[0] = 0xb4;
  ioctl(data->fd1, PX_IOCSSATV, nums);

  return TRUE;
}  



static int pic_control ( GtkWidget * widget, Info * data ) {
  static GtkWidget *window = NULL;

  GtkWidget * vbox;
  GtkObject * adjustment;
  GtkWidget * button;
  GtkWidget * scale;
  //GtkWidget * scrollbar;
  GtkWidget * label;

  unsigned long nums[4];

  if (!window)
    {
      window = gtk_window_new (GTK_WINDOW_TOPLEVEL);

      gtk_signal_connect (GTK_OBJECT (window), "destroy",
			  GTK_SIGNAL_FUNC (gtk_widget_destroyed),
			  &window);

      gtk_window_set_title (GTK_WINDOW (window), "Picture Control");
      gtk_container_set_border_width (GTK_CONTAINER (window), 3);

      vbox = gtk_vbox_new (FALSE, 0);
      gtk_container_add ( GTK_CONTAINER (window), vbox );

      label = gtk_label_new ( "Brightness" );
      gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);

      ioctl (data->fd1, PX_IOCGBRIGHT, nums);

      adjustment = gtk_adjustment_new ((float) (char) nums[0] , -128., 132., 1., 5., 5.);

      scale = gtk_hscale_new (GTK_ADJUSTMENT (adjustment));
      gtk_widget_set_usize (GTK_WIDGET (scale), 200, 30);
      gtk_range_set_update_policy (GTK_RANGE (scale), GTK_UPDATE_CONTINUOUS);
      gtk_scale_set_digits (GTK_SCALE (scale), 0);
      gtk_scale_set_draw_value (GTK_SCALE (scale), TRUE);
      gtk_box_pack_start (GTK_BOX (vbox), scale, TRUE, TRUE, 0);
      gtk_widget_show (scale);

#if 0      
      scrollbar = gtk_hscrollbar_new (GTK_ADJUSTMENT (adjustment));
      gtk_widget_set_usize (GTK_WIDGET (scrollbar), 200, 30);
      gtk_range_set_update_policy (GTK_RANGE (scrollbar), 
				   GTK_UPDATE_CONTINUOUS);
      gtk_box_pack_start (GTK_BOX (vbox), scrollbar, TRUE, TRUE, 0);
      gtk_widget_show (scrollbar);
#endif

      gtk_signal_connect (GTK_OBJECT (adjustment), "value_changed",
			  (GtkSignalFunc) (bright_control), data);
 
      label = gtk_label_new ( "Contrast" );
      gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);

      ioctl (data->fd1, PX_IOCGCONTRAST, nums);

      adjustment = gtk_adjustment_new ((float) nums[0] , 0., 516., 1., 5., 5.);

      scale = gtk_hscale_new (GTK_ADJUSTMENT (adjustment));
      gtk_widget_set_usize (GTK_WIDGET (scale), 200, 30);
      gtk_range_set_update_policy (GTK_RANGE (scale), GTK_UPDATE_CONTINUOUS);
      gtk_scale_set_digits (GTK_SCALE (scale), 0);
      gtk_scale_set_draw_value (GTK_SCALE (scale), TRUE);
      gtk_box_pack_start (GTK_BOX (vbox), scale, TRUE, TRUE, 0);
      gtk_widget_show (scale);

      gtk_signal_connect (GTK_OBJECT (adjustment), "value_changed",
			  (GtkSignalFunc) (contrast_control), data);
 
      label = gtk_label_new ( "Hue" );
      gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);

      ioctl (data->fd1, PX_IOCGHUE, nums);

      adjustment = gtk_adjustment_new ((float) (char) nums[0] , -127., 132., 1., 5., 5.);

      scale = gtk_hscale_new (GTK_ADJUSTMENT (adjustment));
      gtk_widget_set_usize (GTK_WIDGET (scale), 200, 30);
      gtk_range_set_update_policy (GTK_RANGE (scale), GTK_UPDATE_CONTINUOUS);
      gtk_scale_set_digits (GTK_SCALE (scale), 0);
      gtk_scale_set_draw_value (GTK_SCALE (scale), TRUE);
      gtk_box_pack_start (GTK_BOX (vbox), scale, TRUE, TRUE, 0);
      gtk_widget_show (scale);

      gtk_signal_connect (GTK_OBJECT (adjustment), "value_changed",
			  (GtkSignalFunc) (hue_control), data);
 
      label = gtk_label_new ( "Sat U" );
      gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);

      ioctl (data->fd1, PX_IOCGSATU, nums);

      adjustment = gtk_adjustment_new ((float) nums[0] , 0., 516., 1., 5., 5.);

      scale = gtk_hscale_new (GTK_ADJUSTMENT (adjustment));
      gtk_widget_set_usize (GTK_WIDGET (scale), 200, 30);
      gtk_range_set_update_policy (GTK_RANGE (scale), GTK_UPDATE_CONTINUOUS);
      gtk_scale_set_digits (GTK_SCALE (scale), 0);
      gtk_scale_set_draw_value (GTK_SCALE (scale), TRUE);
      gtk_box_pack_start (GTK_BOX (vbox), scale, TRUE, TRUE, 0);
      gtk_widget_show (scale);

      gtk_signal_connect (GTK_OBJECT (adjustment), "value_changed",
			  (GtkSignalFunc) (sat_u_control), data);

      label = gtk_label_new ( "Sat V" );
      gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);

      ioctl (data->fd1, PX_IOCGSATV, nums);

      adjustment = gtk_adjustment_new ((float) nums[0] , 0., 516., 1., 5., 5.);

      scale = gtk_hscale_new (GTK_ADJUSTMENT (adjustment));
      gtk_widget_set_usize (GTK_WIDGET (scale), 200, 30);
      gtk_range_set_update_policy (GTK_RANGE (scale), GTK_UPDATE_CONTINUOUS);
      gtk_scale_set_digits (GTK_SCALE (scale), 0);
      gtk_scale_set_draw_value (GTK_SCALE (scale), TRUE);
      gtk_box_pack_start (GTK_BOX (vbox), scale, TRUE, TRUE, 0);
      gtk_widget_show (scale);

      gtk_signal_connect (GTK_OBJECT (adjustment), "value_changed",
			  (GtkSignalFunc) (sat_v_control), data);

      label = gtk_label_new ( "Sat" );
      gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);

      adjustment = gtk_adjustment_new ( 100. , 0., 205., 1., 5., 5.);

      scale = gtk_hscale_new (GTK_ADJUSTMENT (adjustment));
      gtk_widget_set_usize (GTK_WIDGET (scale), 200, 30);
      gtk_range_set_update_policy (GTK_RANGE (scale), GTK_UPDATE_CONTINUOUS);
      gtk_scale_set_digits (GTK_SCALE (scale), 0);
      gtk_scale_set_draw_value (GTK_SCALE (scale), TRUE);
      gtk_box_pack_start (GTK_BOX (vbox), scale, TRUE, TRUE, 0);
      gtk_widget_show (scale);

      gtk_signal_connect (GTK_OBJECT (adjustment), "value_changed",
			  (GtkSignalFunc) (sat_control), data);

      button = gtk_button_new_with_label ("Reset");
      gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 0);
      gtk_signal_connect (GTK_OBJECT (button), "clicked",
			  (GtkSignalFunc) (reset_pic_control), data);
      gtk_widget_show (button);

      gtk_widget_show ( vbox );
    }

  if (!GTK_WIDGET_VISIBLE (window))
    gtk_widget_show_all (window);
  else
    gtk_widget_destroy (window);

  return TRUE;
}

static void main_window () {
  GtkWidget * window;
  GtkWidget * button;
  GtkWidget * vbox;
  GtkWidget * separator;

  GtkWidget * view_port;
  GtkWidget * view_area;
  GtkWidget * h_control_bar;

  Info * data;
 
  
  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (window), "PXC200 Frame Grabber");
  gtk_container_border_width (GTK_CONTAINER (window), 5);
   
  gtk_signal_connect (GTK_OBJECT (window), "delete_event",
		      GTK_SIGNAL_FUNC (quit_application), NULL);

  vbox = gtk_vbox_new (FALSE, 5);
  gtk_container_add (GTK_CONTAINER (window), vbox);

  view_port = gtk_hbox_new (FALSE, 5);
  gtk_box_pack_start (GTK_BOX (vbox), view_port, FALSE, FALSE, 0);

  view_area = gtk_drawing_area_new ();
  gtk_widget_set_usize (view_area, WIDTH, HEIGHT);
  gtk_box_pack_start (GTK_BOX (view_port), view_area, FALSE, FALSE, 0);
  gtk_widget_show (view_area);
  
  gtk_widget_show (view_port);

  separator = gtk_hseparator_new ();
  gtk_box_pack_start (GTK_BOX (vbox), separator, FALSE, TRUE, 0);
  gtk_widget_show (separator);

  h_control_bar = gtk_hbox_new (FALSE, 5);
  gtk_box_pack_start (GTK_BOX (vbox), h_control_bar, FALSE, FALSE, 0);

  button = gtk_button_new_with_label ("Quit");
  gtk_box_pack_start (GTK_BOX (h_control_bar), button, FALSE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
			     (GtkSignalFunc) (quit_application), NULL);
  gtk_widget_show (button);

  data = g_new (Info, 1);
  data->view_port = view_port;
  data->view_area = view_area;
 
  init ( data );

  button = gtk_toggle_button_new_with_label ("Start / Stop");
  gtk_box_pack_start (GTK_BOX (h_control_bar), button, FALSE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
			     (GtkSignalFunc) (view_toggle), data); 
  gtk_widget_show (button);

  button = gtk_button_new_with_label ("Picture Control");
  gtk_box_pack_start (GTK_BOX (h_control_bar), button, FALSE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
			     (GtkSignalFunc) (pic_control), data);
  gtk_widget_show (button);

  gtk_widget_show (h_control_bar);  

  gtk_widget_show (vbox);
 
  gtk_widget_show (window);

 
}


int main (int argc, char *argv[]) {
  gtk_init (&argc, &argv);

  gdk_rgb_set_verbose (TRUE);

  gdk_rgb_init ();

  gtk_widget_set_default_colormap (gdk_rgb_get_cmap ());
  gtk_widget_set_default_visual (gdk_rgb_get_visual ());

  main_window ();

  gtk_main ();

  return 0;
}










