#include <sys/ioctl.h>
#include <asm/ioctl.h>
#include <asm/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>

#include "../driver/nicts.h"
#include "nidaq.h"

static struct {
  int aih, nch;
  __u16 gains[NI_MAX_AICH], polars[NI_MAX_AICH];
} ai[NI_MAX_BOARDS];

int LxAiOpen(__u16 device)
{
  char fname[20];
  if(device<1 || device>NI_MAX_BOARDS) return -1;
  if(ai[device-1].aih>0)close(ai[device-1].aih);
  sprintf(fname,"/dev/nidaq%d",device-1);
  ai[device-1].aih=open(fname,O_RDONLY);
  LOG("device %s ",fname);LOG("ret=%d\n",ai[device-1].aih);
  return ai[device-1].aih;
}

int LxAiCloseAll()
{
  int i;
  for(i=0;i<NI_MAX_BOARDS;i++)if(ai[i].aih>0)close(ai[i].aih);
  return 0;
}

int static fh2dev(__s32 fh)
{
  int i;
  for(i=0;i<NI_MAX_BOARDS;i++)if(ai[i].aih==fh)return i;
  return -1;
}

int static addchan(__s32 fh, __u16 channel, __u16 gain, __u16 pol,
		   __u16 refer, __u16 last)
{
  __u32 channel_descriptor = 0;

  channel_descriptor |=  ((channel & 0xf) << 16);
  channel_descriptor |=  (((refer+1) & 0x7) << 28);
  channel_descriptor |=  ((pol & 0x1) << 8);
  channel_descriptor |=  (gain & 0x7);
  if(last) channel_descriptor |=  0x8000;
  return ioctl(fh, NICTL_AI_ADDCH, channel_descriptor);
}

int LxAiScale(__s32 fh, __u32 nscans, __s16 rddata[], float voltages[])
{
  static double gmults[8]={20.0, 10.0, 5.0, 2.0, 1.0, 0.5, 0.2, 0.1};
  __u16 ch; __u32 i; double scdata;

  device_info dinfo;
  int dev, nch;

  if((dev=fh2dev(fh))<0) return -1;
  nch=ai[dev].nch;

  ioctl(fh, NICTL_DEVICE_INFO, &dinfo);
  for(i=0;i<nch*nscans;i++){
    ch=i%nch;
    scdata=ai[dev].polars[ch]?(__u16)rddata[i]:rddata[i];
    scdata /= (1L<<dinfo.aires);
    scdata *= gmults[ai[dev].gains[ch]&7];
    voltages[i]=scdata;
  }
  return 0;
}

#if 0
int LxAiRead(__s32 fh, __u16 channel, __u16 gain, __u16 polar, __s16 *reading)
{
  ioctl (fh, NICTL_AI_RESET, 0);
  addchan(fh, channel, gain, polar, 0, 1); 
  ioctl (fh, NICTL_AI_ONEPT_SETUP, 0);
  ioctl (fh, NICTL_AI_CONVERT, 0);
  return 0;
}

int LxAiVRead(__s32 fh, __u16 channel, __u16 gain,__u16 polar, double *voltage)
{
  __s16 rd;
  LxAiRead(fh, channel, gain, polar, &rd);
  aiscale(fh, 1, 1, &gain, &polar, &rd, voltage);
  return 0;
}
#endif

int LxAiBoardConfig(__s32 fh, __u16 nch, __u16 channels[], __u16 gains[],
		    __u16 polars[], __u16 refers[])
{
  int i,dev;

  if((dev=fh2dev(fh))<0) return -1;

  ioctl(fh,  NICTL_AI_RESET, 0);
  ai[dev].nch=nch;
  for(i=0; i<nch; i++){
    addchan(fh, channels[i], gains[i], polars[i],refers[i],i==nch-1);
    ai[dev].gains[i]=gains[i];
    ai[dev].polars[i]=polars[i];
  }
  return 0;
}

int LxAiStart(__s32 fh, __s32 no_of_scans)
{
  ioctl(fh, NICTL_AI_SET_SC, no_of_scans-1);

  ioctl(fh, NICTL_AI_SCAN_SETUP, 0);
  ioctl(fh, NICTL_AI_SCAN_START, 0);

  return 0;
}

int LxAiGetnch(__s32 fh)
{
  int dev;
  if((dev=fh2dev(fh))<0) return -1;
  return ai[dev].nch;
}

int LxAiRead(__s32 fh, __s32 no_scans, __s16 buff[], __s32 *scans_read)
{
  int bytesread, nch;
  int dev;

  if((dev=fh2dev(fh))<0) return -1;
  nch=ai[dev].nch;

  if((bytesread=read(fh, buff, 2*no_scans* nch))<0) return bytesread;
  *scans_read=bytesread/(nch *2);
 
  return 0;
}

int LxAiClear(__s32 fh)
{
  ioctl(fh, NICTL_AI_RESET, 0);
  return 0;
}

int LxAiTimeConfig(__s32 fh, double scan_freq, double spl_freq, 
		   double *act_scan_freq, double *act_spl_freq)
{
  __u32 si, si2; int dev;
  device_info dinfo;
  double clkfreq=20e6;

  if((dev=fh2dev(fh))<0) return -1;
  
  ioctl(fh, NICTL_DEVICE_INFO, &dinfo);

  if(spl_freq > dinfo.aimaxspl) return -10002;
  if(spl_freq <= 0) spl_freq=dinfo.aimaxspl;
  if(scan_freq > spl_freq / ai[dev].nch) return -10002;
  if(scan_freq <=0 ) scan_freq=spl_freq/ai[dev].nch;

  si=clkfreq/scan_freq; *act_scan_freq=clkfreq/si;
  si2=clkfreq/spl_freq; *act_spl_freq=clkfreq/si2;
  ioctl(fh, NICTL_AI_SET_SI, si-1);
  ioctl(fh, NICTL_AI_SET_SI2, si2-1);  
  return 0;
}

int LxAiSingleScan(__s32 fh, __s16 buff[])
{
  int bytesread, nch, dev;

  if((dev=fh2dev(fh))<0) return -10000;
  nch=ai[dev].nch;

  ioctl(fh, NICTL_AI_SET_SC, 0); /* 1 scan */
  ioctl(fh, NICTL_AI_SCAN_SETUP, 0);
  ioctl(fh, NICTL_AI_SCAN_START, 0);
  if((bytesread=read(fh, buff, 2* nch))<0) return bytesread;
  if(bytesread!=2*nch) return -10000;

  return 0;
}




