#include <linux/fs.h>
#include <linux/sched.h>

#include "eserrlp.h"
#include "daq_driver.h"
#include "misc.h"

static void GPCT_Reset_All(pd dev, int no);

static void GPCT_Arm_All(pd dev, int no);
static void GPCT_Disarm(pd dev, int no);
static void GPCT_Enable_Out(pd dev, int no);

static void GPCT_LoadA(pd dev, int no, unsigned long x);
static void GPCT_LoadB(pd dev, int no, unsigned long x);
static void GPCT_LoadUsingA(pd dev, int no, unsigned long x);
static void GPCT_Prepare(pd dev, int no, int mode);
static void GPCT_Read_HW_Regs(pd dev, int no, unsigned long *uCnt);
static void GPCT_Read_Soft_Regs(pd dev, int no, unsigned long *uCnt);

int gpct_read(pd dev, int no, struct file * file, char * buf, int count)
{ 
  gpct_private_data *gp = &dev->gc[no];

  /* if(gp->g_buff_done) return -EAGAIN; */
  if(gp->gate_error || gp->rollover || gp->fifo_overflow) return -ETIME;
  if(gp->pread==gp->pwrite) if(!gp->g_buff_done)
    interruptible_sleep_on(&gp->wait_queue); else return -ETIME;
  if(gp->gate_error || gp->rollover || gp->fifo_overflow) return -ETIME;
  if(gp->pread==gp->pwrite)return -EAGAIN;
  copy_to_user (buf, &gp->buff[gp->pread++], 4);
  if(gp->pread==GPCTBUFFSIZE)gp->pread=0;
  return 4;
}

int gpct_write(pd dev, int no, struct file * file, char const * buf, int count)
{ 
  return -EINVAL;
}

int gpct_ioctl(pd dev, int no, struct file *file, unsigned int cmd, unsigned long arg)
{
  gpct_private_data *gp = &dev->gc[no];
  unsigned long cnt, status;

  //  printk("gpct ioctl %d %ld\n", _IOC_NR(cmd), arg); 

  switch(_IOC_NR(cmd)){
  case _IOC_NR(NICTL_GPCT_RESET):
    GPCT_Reset_All(dev, no);
    gp->src_sel = gp->src_pol = gp->gate_sel = gp->gate_pol =
      gp->out_pol = 0; gp->out_mode =  gp->up_down =1;
    gp->icnt = gp->cnt1 = gp->cnt2 = gp->cnt3 =  gp->cnt4 =0;
    gp->pread=gp->pwrite=0; gp->g_buff_done=1; gp->tot_pts=GPCTBUFFSIZE;
    gp->pts_read=0;
    gp->gate_error = gp->rollover = gp->fifo_overflow = 0;
    gp->wait_queue=0;
    return 0;

  case _IOC_NR( NICTL_GPCT_SRC_SEL):
    gp->src_sel=arg;
    return 0;
  case _IOC_NR( NICTL_GPCT_SRC_POL):
    gp->src_pol=arg;
    return 0;
  case _IOC_NR( NICTL_GPCT_GATE_SEL):
    gp->gate_sel=arg;
    return 0;
  case _IOC_NR( NICTL_GPCT_GATE_POL):
    gp->gate_pol=arg;
    return 0;
  case _IOC_NR( NICTL_GPCT_OUT_MODE):
    gp->out_mode=arg;
    return 0;
  case _IOC_NR( NICTL_GPCT_OUT_POL):
    gp->out_pol=arg;
    return 0;
  case _IOC_NR( NICTL_GPCT_UP_DOWN):
    gp->up_down=arg;
    return 0;
  case _IOC_NR( NICTL_GPCT_SET_CNT1):
    gp->cnt1=arg;
    return 0;
  case _IOC_NR( NICTL_GPCT_SET_CNT2):
    gp->cnt2=arg;
    return 0;
  case _IOC_NR( NICTL_GPCT_SET_CNT3):
    gp->cnt3=arg;
    return 0;
  case _IOC_NR( NICTL_GPCT_SET_CNT4):
    gp->cnt4=arg;
    return 0;
  case _IOC_NR( NICTL_GPCT_SET_NPTS):
    gp->tot_pts=arg;
    return 0;

  case _IOC_NR( NICTL_GPCT_READ):
    if(gp->mode<10)
      GPCT_Read_Soft_Regs(dev, no, &cnt);
    else GPCT_Read_HW_Regs(dev, no, &cnt);
    copy_to_user ((unsigned long *)arg, &cnt, 4);
    return 0;
  case _IOC_NR( NICTL_GPCT_ARMED):
    status=!!(DAQ_STC_Windowed_Mode_Read(dev, G_Status_Register)&
	      (no?0x0200:0x0100));
    copy_to_user ((unsigned long *)arg, &status,4);
    return 0;

  case _IOC_NR(NICTL_GPCT_APP_PULSE_TRAIN):
    gp->mode=30; gp->up_down=0; gp->cnt3=1;
    gp->cnt1=5000000; gp->cnt2=10000000;
    return 0;
  case _IOC_NR(NICTL_GPCT_APP_PULSE):
    gp->mode=31; gp->up_down=0; gp->cnt1=1000000; gp->cnt3=1;
    return 0;
  case _IOC_NR(NICTL_GPCT_APP_PULSE_TRIG):
    gp->mode=32; gp->up_down=0; gp->cnt1=1000000; gp->cnt3=1;
    return 0;
  case _IOC_NR(NICTL_GPCT_APP_PULSE_RETRIG):
    gp->mode=33; gp->up_down=0; gp->cnt3=1;
    return 0;
     
  case _IOC_NR(NICTL_GPCT_APP_EVENT_CNT):
    gp->mode=1;
    return 0;

  case _IOC_NR(NICTL_GPCT_APP_PW_MEAS):
    gp->mode=10; return 0;
  case _IOC_NR(NICTL_GPCT_APP_PER_MEAS):
    gp->mode=11; return 0;
  case _IOC_NR(NICTL_GPCT_APP_BUF_PW_MEAS):
    gp->mode=21; return 0;
  case _IOC_NR(NICTL_GPCT_APP_BUF_PER_MEAS):
    gp->mode=22; return 0;
  case _IOC_NR(NICTL_GPCT_APP_BUF_SEM_MEAS):
    gp->mode=23; return 0;

  case _IOC_NR(NICTL_GPCT_PREPARE):
    GPCT_Reset_All(dev, no);
    MSC_IO_Pin_Configure(dev);
    MSC_Clock_Configure(dev);
    GPCT_Prepare(dev, no, gp->mode);
    return 0;

  case _IOC_NR(NICTL_GPCT_ARM):
    gp->g_buff_done=0;
    GPCT_Arm_All(dev, no);
    return 0;

  case _IOC_NR(NICTL_GPCT_DISARM):
    GPCT_Disarm(dev, no);
    return 0;
}

  return -EINVAL;
}

void gpct_interrupt(pd dev, int no)
{
  gpct_private_data *gp;
  unsigned short uGStatus;
  unsigned long uCnt;
  gp=&dev->gc[no];
    
  uCnt=DAQ_STC_Windowed_Mode_Read(dev, G0_HW_Save_Registers+1+2*no)
    |(DAQ_STC_Windowed_Mode_Read(dev, G0_HW_Save_Registers+2*no)<<16);
  
  uGStatus=DAQ_STC_Windowed_Mode_Read(dev, G_Status_Register);
  if(uGStatus & (no?0x0080:0x0040)) uCnt=0; /* stale data */

  if(!gp->g_buff_done){
    gp->buff[gp->pwrite++]= uCnt; if(gp->pwrite==GPCTBUFFSIZE) gp->pwrite=0;
    gp->pts_read++;
    if(gp->pwrite==gp->pread)gp->fifo_overflow=1;
  }

  /* ack the gate interrupt */
  DAQ_STC_Windowed_Mode_Strobe_Write(dev, no?Interrupt_B_Ack_Register:
				     Interrupt_A_Ack_Register,0x8000);
  
  if(uGStatus & (no?0x8000:0x4000)){ gp->gate_error=1; /* gate error */
  DAQ_STC_Windowed_Mode_Strobe_Write(dev,no?Interrupt_B_Ack_Register:
				     Interrupt_A_Ack_Register,no?0002:0x0020);
  }
  uGStatus=DAQ_STC_Windowed_Mode_Read(dev, no?AO_Status_1_Register:
				      AI_Status_1_Register);
  if(uGStatus & 0x0008){ gp->rollover=1; /* TC st */
  DAQ_STC_Windowed_Mode_Strobe_Write(dev,no?Interrupt_B_Ack_Register:
				     Interrupt_A_Ack_Register,0x4000);
  }
  if(gp->fifo_overflow || gp->gate_error || gp->rollover ||
     gp->pts_read>=gp->tot_pts) {
    /* Disarms the counter */
    DAQ_STC_Windowed_Mode_Strobe_Write(dev, G0_Command_Register+no, 0x0010);
    gp->g_buff_done=1;
  }

  wake_up_interruptible(&gp->wait_queue);

}

void GPCT_Read_HW_Regs(pd dev, int no, unsigned long *uCnt)
{
  *uCnt=DAQ_STC_Windowed_Mode_Read(dev, G0_HW_Save_Registers+1+2*no)
    |(DAQ_STC_Windowed_Mode_Read(dev, G0_HW_Save_Registers+2*no)<<16);
}
void GPCT_Read_Soft_Regs(pd dev, int no, unsigned long *uCnt)
{
  unsigned long save_1=0, save_2=0;
  /* Gi_Save_Trace<=0 */
  DAQ_STC_Windowed_Mode_Masked_Write(dev, G0_Command_Register+no,
				     0x0000, 0x0002);
  /* Gi_Save_Trace<=1  */
  DAQ_STC_Windowed_Mode_Masked_Write(dev, G0_Command_Register+no,
				     0x0002,0x0002);
  save_1=DAQ_STC_Windowed_Mode_Read(dev, G0_Save_Registers+1+2*no)
    |(DAQ_STC_Windowed_Mode_Read(dev, G0_Save_Registers+2*no)<<16);
  save_2=DAQ_STC_Windowed_Mode_Read(dev, G0_Save_Registers+1+2*no)
    |(DAQ_STC_Windowed_Mode_Read(dev, G0_Save_Registers+2*no)<<16);
  if(save_2 != save_1)
    save_1=DAQ_STC_Windowed_Mode_Read(dev, G0_Save_Registers+1+2*no)
      |(DAQ_STC_Windowed_Mode_Read(dev, G0_Save_Registers+2*no)<<16);
  *uCnt=save_1;
}

void GPCT_LoadUsingA(pd dev, int no, unsigned long x)
{
  /*  Writing to G0_Mode_Register with address 26
   *      G0_Load_Source_Select<=0
   *      pattern = 0x0000
   */
  DAQ_STC_Windowed_Mode_Masked_Write(dev, G0_Mode_Register+no,0x0000,0x0080);
  
  /*      Writing to the G0_Load_A_Register with address 28 */
  DAQ_STC_Windowed_Mode_Write(dev, G0_Load_A_Registers+4*no,
			      (x>>16) & 0x00ff);
  DAQ_STC_Windowed_Mode_Write(dev, G0_Load_A_Registers+1+4*no,
			      x & 0xffff);      
  /*      Writing to G0_Command_Register with address 6
   *      G0_Load <=1     
   *      pattern=0x0104
   */
  DAQ_STC_Windowed_Mode_Strobe_Write(dev, G0_Command_Register+no,0x0004);
}

void GPCT_LoadA(pd dev, int no, unsigned long x)
{
  DAQ_STC_Windowed_Mode_Write(dev, G0_Load_A_Registers+4*no,
			      (x>>16) & 0x00ff);
  DAQ_STC_Windowed_Mode_Write(dev, G0_Load_A_Registers+1+4*no,
			      x & 0xffff);
}
void GPCT_LoadB(pd dev, int no, unsigned long x)
{
  DAQ_STC_Windowed_Mode_Write(dev, G0_Load_B_Registers+4*no,
			      (x>>16) & 0x00ff);
  DAQ_STC_Windowed_Mode_Write(dev, G0_Load_B_Registers+1+4*no,
			      x & 0xffff);     
}

void GPCT_Prepare(pd dev, int no, int mode)
{
  gpct_private_data *gp = &dev->gc[no];

  unsigned short uMode=0, uComm=0, uInpsel=0,
    or_gate=0, gate_sel_load_src=0, /* in inpsel */
    bank_sw_en=0, bank_sw_mode=0, /* in comm */
    load_src_sel=0, reload_src_sw=0, load_on_gate=0, load_on_tc=0,
    gate_mode=0, gate_on_both_edges=0, trig_mode_for_edge_gate=0,
    stop_mode=0, counting_once=0;
  int gate_irq_en=0;

  switch (mode){
  case 1: /* simple event count OK */
    GPCT_LoadUsingA(dev, no, gp->icnt);
    gate_mode=1; trig_mode_for_edge_gate=2;
    break;
  case 10: /* simple pw meas DOESNT WORK  */ 
    GPCT_LoadUsingA(dev, no, gp->icnt);
    //load_on_gate=1; counting_once=2; trig_mode_for_edge_gate=2; gate_mode=1;
    load_on_gate=0; counting_once=2; trig_mode_for_edge_gate=2; gate_mode=1;
    break;
  case 11: /* simple period meas  OK */
    GPCT_LoadUsingA(dev, no, gp->icnt);
    reload_src_sw=1; counting_once=2; gate_mode=2;
    break;
  case 30: /* pulse train generation OK */
    GPCT_LoadUsingA(dev, no, gp->cnt3);  /*delay from trig */
    GPCT_LoadA(dev, no, gp->cnt1); /* interval */
    GPCT_LoadB(dev, no, gp->cnt2); /* pulse width */
    load_src_sel=1; gp->out_mode=2; reload_src_sw=1; load_on_tc=1;
    gate_mode=0; trig_mode_for_edge_gate=2;
    GPCT_Enable_Out(dev, no);
    break;

  case 31: case 32: case 33: /* pulse, trig. pulse & retrig_pulse OK */
    GPCT_LoadUsingA(dev, no, gp->cnt3); /* delay from trigger */
    GPCT_LoadB(dev, no, gp->cnt1);
    load_src_sel=1; gp->out_mode=2; reload_src_sw=1; load_on_tc=1;
    trig_mode_for_edge_gate=2; stop_mode=2;
    if(mode!=31)gate_mode=2; if(mode!=33)counting_once=1;
    GPCT_Enable_Out(dev, no);
    break;

  case 21: case 22: case 23: /* buff pw, per and semiper meas */
    GPCT_LoadUsingA(dev, no, gp->icnt);
    load_on_gate=1; reload_src_sw=0;
    if(mode==21)gate_mode=1; /* OK */
    if(mode==22)gate_mode=3; /* OK, it's period */
    if(mode==23){gate_mode=3; gate_on_both_edges=1;} /* DOESNT WORK */
    trig_mode_for_edge_gate=3;
    gate_irq_en=1;
    break;
  }

  uInpsel=(gp->src_sel<<2) | (gp->gate_sel<<7) | (gate_sel_load_src<<12) |
    (or_gate<<13) | (gp->out_pol<<14) | (gp->src_pol<<15);
  uComm=(gp->up_down<<5) | (bank_sw_mode<<11) | (bank_sw_en<<12);
  uMode=(gate_mode<<0) | (gate_on_both_edges<<2) |
    (trig_mode_for_edge_gate<<3) | (stop_mode<<5) | (load_src_sel<<7) |
    (gp->out_mode<<8) | (counting_once<<10) | (load_on_tc<<12) |
    (gp->gate_pol<<13) | (load_on_gate<<14) | (reload_src_sw<<15);

  DAQ_STC_Windowed_Mode_Masked_Write(dev,G0_Input_Select_Register+no,
				     uInpsel, 0xFFFC);
  DAQ_STC_Windowed_Mode_Masked_Write(dev,G0_Command_Register+no,
				     uComm, 0x8160); 
  DAQ_STC_Windowed_Mode_Masked_Write(dev, G0_Mode_Register+no,
				     uMode, 0xFFFF);
  //  printk("g%1d mode=%x\n",no,uMode);
  if(gate_irq_en) if(!no)
    DAQ_STC_Windowed_Mode_Masked_Write(dev, Interrupt_A_Enable_Register,
				       0x0100, 0x0100);
  else
    DAQ_STC_Windowed_Mode_Masked_Write(dev, Interrupt_B_Enable_Register,
				       0x0400, 0x0400); 
}

void GPCT_Reset_All(pd dev, int no) /*** AS IN STC MANUAL **/
{
  /* Reset Gi */
  DAQ_STC_Windowed_Mode_Strobe_Write(dev,Joint_Reset_Register,no?0x0008:0x0004); 

  /*      Clear G0_Mode_Register with address 26
   *      G0_Mode_Register<=0x0000
   *      pattern=0x0000 
   */
  DAQ_STC_Windowed_Mode_Write(dev, G0_Mode_Register+no,0x0000);
   
  /*  Clear G0_Command_Register with address 6
   *      G0_Command_Register<=0x0000
   *      pattern=0x0000
   */
  DAQ_STC_Windowed_Mode_Write(dev,G0_Command_Register+no,0x0000); 
        
  /*  Clear GO_Input_Select_Register with address 36
   *      G0_Input_Select_Register<=0x0000
   *      pattern=0x0000
   */
  DAQ_STC_Windowed_Mode_Write(dev,G0_Input_Select_Register+no,0x0000);
  
  /*  Clear G0_Autoincrement_Register with address 68
   *      G0_Autoincrement_Register<=0x0000
   *      pattern=0x0000
   */
  DAQ_STC_Windowed_Mode_Write(dev,G0_Autoincrement_Register+no,0x0000);
  
  /*  Writing to Interrupt_A_Enable_Register with address 73
 *      G0_TC_Interrupt_Enable<=0
 *      G0_Gate_Interrupt_Enble<=0
 *      pattern=0x0000
 */
  if(!no)
    DAQ_STC_Windowed_Mode_Masked_Write(dev, Interrupt_A_Enable_Register,
				       0x0000, 0x0140);
  else
    DAQ_STC_Windowed_Mode_Masked_Write(dev, Interrupt_B_Enable_Register,
				       0x0000, 0x0600);

  /*  Writing to G0_Command_Register with address 6
   *      GO_Synchronized_Gate<=1
   *      pattern =0x0100
   */
  DAQ_STC_Windowed_Mode_Masked_Write(dev, G0_Command_Register+no,0x0100,0x0100);

  /*  Writing to Interrupt_A_Ack_Register with address 2
   *      G0_Gate_Error_Confirm<=1
   *  G0_TC_Error_Confirm<=1
   *      G0_TC_Interrupt_Ack<=1
   *  G0_Gate_Interrupt_Ack<=1
   *  pattern=0xC060
   */
  if(!no)DAQ_STC_Windowed_Mode_Masked_Write(dev, Interrupt_A_Ack_Register,
					    0xC060, 0xC060);
  else DAQ_STC_Windowed_Mode_Masked_Write(dev, Interrupt_B_Ack_Register,
					    0xC006, 0xC006);

/*  Writing to G0_Autoincrement_Register with address 68
 *      GO_Autoincrement<=0
 *      pattern=0x0000 
 */
  DAQ_STC_Windowed_Mode_Write(dev, G0_Autoincrement_Register+no,0x0000);

}

void GPCT_Arm_All(pd dev, int no){
  /*  Writing to G0_Command_Register with address 6
   *      G0_Arm<=1
   */
  DAQ_STC_Windowed_Mode_Masked_Write(dev, G0_Command_Register+no,1,1);
 }

void GPCT_Enable_Out(pd dev, int no){
/* Writing to Analog_Trigger_Etc_Register with address 61
 * GPFO_i_Output_Enable<=1
 * GPFO_i_Output_Select <=0
 */
   if(!no) DAQ_STC_Windowed_Mode_Masked_Write(dev, Analog_Trigger_Etc_Register,
					      0x4000, 0x7800);
   else DAQ_STC_Windowed_Mode_Masked_Write(dev, Analog_Trigger_Etc_Register,
					      0x8000, 0x8080);
}
      
void GPCT_Disarm(pd dev, int no)
{
  /* Disarms the counter */
  DAQ_STC_Windowed_Mode_Strobe_Write(dev, G0_Command_Register+no, 0x0010);
}



