/*********************************************************************
 * Distinct ONC RPC/XDR for Java
 * Copyright 1997 - 1999 by Distinct Corporation. All rights reserved.
 *********************************************************************/

import java.io.*;
import java.util.Date;
import java.text.SimpleDateFormat;
import java.util.StringTokenizer;
import java.util.Properties;
import java.net.InetAddress;
import javax.servlet.*;
import javax.servlet.http.*;

import distinct.rpc.*;

public class RPCGWServlet extends HttpServlet {
	RandomAccessFile logfd = null;
	boolean secure = true;
	File configFile;
	long configFileDate;
	SimpleDateFormat formatter;
	ACLEntry ACL = null;
	String br = "\n";

	class XDRraw implements XDRType{

		public byte[] value;

		public void xdr_encode(XDRStream xdrs)
		{
			xdrs.put_bytes(value, value.length);
			return;
		};

		public void xdr_decode(XDRStream xdrs) throws RPCError
		{
			value = xdrs.get_bytes(xdrs.get_length());
			return;
		};
	};

	class ACLEntry {
		String host;
		int prog;
		int vers;
		String prot;
		int proc;

		ACLEntry next;

		ACLEntry() {
			host = prot = "";
			prog = vers = proc = -1;
		}
	}

	public void init(ServletConfig config)
	throws ServletException
	{
		br = System.getProperty("line.separator", "\n");

		try {
			String logFileName = config.getInitParameter("LogFileName");
			if (logFileName != null) {
				logfd = new RandomAccessFile(logFileName, "rw");
				logfd.seek(logfd.length());
			}

			String dateFormat = config.getInitParameter("DateFormat");
			if (dateFormat != null)
					formatter = new SimpleDateFormat(dateFormat);
			else
					formatter = new SimpleDateFormat("EEE, MMM d, yyyy HH:mm:ss");

			String security = config.getInitParameter("Security");
			if (security != null)
				if (security.equals("false"))
					secure = false;
			
			if (secure) {
				String configFileName = config.getInitParameter("ConfigFileName");
				if (configFileName == null)
					configFileName = "rpcgw.cfg";
				configFile = new File(configFileName);
				configFileDate = configFile.lastModified();
				readConfig(configFile);
			}
		}
		catch (IOException e) {
			throw new ServletException(e.getMessage());
		}
	}

	synchronized void readConfig(File configFile) throws IOException {
		FileReader fr;

		ACL = null;
		fr = new FileReader(configFile);
		BufferedReader r = new BufferedReader(fr);

		while (true) {
			StringTokenizer s;

			try {
				String str= r.readLine();
				if (str == null)
					break;
				s = new StringTokenizer(str, " \t");
			}
			catch (IOException e) {
				break;
			}

			if (s.countTokens() != 5)
				continue;

			String token = s.nextToken();
			if (token.charAt(0) == '#')
				continue;

			ACLEntry e = new ACLEntry();
			
			e.host = token;

			token = s.nextToken();
			if (token.equals("*"))
				e.prog = -1;
			else
				e.prog = Integer.parseInt(token, 10);

			token = s.nextToken();
			if (token.equals("*"))
				e.vers = -1;
			else
				e.vers = Integer.parseInt(token, 10);

			e.prot = s.nextToken();

			token = s.nextToken();
			if (token.equals("*"))
				e.proc = -1;
			else
				e.proc = Integer.parseInt(token, 10);

			e.next = ACL;
			ACL = e;
		}
		fr.close();
	}

	synchronized boolean checkACL(String host, int prog, int vers, boolean useTCP, int proc) {
		for (ACLEntry acl = ACL; acl != null; acl = acl.next) {
			if ((acl.prog == -1) || (acl.prog == prog))
				if ((acl.vers == -1) || (acl.vers == vers))
					if ((acl.proc == -1) || (acl.proc == proc))
						if (acl.host.equals("*") || acl.host.equals(host))
							if (acl.prot.equals("*") || acl.prot.equals(useTCP?"tcp":"udp"))
								return true;
		}
		return false;
	}

	public void doPost(HttpServletRequest req, HttpServletResponse res)
	throws ServletException, IOException {
		try {
			if (logfd != null) {

				logfd.writeBytes("["+formatter.format(new Date())+"] ");

				String host = req.getRemoteHost();
				if (host != null)
					logfd.writeBytes(host+" ");
				else {
					String addr = req.getRemoteAddr();
					if (addr != null)
						logfd.writeBytes(addr+" ");
				}
			}

			BufferedReader instr = new BufferedReader(
				new InputStreamReader(req.getInputStream()));
			XDRStream xdrsin = new XDRStream();
			int len = req.getContentLength();

			while (len > 0) {
				len--;
				char c = (char)instr.read();
				switch (c) {
				  case '+':
					xdrsin.put_byte((byte)' ');
					break;
				  case '%':
					try {
						StringBuffer s = new StringBuffer();
						s.append((char)instr.read());
						s.append((char)instr.read());
						xdrsin.put_byte((byte) Integer.parseInt(s.toString(), 16));
						len -= 2;
					} catch (NumberFormatException e) {
						throw new IllegalArgumentException();
					}
					break;
				  default:
					xdrsin.put_byte((byte)c);
					break;
				}
			}

			byte id[] = xdrsin.get_bytes(4);
			String RPCHost = xdrsin.xdr_decode_string();		
			int prog = xdrsin.xdr_decode_int();		
			int vers = xdrsin.xdr_decode_int();		
			int proc = xdrsin.xdr_decode_int();		
			boolean useTCP = xdrsin.xdr_decode_boolean();		
			int Timeout = xdrsin.xdr_decode_int();		
			XDRraw in = new XDRraw();
			in.xdr_decode(xdrsin);

			if (logfd != null)
				logfd.writeBytes("request "+RPCHost+","+prog+","+vers+","+(useTCP?"tcp":"udp")+","+proc);

			if (secure) {
				if (configFile.lastModified() > configFileDate) {
					configFileDate = configFile.lastModified();
					readConfig(configFile);
				}

				if (!checkACL(RPCHost, prog, vers, useTCP, proc))
					throw new RPCError("Security violation", 0);
			}

			JRPCClient cl = new JRPCClient(InetAddress.getByName(RPCHost), prog, vers, useTCP); 
			cl.SetTimeout(Timeout);
			XDRraw out = new XDRraw();
			cl.Call(proc, in, out);
			cl.CloseClient();

			res.setContentType("application/octet-stream");
			ServletOutputStream outstr = res.getOutputStream();
			outstr.write(id);
			outstr.write(out.value);

			if (logfd != null)
				logfd.writeBytes(" successful"+br);
		}
		catch (RPCError e) {
			if (logfd != null)
				logfd.writeBytes(" Error : "+e.getMessage()+br);

			res.setContentType("text/html");
			ServletOutputStream outstr = res.getOutputStream();
			outstr.print(e.getMessage());

//			throw new ServletException(e.getMessage());
		}
	}

	public String getServletInfo() {
		return "RPCGWServlet translates and executes RPCs that are encapsulated in HTTP-requests.";
	}
}
