/* FILE: targetAcqMain.cpp
   AUTHOR: Michael Wagner
   CREATED: Apr 19, 1999
   DESCRIPTION: This is code that uses a targetAcq object. It serves as a link
     between this object and NDDS.
*/

#include <iostream.h>
#include <math.h>
#include "ndds/NDDS.h"
#include "targetAcq.h"
#include "targetAcqDef.h"
#include "acqDriverDef.h"
#include "targetAcqMsgs.h"
#include "dbRecord.h"
#include "dbMsgs.h"
#include "dbDef.h"
#include "database.h"
#include "SAS_ServiceNames.h"
#include "targetAcqMain.h"
#include "posDerivedState.h"
#include "nddsMsgNames.h"
#include "SAS_Config.h"
#include "telemetryMsgs.h"

#ifndef PI
#define PI 3.14159265359
#endif

// Includes for target acquisition manager drivers. 
// *** Add new includes when you want to add a new target acquisition manager driver.
#include "acqDriver.h"
#include "acqHiResDriver.h"

// Service routines
// These return RTI_TRUE when a reply should be sent,
// RTI_FALSE if no reply should be sent.
RTIBool targetAcqServiceRtn(NDDSObjectInstance reply,
			    NDDSObjectInstance request,
			    void *userData) 
{
  targetAcqRequest *serverRequest = (targetAcqRequest *)request;
  targetAcqReply   *serverReply   = (targetAcqReply *)reply;
  targetAcqServiceParam *param    = (targetAcqServiceParam *)userData;
  targetAcq *acqManager           = param->acq;
  acqDriver *driver               = acqManager->getDriver(serverRequest->driverNum);

  // Respond to requests

  // Check to make sure the driver number is ok.
  if(driver == NULL) {
    // This driver number is not valid!
    serverReply->status = TARGET_ACQ_NO_SUCH_DRIVER;
  } else {
    // The driver is valid...set its function element in the driverFunction array.
    serverReply->status = TARGET_ACQ_SUBMITTED_REQ;
    if(serverRequest->function == TARGET_ACQ_START) {
      cerr << "[targetAcqMain] *** Starting up driver " << serverRequest->driverNum << endl;
    } else if(serverRequest->function == TARGET_ACQ_STOP) {
      cerr << "[targetAcqMain] *** Stopping driver " << serverRequest->driverNum << endl;
    }
    param->driverFunction[serverRequest->driverNum] = (TARGET_ACQ_FUNCTION)serverRequest->function;
  }
  return(RTI_TRUE);
}

// NDDS server and client setup messages.
void dbClientSetup(NDDSClient *client) {
  *client = NddsClientCreate(DB_SERVICE,
			     dbReplyPublicationType,
			     dbRequestPublicationType);
}

void targetAcqServerSetup(targetAcq *acqManager, TARGET_ACQ_FUNCTION *driverFunctions) {
  targetAcqRequest *targetAcqRequestMsg = (targetAcqRequest *)calloc(1, sizeof(targetAcqRequest));
  targetAcqReply   *targetAcqReplyMsg = (targetAcqReply *)calloc(1, sizeof(targetAcqReply));
  targetAcqServiceParam *param = (targetAcqServiceParam *)calloc(1, sizeof(targetAcqServiceParam));

  param->acq = acqManager;
  param->driverFunction = driverFunctions;

  NddsServerCreate(TARGET_ACQ_SERVICE, NDDS_SERVER_IMMEDIATE,
		   TARGET_ACQ_SERVER_STRENGTH,
		   targetAcqReplyMsg, targetAcqRequestMsg,
		   targetAcqReplyPublicationType,
		   targetAcqRequestPublicationType,
		   targetAcqServiceRtn, param);
}


// NDDS publish/subscribe messages.
RTIBool sensorRegCallback(NDDSRecvInfo *issue) {
  sensorReg *item = (sensorReg *)issue->instance;
  targetAcq *acqManager = (targetAcq *)issue->callBackRtnParam;

  if (issue->status == NDDS_FRESH_DATA) {
    int sensorNum;
    if(acqManager->addSensor(item->sensorID, (SENSOR_STATUS)item->status, &sensorNum) == TARGET_ACQ_SUCCESSFUL) {
      cerr << "[targetAcqMain] Registered sensor \"" << item->sensorID << "\"" << endl;
    }
  }

  return(RTI_TRUE);
}

RTIBool telemetryCallback(NDDSRecvInfo *issue) {
  posDerivedState *item = (posDerivedState *)issue->instance;
  telemetryParam *param = (telemetryParam *)issue->callBackRtnParam;

  if (issue->status == NDDS_FRESH_DATA) {
    param->currPosition->x = item->posWest / -100.0;
    param->currPosition->y = item->posNorth / 100.0;
    param->currPosition->z = item->posUp / 100.0;
    param->currPose->pitch = item->rotX / 100.0;
    param->currPose->roll  = item->rotY / 100.0;
    param->currPose->yaw   = 2 * PI - item->rotZ / 100.0;
  }

  return(RTI_TRUE);
}

void publishComplete(NDDSPublication pub, targetAcqComplete *msg, TARGET_ACQ_STATUS status, int *newTargetIDs, int numTargets) {
  msg->status = status;
  memcpy(&msg->newTargetIDs, newTargetIDs, MAX_TARGETS_IN_ACQ_DATA * sizeof(int));
  msg->numTargets = numTargets;
  NddsPublicationSend(pub);
}

int main(int argc, char *argv[]) {
#ifdef THRESHOLD_TEST
  if(argc != 2) {
    cerr << "Usage: targetAcqMain image.ppm\n";
    return(-1);
  }
#ifdef RED_TEST
  acqResults *redresults = new acqResults();
  cerr << "[targetAcqMain] Performing red thresholding test...results will be in redThresholdResults.ppm" << endl;
  ppmRockSeg *image = new ppmRockSeg();
  cerr << MAX_TARGETS_IN_ACQ_DATA << endl;
  image->read(argv[1]);
  image->redThreshhold(NULL, NULL, 0, 480, redresults);
  image->write("redThresholdResults.ppm");
  cerr << "found " << redresults->numTargets << " targets" << endl;
  delete(redresults);
  return(0);
#else
  acqResults *kimresults = new acqResults();
  cerr << "[targetAcqMain] Performing kim thresholding test...results will be in kimThresholdResults.ppm" << endl;
  ppmRockSeg *image = new ppmRockSeg();
  cerr << MAX_TARGETS_IN_ACQ_DATA << endl;
  image->read(argv[1]);
  image->kimThreshhold(NULL, NULL, 0, 480, kimresults);
  image->write("kimThresholdResults.ppm");
  cerr << "found " << kimresults->numTargets << " targets" << endl;
  delete(kimresults);
  return(0);
#endif
#endif

  int nddsDomain;
  if(!getSAS_Config(NDDS_DOMAIN, nddsDomain)) {
    cerr << "[targetAcqMain] ERROR: Cannot read " << NDDS_DOMAIN << " from config file!" << endl;
    return(-1);
  }

  NddsInit(nddsDomain, NULL);
  cerr << "[targetAcqMain] Initialized NDDS" << endl;

  // Sensor registration stuff
  NDDSSubscription sensorRegSub;
  sensorReg *sensorRegMsg = NULL;

  // Telemetry stuff
  NDDSSubscription posDerivedStateSub;
  posDerivedState *posDerivedStateMsg = NULL;
  DGPS *currPosition = new DGPS();
  pose *currPose = new pose();
  telemetryParam *tParam = new telemetryParam();
  tParam->currPosition = currPosition;
  tParam->currPose = currPose;

  // Database stuff
  NDDSClient            dbClient;
  NDDSClientReplyStatus dbClientStatus;
  dbRequest             *dbRequestMsg = (dbRequest *)calloc(1, sizeof(dbRequest));
  dbReply               *dbReplyMsg   = (dbReply *)calloc(1, sizeof(dbReply));

  // NDDS client/server messages to/from the Target Acquisition Manager, and publications that will be
  // made when a Target Acquisition Manager request is completed, along with their publications.
  targetAcqRequest  *targetAcqRequestMsg  = (targetAcqRequest *)calloc(1, sizeof(targetAcqRequest));
  targetAcqComplete *targetAcqCompleteMsg = (targetAcqComplete *)calloc(1, sizeof(targetAcqComplete));
  NDDSPublication    targetAcqPublication;

  // Other objects
  database *db = new database();
  targetAcq *acqManager = new targetAcq();
  float acqPeriod = TARGET_ACQ_DEFAULT_PERIOD;

  // Register all NDDS types
  sensorRegNddsRegister();
  targetAcqReplyNddsRegister();
  targetAcqRequestNddsRegister();
  targetAcqCompleteNddsRegister();
  dbReplyNddsRegister();
  dbRequestNddsRegister();
  posDerivedStateNddsRegister();

  // First we have to initialize all target acquisition manager drivers here.
  // *** This is the part of the code you'll need to modify to add new drivers ***

  // *** numDesiredDrivers should be equal to the number of drivers you want in the system.
  const int numDesiredDrivers = 1;

  int numDrivers;

  // *** Use initDriver to add new drivers here.
  acqManager->initDriver((acqDriver *)(new acqHiResDriver(currPosition, currPose)), &numDrivers);

  if(acqManager->getNumDrivers() != numDesiredDrivers) {
    cerr << "[targetAcqMain] ERROR: Failed to register target acquisition manager drivers!" << endl;
    return(-1);
  }

  // Next set up the target acquisition manager subscriber for registration of sensors.
  // When a sensor driver registers itself, it will set up the appropriate
  // sensorClient in the array.
  cerr << "[targetAcqMain] Setting up sensor registration subscriber...";
  sensorRegMsg = (sensorReg *)calloc(1, sizeof(sensorReg));
  sensorRegSub = NddsSubscriptionCreate(NDDS_SUBSCRIPTION_IMMEDIATE,
					SENSOR_REG_MSG_NAME, sensorRegPublicationType,
					sensorRegMsg, SENSOR_REG_DEADLINE,
					SENSOR_REG_MIN_SEPARATION, sensorRegCallback, acqManager);
  cerr << "done! Awaiting sensor registration messages." << endl;

  // Now we start reading in telemetry
  cerr << "[targetAcqMain] Setting up telemetry subscriber...";
  posDerivedStateMsg = (posDerivedState *)calloc(1, sizeof(posDerivedState));
  posDerivedStateSub = NddsSubscriptionCreate(NDDS_SUBSCRIPTION_IMMEDIATE,
					      POS_DERIVED_STATE_MSG_NAME, posDerivedStatePublicationType,
					      posDerivedStateMsg, TELEMETRY_DEADLINE,
					      TELEMETRY_MIN_SEPARATION, telemetryCallback, tParam);
  cerr << "done! Awaiting sensor registration messages." << endl;

  // Before the mission planner can command the target acquisition manager, a database client
  // must be set up.
  cerr << "[targetAcqMain] Setting up the database client...";
  dbClientSetup(&dbClient);
  if(NddsClientServerWait(dbClient, 2.0, 5, 1) == RTI_FALSE) { 
    cerr << "[targetAcqMain] ERROR: \"" << DB_SERVICE << "\" server missing!" << endl;
    return(-1);
  } 
  cerr << "done!" << endl;

  // Now we can set up the target acquisition manager server that accepts command request from the
  // mission planner.
  cerr << "[targetAcqMain] Setting up Target Acquisition Manager server...";
  TARGET_ACQ_FUNCTION acqFunction[TARGET_ACQ_MAX_DRIVERS];
  for(int i=0; i < TARGET_ACQ_MAX_DRIVERS; i++) {
    acqFunction[i] = TARGET_ACQ_STOP;
  }
  targetAcqServerSetup(acqManager, acqFunction);
  cerr << "done!" << endl;

  // Now we set up the publishers, which will create messages when target(s) are found
  cerr << "[targetAcqMain] Setting up publishers...";
  targetAcqPublication = NddsPublicationCreate(TARGET_ACQ_COMPLETE_MSG_NAME, targetAcqCompletePublicationType,
					       targetAcqCompleteMsg, TARGET_ACQ_COMPLETE_PERSISTANCE,
					       TARGET_ACQ_COMPLETE_STRENGTH);
  cerr << "done!" << endl;  

  // Set stuff up for acquiring targets
  int newTargetIDs[MAX_TARGETS_IN_ACQ_DATA];
  acqDriver *currentDriver;
  acqResults results;
  int examineReturnVal;
  unsigned int startExamineTime, endExamineTime; /* These are used to time target acq
						    (stores posDerivedStateMsg->msecs) */

  while(1) {
    // Now perform any acquisition attempts 
    for(int i=0; i < TARGET_ACQ_MAX_DRIVERS; i++) {
      if(acqManager->isValidDriver(i)) {
	switch(acqFunction[i]) {
	case TARGET_ACQ_START:
	  // The driver has just started being used, therefore set it up first.
	  currentDriver = acqManager->getDriver(i);
	  if(currentDriver->setupAcq() != 0) {
	    cerr << "[targetAcqMain] ERROR: Failed setupAcq for target acquisition driver \"" << currentDriver->driverID << "\"" << endl;
	  }
	  
	  // Now that it's set up, start it into regular running mode
	  acqFunction[i] = TARGET_ACQ_RUNNING;

	case TARGET_ACQ_RUNNING:
	  cerr << "[targetAcqMain] Acquiring targets with driver #" << i << "..." << endl;

	  currentDriver = acqManager->getDriver(i);
	  startExamineTime = posDerivedStateMsg->msecs;
	  examineReturnVal = currentDriver->examine(&results);
	  endExamineTime = posDerivedStateMsg->msecs;

	  cerr << "[targetAcqMain] Done! Target acq took " << ((endExamineTime - startExamineTime) / 1000.0) 
	       << " seconds to find " << results.numTargets << " targets." << endl;

	  if((examineReturnVal == TARGET_ACQ_SUCCESSFUL) && (results.numTargets > 0)) {

	    // Insert new targets into database
	    cerr << "[targetAcqMain] Inserting " << results.numTargets << " targets into database." << endl;

	    dbRequestMsg->function = DB_INSERT_MULTI_FUNCTION;
	    dbRequestMsg->numIDs = results.numTargets;
	    dbRequestMsg->driverNum = targetAcqRequestMsg->driverNum;
	    dbClientStatus = NddsClientCallAndWait(dbClient,
						   dbReplyMsg, dbRequestMsg,
						   DB_MIN_WAIT_TIME,
						   DB_MAX_WAIT_TIME);
	    if(dbClientStatus == NDDS_CLIENT_RECV_REPLY) {
	      if(dbReplyMsg->status != DB_OK) {
		// Database returned fail condition. Set status and return.
		// TBD: Do a real diagnosis?
		cerr << "[targetAcqMain] ERROR: Database insert multi record returned " << dbReplyMsg->status << endl;
		publishComplete(targetAcqPublication, targetAcqCompleteMsg, TARGET_ACQ_FAILED, NULL, 0);
	      }
	    } else {
	      // Database did not respond. 
	      cerr << "[targetAcqMain] ERROR: Database did not respond to insert record request" << endl;
	      publishComplete(targetAcqPublication, targetAcqCompleteMsg, TARGET_ACQ_FAILED, NULL, 0);
	    }
	    
	    for(int i=0; i < results.numTargets; i++) {
	      newTargetIDs[i] = dbReplyMsg->IDs[0] + i;
	      NddsUtilitySleep(0.01);
	      
	      // Now put GPS info
	      dbRequestMsg->function = DB_PUT_FUNCTION;
	      dbRequestMsg->driverNum = targetAcqRequestMsg->driverNum;
	      dbRequestMsg->record.targetID = dbReplyMsg->IDs[0] + i;
	      dbRequestMsg->record.DGPS_coord.x = results.targetPos[i].x;
	      dbRequestMsg->record.DGPS_coord.y = results.targetPos[i].y;
	      dbRequestMsg->record.DGPS_coord.z = results.targetPos[i].z;
	      dbRequestMsg->recordBitMask.DGPS_coord = DB_ALTERED;

	      dbClientStatus = NddsClientCallAndWait(dbClient,
						     dbReplyMsg, dbRequestMsg,
						     DB_MIN_WAIT_TIME,
						     DB_MAX_WAIT_TIME);
	      if(dbClientStatus == NDDS_CLIENT_RECV_REPLY) {
		if(dbReplyMsg->status != DB_OK) {
		  // Database returned fail condition. Set status and return.
		  // TBD: Do a real diagnosis?
		  cerr << "[targetAcqMain] ERROR: Database put data returned " << dbReplyMsg->status << endl;
		  publishComplete(targetAcqPublication, targetAcqCompleteMsg, TARGET_ACQ_FAILED, NULL, 0);
		  continue;
		}
	      } else {
		// Database did not respond. 
		cerr << "[targetAcqMain] ERROR: Database did not respond to put data request" << endl;
		publishComplete(targetAcqPublication, targetAcqCompleteMsg, TARGET_ACQ_FAILED, NULL, 0);
		continue;
	      }
	      
	      NddsUtilitySleep(0.01);
	    }
	    
	    // Success!
	    publishComplete(targetAcqPublication, targetAcqCompleteMsg, TARGET_ACQ_SUCCESSFUL, 
			    newTargetIDs, results.numTargets);
	  }

	  break;
	  
	}
      }
    }
 
    // TBD: Implement a real protocol to keep the period
    //    NddsUtilitySleep(acqPeriod);
  }
  
  cerr << "[targetAcqMain] ERROR: main() should not return!" << endl;
  return(-1);

}
