/* FILE: ppmRockSeg.cpp
   CREATED: May 24, 1999
   AUTHOR: Michael Wagner
   DESCRIPTION: This file implements a rock on ice segmentation algorithm
     that uses Fisher's linear discriminant on PPM images. 
*/

#include <iostream.h>
#include <math.h>
#include "ppmRockSeg.h"
#include "gps_calc.h"
#include "hiResSensorDef.h"
#include "lensController.h"
#include "targetAcqDef.h"
#include "SAS_Config.h"

#ifndef MIN
#define MIN(_a_, _b_) ((_a_ < _b_) ? _a_ : _b_)
#endif

// **** ppmRockSeg class methods ****

ppmRockSeg::ppmRockSeg() {
  this->topBoxSize = DEFAULT_TOP_BOX_SIZE;
  this->bottomBoxSize = DEFAULT_BOTTOM_BOX_SIZE;
  this->topPixSize = DEFAULT_TOP_PIX_SIZE;
  this->bottomPixSize = DEFAULT_BOTTOM_PIX_SIZE;
  this->laser = NULL;

  lastWeightVector = new imageWeightVector();
  lastWeightVector->w[0] = -0.0001;
  lastWeightVector->w[1] = -0.0011;
  lastWeightVector->w[2] =  0.0018;
  lastWeightVector->w[3] =  0.0001;
  lastWeightVector->w[4] = -0.0005;
  lastWeightVector->w[5] = -0.0005;
  lastWeightVector->w[6] =  0.0001;
  lastWeightVector->w[7] =  0.0003;
  lastWeightVector->w[8] = -0.0007;
  lastWeightVector->threshhold = 9.7005e-04;
}

ppmRockSeg::ppmRockSeg(int topBoxSize, int bottomBoxSize, int topPixSize, int bottomPixSize, laserDerivedState *laser) {
  this->topBoxSize = topBoxSize;
  this->bottomBoxSize = bottomBoxSize;
  this->topPixSize = topPixSize;
  this->bottomPixSize = bottomPixSize;
  this->laser = laser;

  lastWeightVector = new imageWeightVector();
  lastWeightVector->w[0] = -0.0001;
  lastWeightVector->w[1] = -0.0011;
  lastWeightVector->w[2] =  0.0018;
  lastWeightVector->w[3] =  0.0001;
  lastWeightVector->w[4] = -0.0005;
  lastWeightVector->w[5] = -0.0005;
  lastWeightVector->w[6] =  0.0001;
  lastWeightVector->w[7] =  0.0003;
  lastWeightVector->w[8] = -0.0007;
  lastWeightVector->threshhold = 9.7005e-04;
}

ppmRockSeg::ppmRockSeg(laserDerivedState *laser) {
  this->topBoxSize = DEFAULT_TOP_BOX_SIZE;
  this->bottomBoxSize = DEFAULT_BOTTOM_BOX_SIZE;
  this->topPixSize = DEFAULT_TOP_PIX_SIZE;
  this->bottomPixSize = DEFAULT_BOTTOM_PIX_SIZE;
  this->laser = laser;

  lastWeightVector = new imageWeightVector();
  lastWeightVector->w[0] = -0.0001;
  lastWeightVector->w[1] = -0.0011;
  lastWeightVector->w[2] =  0.0018;
  lastWeightVector->w[3] =  0.0001;
  lastWeightVector->w[4] = -0.0005;
  lastWeightVector->w[5] = -0.0005;
  lastWeightVector->w[6] =  0.0001;
  lastWeightVector->w[7] =  0.0003;
  lastWeightVector->w[8] = -0.0007;
  lastWeightVector->threshhold = 9.7005e-04;
}

ppmRockSeg::fldThreshhold(DGPS *robotPosition, pose *robotPose, int minRow, int maxRow, acqResults *results) {
  if(strcmp(format, "P6") != 0) {
    cerr << "[ppmRockSeg] Cannot perform threshhold on this type of image!\n";
    return(0);
  }

  if(results == NULL) {
    cerr << "[ppmRockSeg] ERROR: results == NULL\n";
    return(0);
  }

  float feature[9];  // R, G, B, R/G, R/B, G/R, G/B, B/R, B/G
  float threshhold;
  
  int currPixel;
  float sum;
  for(int i=minRow; i < maxRow; i++) {
    for(int j=MIN_COL; j < MAX_COL; j++) {
      currPixel = i * numCols + j;
      feature[R] = red[currPixel];
      feature[G] = green[currPixel];
      feature[B] = blue[currPixel];
      feature[R_G] = feature[R] / feature[G];
      feature[R_B] = feature[R] / feature[B];
      feature[G_R] = feature[G] / feature[R];
      feature[G_B] = feature[G] / feature[B];
      feature[B_R] = feature[B] / feature[R];
      feature[B_G] = feature[B] / feature[G];
      
      sum = 0.0;
      for(int k=0; k < 9; k++) {
	sum += lastWeightVector->w[k] * feature[k];
      }

      if(sum > lastWeightVector->threshhold) {
	red[currPixel] = green[currPixel] = blue[currPixel] = 0;
      } else {
	red[currPixel] = green[currPixel] = blue[currPixel] = 255;
      }
    }
  }

  // Now compute a new value for lastWeightVector with FLD

  // TBD: actually get the number and location of targets.
  results->numTargets = -1;
  return(1);
}




































void ppmRockSeg::addToLists(int row, int col, target_type* target) {
  list *templist;

  if (target->leafs == NULL) {
    target->leafs = (list *)malloc(sizeof(list));
    templist = target->leafs; 
  } else {
    templist = target->leafs;
    while (templist->next != NULL) templist = templist->next;
    templist->next = (list *)malloc(sizeof(list));
    templist = templist->next; 
  }
  templist->data.row = row;
  templist->data.column = col;
  templist->next = NULL;
}

float ppmRockSeg::sqr(int a) {
  return((float)a * (float)a);
}


int ppmRockSeg::kimThreshhold(DGPS *robotPosition, pose *robotPose, int minRow, int maxRow, acqResults *results) {
  if(strcmp(format, "P6") != 0) {
    cerr << "[ppmRockSeg] Cannot perform threshhold on this type of image!\n";
    return(0);
  }

  if(results == NULL) {
    cerr << "[ppmRockSeg] ERROR: results == NULL\n";
    return(0);
  }

  int currPixel;

  // *** First find totals and averages ***

  int totalIntensity = 0;
  int bgRatioTmp;
  int totalRatio = 0;
  int maxRatio = 0;

  int *intensity = (int *)calloc(numRows*numCols, sizeof(int));
  int *bgRatio = (int *)calloc(numRows*numCols, sizeof(int));
  
  for(int i=minRow; i < maxRow; i++) {
    for(int j=MIN_COL; j < MAX_COL; j++) {
      currPixel = i * numCols + j;
      if(green[currPixel] == 0) {
	bgRatioTmp = pixelDepth;
      } else {
	bgRatioTmp = (int)((float)blue[currPixel] / (float)green[currPixel] * 100.0);
      }

      intensity[currPixel] = red[currPixel] + green[currPixel] + blue[currPixel];
      bgRatio[currPixel] = blue[currPixel] + bgRatioTmp;

      if(bgRatio[currPixel] > maxRatio) {
	maxRatio = bgRatio[currPixel];
      }

      totalRatio += bgRatio[currPixel];
      totalIntensity += intensity[currPixel];
    }
  }

  float averageRatio = totalRatio / ((maxRow-minRow) * (MAX_COL - MIN_COL));
  float averageIntensity = totalIntensity / ((maxRow-minRow) * (MAX_COL - MIN_COL));

  cerr << endl;
  cerr << "[ppmRockSeg] Average ratio is " << averageRatio << ", max is " << maxRatio << endl;
  cerr << "[ppmRockSeg] Average intensity is " << averageIntensity << endl;

  // ** Now do the threshholding ***

  // Find stddev of B/G ratio
  float sum = 0.0;
  for(int i=minRow; i < maxRow; i++) {
    for(int j=MIN_COL; j < MAX_COL; j++) {
      currPixel = i * numCols + j;
      sum += (bgRatio[currPixel] - averageRatio) * (bgRatio[currPixel] - averageRatio);
    }
  }
  int y = (int)sqrt(sum);
  int stdRatio = (int)(y / sqrt((maxRow-minRow) * (MAX_COL-MIN_COL) - 1));
  cerr << "[ppmRockSeg] Standard deviation is " << stdRatio << endl;
  
  if (stdRatio > MAX_STDDEV_RATIO) {
    maxRatio = 0;
    totalRatio = 0;
    /* compensate for shadows */
    for(int i=minRow; i < maxRow; i++) {
      for(int j=MIN_COL; j < MAX_COL; j++) {
	/* check if intensity is less than average - alter processed value */
	currPixel = i * numCols + j;
	if (intensity[currPixel] < averageIntensity) {
	  bgRatio[currPixel] = (int)(CBLUE/CGREEN * bgRatio[currPixel] + (CBLUE - CBLUE/CGREEN) * blue[currPixel]);
	}
	/* check if this pixel has maximum color */
	if (bgRatio[currPixel] > maxRatio) {
	  maxRatio = bgRatio[currPixel];
	}
	totalRatio += bgRatio[currPixel];
      }
    }

    averageRatio = totalRatio / ((maxRow-minRow) * (MAX_COL - MIN_COL));
    cerr << "[ppmRockSeg] New average ratio is " << averageRatio << endl;
  }

  int cutoff = (int)(PERCENTAGE * averageRatio);
  
  for(int i=minRow; i < maxRow; i++) {
    for(int j=MIN_COL; j < MAX_COL; j++) {
      currPixel = i * numCols + j;
      if (bgRatio[currPixel] > cutoff) {
	intensity[currPixel] = 255;
      } else {
	intensity[currPixel] = 0;
      }
    }
  }

  /* now process to find connected objects and pixel locations */
  int furthestCol;
  int row, col, mark;
  list *templist;
  int boxsize, minsize;
  float dist, slope;
  int end;

  int lasttarget = 0;
  int done = 0;

  for (int i=minRow; (i < maxRow) && !done; i++) {
    for (int j=MIN_COL; (j < MAX_COL) && !done; j++) {
      currPixel = i * numCols + j;
      if (intensity[currPixel] == 0) {  
	/* we've found a rock pixel */

	/* define a target number for this object */
	results->targets[lasttarget].targetnum = lasttarget;

	/* initialize total number of pixels, maximum distance, furthest
	   and first pixels, and prime the list of pixels in this target,
	   to get the proper initial conditions for the while loop below */
	results->targets[lasttarget].totalpix = 0;
	results->targets[lasttarget].maxdist = 0.0;
	results->targets[lasttarget].leafs = NULL;
	results->targets[lasttarget].lastPixel.row = i;
	results->targets[lasttarget].lastPixel.column = j;
	results->targets[lasttarget].firstPixel.row = i;
	results->targets[lasttarget].firstPixel.column = j;
	furthestCol = j;
	addToLists(i, j, &results->targets[lasttarget]);

	/* turn pixel white so it doesn't get counted twice */
	intensity[currPixel] = 255;
	
	/* now find all connected pixels using a breadth first search
	   method */
	while (results->targets[lasttarget].leafs != NULL) {
	  row = results->targets[lasttarget].leafs->data.row;
	  col = results->targets[lasttarget].leafs->data.column;

	  /* check furthest distance from start */
	  dist = (float)sqrt(sqr(results->targets[lasttarget].firstPixel.row - row) +
			     sqr(results->targets[lasttarget].firstPixel.column - col));
	  if (dist > results->targets[lasttarget].maxdist) {
	    results->targets[lasttarget].maxdist = dist;
	    results->targets[lasttarget].lastPixel.row = row;
	    results->targets[lasttarget].lastPixel.column = col; 
	  }

	  /* keep track of furthest column in this row, so we won't try to segment
	     pixels already segmented when we continue the search through the image */
	  if (row == results->targets[lasttarget].firstPixel.row && col > furthestCol) {
	    furthestCol = col;
	  }

	  /* increment totalpix */
	  ++results->targets[lasttarget].totalpix;
	  
	  /* add adjacent pixels (if already checked, they will be white) */
	  end = 1; // If end stays 1, then there are no more adjacent pixels in this blob
	  if (row != 0) {
	    if (col != 0 && intensity[(row-1)*numCols+(col-1)] == 0) {
	      addToLists(row-1, col-1, &results->targets[lasttarget]);
	      intensity[(row-1)*numCols+(col-1)] = 255;
	      end = 0; 
	    }
	    if (intensity[(row-1)*numCols+(col)] == 0) {
	      addToLists(row-1, col, &results->targets[lasttarget]);
	      intensity[(row-1)*numCols+(col)] = 255;
	      end = 0;
	    }
	    if (col != numCols-1 && intensity == 0) {
	      addToLists(row-1, col+1, &results->targets[lasttarget]);
	      intensity[(row-1)*numCols+(col+1)] = 255;
	      end = 0;
	    } 
	  }
	  if (col != 0 && intensity[(row)*numCols+(col-1)] == 0) {
	    addToLists(row, col-1, &results->targets[lasttarget]);
	    intensity[(row)*numCols+(col-1)] = 255;
	    end = 0;
	  }
	  if (col != numCols-1 && intensity[(row)*numCols+(col+1)] == 0) {
	    addToLists(row, col+1, &results->targets[lasttarget]);
	    intensity[(row)*numCols+(col+1)] = 255;
	    end = 0;
	  }
	  if (row != numRows-1) {
	    if (col != 0 && intensity[(row+1)*numCols+(col-1)] == 0) {
	      addToLists(row+1, col-1, &results->targets[lasttarget]);
	      intensity[(row+1)*numCols+(col-1)] = 255;
	      end = 0;
	    }
	    if (intensity[(row+1)*numCols+(col)] == 0) {
	      addToLists(row+1, col, &results->targets[lasttarget]);
	      intensity[(row+1)*numCols+(col)] = 255;
	      end = 0;
	    }
	    if (col != numCols-1 && intensity[(row+1)*numCols+(col+1)] == 0) {
	      addToLists(row+1, col+1, &results->targets[lasttarget]);
	      intensity[(row+1)*numCols+(col+1)] = 255;
	      end = 0;
	    }
	  }
	  
	  /* If there are no more directly adjacent rock pixels in this blob, then
	     check for rock pixels NEARBY, but not directly connected */
	  if (end == 1) {
	    /* size of box depends on row (==> distance from camera):
	       the smaller row is, the smaller the box is */
	    boxsize = topBoxSize + (int)((bottomBoxSize - topBoxSize) *
					 (float)row / (float)numRows);
	    for (int i=-boxsize/2; i<boxsize/2; ++i) {
	      for (int j=-boxsize/2; j<boxsize/2; ++j) {
		if (col+j >= 0 && col+j < numCols &&
		    row+i >=0 && row+i < numRows) {
		  if (intensity[(row+1)*numCols+(col+j)] == 0) {
		    addToLists(row+i, col+j, &results->targets[lasttarget]);
		    intensity[(row+1)*numCols+(col+j)] = 255; 
		  } 
		}
	      }
	    }
	  }
	  
	  // Go to next item in list, and then remove the item we just examined 
	  templist = results->targets[lasttarget].leafs;
	  results->targets[lasttarget].leafs = templist->next;
	  free(templist);
	}
	
	/* Now we want to eliminate lines and artifacts */
	//printf("size %d pixels, ", results->targets[lasttarget].totalpix);
	//printf("size/length ratio = %f: ",
	//       results->targets[lasttarget].totalpix / results->targets[lasttarget].maxdist);
	
	/* approximate size based on distance (location in picture) 
	   rocks must be minimum number of pixels depending on distance 
	   at bottom of image, 8 pixels?, at top, 3 pixels? 
	   all targets must have more pixels that "minsize" to be kept */
	minsize = topPixSize + (int)((bottomPixSize - topPixSize) * (float)row / (float)numRows);

	if (results->targets[lasttarget].totalpix >= minsize) { // The target is big enough, but is it the right shape??

	  /* check for long skinny targets - if they are sufficiently large,
	     these are probably shadow lines */
	  if (results->targets[lasttarget].totalpix <= 2 * minsize || (float)results->targets[lasttarget].totalpix /
	                                                                    results->targets[lasttarget].maxdist > LINERATIO) {

	    /* find pixel halfway between results->targets[lasttarget].firstPixel and results->targets[lasttarget].lastPixel */
	    cerr << "target #" << lasttarget << " firstPixel.row = " << results->targets[lasttarget].firstPixel.row 
		 << " firstPixel.col = " << results->targets[lasttarget].firstPixel.column
		 << " lastPixel.row = " << results->targets[lasttarget].lastPixel.row
		 << " lastPixel.column = " << results->targets[lasttarget].lastPixel.column << endl;
	    
	    results->targets[lasttarget].midpt.row = 
	      (int)(abs(results->targets[lasttarget].lastPixel.row - results->targets[lasttarget].firstPixel.row) / 2) +
	           MIN(results->targets[lasttarget].firstPixel.row, results->targets[lasttarget].lastPixel.row);

	    results->targets[lasttarget].midpt.column = 
	      (int)(abs(results->targets[lasttarget].lastPixel.column - results->targets[lasttarget].firstPixel.column) / 2) +
	           MIN(results->targets[lasttarget].firstPixel.column, results->targets[lasttarget].lastPixel.column);

#ifdef ORIGINAL_KIM_CODE
	    dist = sqrt((float)sqr(results->targets[lasttarget].firstPixel.column - results->targets[lasttarget].lastPixel.column) +
			(float)sqr(results->targets[lasttarget].firstPixel.row - results->targets[lasttarget].lastPixel.row));
	    if (results->targets[lasttarget].lastPixel.row == results->targets[lasttarget].firstPixel.row) {
	      results->targets[lasttarget].midpt.row = results->targets[lasttarget].firstPixel.row;
	      if (results->targets[lasttarget].lastPixel.column > results->targets[lasttarget].firstPixel.column) {
		results->targets[lasttarget].midpt.column = results->targets[lasttarget].firstPixel.column +
		  (int)dist/2;
	      } else {
		results->targets[lasttarget].midpt.column = results->targets[lasttarget].firstPixel.column -
		  (int)dist/2;
	      }
	    } else {
	      slope = (float)(results->targets[lasttarget].lastPixel.column - results->targets[lasttarget].firstPixel.column) / 
		(float)(results->targets[lasttarget].lastPixel.row - results->targets[lasttarget].firstPixel.row);
	      if (results->targets[lasttarget].lastPixel.column == results->targets[lasttarget].firstPixel.column) {
		results->targets[lasttarget].midpt.column = results->targets[lasttarget].firstPixel.column;
		if (results->targets[lasttarget].lastPixel.row > results->targets[lasttarget].firstPixel.column) {
		  results->targets[lasttarget].midpt.row = results->targets[lasttarget].firstPixel.row +
		    (int)(dist/2.0);
		} else {
		  results->targets[lasttarget].midpt.row = results->targets[lasttarget].firstPixel.row -
		    (int)(dist/2.0);
		}
	      } else {
		if (results->targets[lasttarget].lastPixel.column > results->targets[lasttarget].firstPixel.column) {
		  results->targets[lasttarget].midpt.column = results->targets[lasttarget].firstPixel.column +
		    (int)(dist/(2.0*sqrt(1.0 + 1.0/(slope*slope))));
		} else {
		  results->targets[lasttarget].midpt.column = results->targets[lasttarget].firstPixel.column -
		    (int)(dist/(2.0*sqrt(1.0 + 1.0/(slope*slope))));
		}
		results->targets[lasttarget].midpt.row = results->targets[lasttarget].firstPixel.row +
		  (int)((results->targets[lasttarget].midpt.column - results->targets[lasttarget].firstPixel.column) / slope);
	      }
	    }
#endif

	    if(robotPosition != NULL && robotPose != NULL) {
	      cerr << "** Target is at row = " << results->targets[lasttarget].midpt.row << ", col = " 
		                               <<  results->targets[lasttarget].midpt.column << endl;

	      float distanceEst;
	      // Do initial guess with height = 0 assumption
	      get_gps(TARGET_ACQ_CAM_PAN, TARGET_ACQ_CAM_TILT + TARGET_ACQ_CAM_TILT_FUDGE, results->targets[lasttarget].midpt.row, 
		      results->targets[lasttarget].midpt.column, focalLength(TARGET_ACQ_CAM_ZOOM), 
		      robotPose->pitch, robotPose->roll, robotPose->yaw, 0.0,
		      &results->targetPos[lasttarget].x, &results->targetPos[lasttarget].y, 
		      &results->targetPos[lasttarget].z, &distanceEst);
	      
	      if(laser != NULL) {
		// Now add laser data into the mix to improve flat earth assumption
		int startRow, endRow;
		get_laser_start_end(results->targetPos[lasttarget].x, results->targetPos[lasttarget].y, TARGET_ACQ_WIDTH, 
				    laser, &startRow, &endRow);
		float realHeight = ground_height(laser, startRow, endRow);
		
		// Now realHeight contains the "approximate" height of the ground at the target.
		get_gps(TARGET_ACQ_CAM_PAN, TARGET_ACQ_CAM_TILT + TARGET_ACQ_CAM_TILT_FUDGE, results->targets[lasttarget].midpt.row, 
			results->targets[lasttarget].midpt.column, focalLength(TARGET_ACQ_CAM_ZOOM), 
			robotPose->pitch, robotPose->roll, robotPose->yaw, realHeight - robotPosition->z,
			&results->targetPos[lasttarget].x, &results->targetPos[lasttarget].y, 
			&results->targetPos[lasttarget].z, &distanceEst);
		
		
		results->targetPos[lasttarget].x += robotPosition->x;
		results->targetPos[lasttarget].y += robotPosition->y;
		results->targetPos[lasttarget].z += robotPosition->z; /* laser calculations ignore DGPS_SENSOR_Z,
									 so DGPS_SENSOR_Z doesn't need to be considered */
	      } else {
		results->targetPos[lasttarget].x += robotPosition->x;
		results->targetPos[lasttarget].y += robotPosition->y;
		results->targetPos[lasttarget].z += robotPosition->z - DGPS_SENSOR_Z;
	      }
	    }
	      
	    /* only if we're keeping this target... */
	    if(++lasttarget >= MAX_TARGETS_IN_ACQ_DATA) {
	      done = 1;
	    }
	  } else {
	    // Target is too similar to a line -- throw it out
	  }
	} else {
	  // Target is too small -- throw it out
	}
      }
    }
  }

  for(int i=minRow; i < maxRow; i++) {
    for(int j=minRow; j < maxRow; j++) {
      currPixel = i * numCols + j;
      mark = 0;
      for(int k=0; k < lasttarget; k++) {
	if(abs(results->targets[k].midpt.row - i) < 3 &&
	   abs(results->targets[k].midpt.column - j) < 3) {
	       mark = 1;
	}
      }
      if(mark) {
	red[currPixel] = 255;
	green[currPixel] = blue[currPixel] = 0;
      } else if(bgRatio[currPixel] > cutoff) {
	red[currPixel] = green[currPixel] = blue[currPixel] = 255;
      } else {
	red[currPixel] = green[currPixel] = blue[currPixel] = 0;
      }
    }
  }

  free(intensity);
  free(bgRatio);
  
  results->numTargets = lasttarget;

  return(1);
}



















































int ppmRockSeg::redThreshhold(DGPS *robotPosition, pose *robotPose, int minRow, int maxRow, acqResults *results) {
  if(strcmp(format, "P6") != 0) {
    cerr << "[ppmRockSeg] Cannot perform threshhold on this type of image!\n";
    return(0);
  }

  if(results == NULL) {
    cerr << "[ppmRockSeg] ERROR: results == NULL\n";
    return(0);
  }

  float redBlueCoeff, redGreenCoeff;
  if(!getSAS_Config(RED_THRESHOLD_RB_COEFF, redBlueCoeff)) {
    cerr << "[ppmRockSeg] ERROR: Cannot read " << RED_THRESHOLD_RB_COEFF << " from config file!";
    return(0);
  }
  if(!getSAS_Config(RED_THRESHOLD_RG_COEFF, redGreenCoeff)) {
    cerr << "[ppmRockSeg] ERROR: Cannot read " << RED_THRESHOLD_RG_COEFF << " from config file!";
    return(0);
  }

  int currPixel;

  // *** First find totals and averages ***

  int totalIntensity = 0;
  int bgRatioTmp;
  int totalRatio = 0;
  int maxRatio = 0;

  int *intensity = (int *)calloc(numRows*numCols, sizeof(int));
    
  for(int i=minRow; i < maxRow; i++) {
    for(int j=MIN_COL; j < MAX_COL; j++) {
      currPixel = i * numCols + j;
      if((red[currPixel] > redBlueCoeff * blue[currPixel]) && (red[currPixel] > redGreenCoeff * green[currPixel])) {
	intensity[currPixel] = 0;
      } else {
	intensity[currPixel] = 255;
      }
    } 
  }

  /* now process to find connected objects and pixel locations */
  int furthestCol;
  int row, col, mark;
  list *templist;
  int boxsize, minsize;
  float dist, slope;
  int end;

  int lasttarget = 0;
  int done = 0;

  for (int i=minRow; (i < maxRow) && !done; i++) {
    for (int j=MIN_COL; (j < MAX_COL) && !done; j++) {
      currPixel = i * numCols + j;
      if (intensity[currPixel] == 0) {  

	/* we've found a rock pixel */
	//printf("Processing target %d...\n", lasttarget);

	/* define a target number for this object */
	results->targets[lasttarget].targetnum = lasttarget;

	/* initialize total number of pixels, maximum distance, furthest
	   and first pixels, and prime the list of pixels in this target,
	   to get the proper initial conditions for the while loop below */
	results->targets[lasttarget].totalpix = 0;
	results->targets[lasttarget].maxdist = 0.0;
	results->targets[lasttarget].leafs = NULL;
	results->targets[lasttarget].lastPixel.row = i;
	results->targets[lasttarget].lastPixel.column = j;
	results->targets[lasttarget].firstPixel.row = i;
	results->targets[lasttarget].firstPixel.column = j;
	furthestCol = j;
	addToLists(i, j, &results->targets[lasttarget]);

	/* turn pixel white so it doesn't get counted twice */
	intensity[currPixel] = 255;
	
	/* now find all connected pixels using a breadth first search
	   method */
	while (results->targets[lasttarget].leafs != NULL) {
	  row = results->targets[lasttarget].leafs->data.row;
	  col = results->targets[lasttarget].leafs->data.column;

	  /* check furthest distance from start */
	  dist = (float)sqrt(sqr(results->targets[lasttarget].firstPixel.row - row) +
			     sqr(results->targets[lasttarget].firstPixel.column - col));
	  if (dist > results->targets[lasttarget].maxdist) {
	    results->targets[lasttarget].maxdist = dist;
	    results->targets[lasttarget].lastPixel.row = row;
	    results->targets[lasttarget].lastPixel.column = col; 
	  }

	  /* keep track of furthest column in this row, so we won't try to segment
	     pixels already segmented when we continue the search through the image */
	  if (row == results->targets[lasttarget].firstPixel.row && col > furthestCol) {
	    furthestCol = col;
	  }

	  /* increment totalpix */
	  ++results->targets[lasttarget].totalpix;
	  
	  /* add adjacent pixels (if already checked, they will be white) */
	  end = 1; // If end stays 1, then there are no more adjacent pixels in this blob
	  if (row != 0) {
	    if (col != 0 && intensity[(row-1)*numCols+(col-1)] == 0) {
	      addToLists(row-1, col-1, &results->targets[lasttarget]);
	      intensity[(row-1)*numCols+(col-1)] = 255;
	      end = 0; 
	    }
	    if (intensity[(row-1)*numCols+(col)] == 0) {
	      addToLists(row-1, col, &results->targets[lasttarget]);
	      intensity[(row-1)*numCols+(col)] = 255;
	      end = 0;
	    }
	    if (col != numCols-1 && intensity == 0) {
	      addToLists(row-1, col+1, &results->targets[lasttarget]);
	      intensity[(row-1)*numCols+(col+1)] = 255;
	      end = 0;
	    } 
	  }
	  if (col != 0 && intensity[(row)*numCols+(col-1)] == 0) {
	    addToLists(row, col-1, &results->targets[lasttarget]);
	    intensity[(row)*numCols+(col-1)] = 255;
	    end = 0;
	  }
	  if (col != numCols-1 && intensity[(row)*numCols+(col+1)] == 0) {
	    addToLists(row, col+1, &results->targets[lasttarget]);
	    intensity[(row)*numCols+(col+1)] = 255;
	    end = 0;
	  }
	  if (row != numRows-1) {
	    if (col != 0 && intensity[(row+1)*numCols+(col-1)] == 0) {
	      addToLists(row+1, col-1, &results->targets[lasttarget]);
	      intensity[(row+1)*numCols+(col-1)] = 255;
	      end = 0;
	    }
	    if (intensity[(row+1)*numCols+(col)] == 0) {
	      addToLists(row+1, col, &results->targets[lasttarget]);
	      intensity[(row+1)*numCols+(col)] = 255;
	      end = 0;
	    }
	    if (col != numCols-1 && intensity[(row+1)*numCols+(col+1)] == 0) {
	      addToLists(row+1, col+1, &results->targets[lasttarget]);
	      intensity[(row+1)*numCols+(col+1)] = 255;
	      end = 0;
	    }
	  }
	  
	  /* If there are no more directly adjacent rock pixels in this blob, then
	     check for rock pixels NEARBY, but not directly connected */
	  if (end == 1) {
	    /* size of box depends on row (==> distance from camera):
	       the smaller row is, the smaller the box is */
	    boxsize = topBoxSize + (int)((bottomBoxSize - topBoxSize) *
					 (float)row / (float)numRows);
	    for (int i=-boxsize/2; i<boxsize/2; ++i) {
	      for (int j=-boxsize/2; j<boxsize/2; ++j) {
		if (col+j >= 0 && col+j < numCols &&
		    row+i >=0 && row+i < numRows) {
		  if (intensity[(row+1)*numCols+(col+j)] == 0) {
		    addToLists(row+i, col+j, &results->targets[lasttarget]);
		    intensity[(row+1)*numCols+(col+j)] = 255; 
		  } 
		}
	      }
	    }
	  }
	  
	  // Go to next item in list, and then remove the item we just examined 
	  templist = results->targets[lasttarget].leafs;
	  results->targets[lasttarget].leafs = templist->next;
	  free(templist);
	}
	
	/* Now we want to eliminate lines and artifacts */
	//printf("size %d pixels, ", results->targets[lasttarget].totalpix);
	//printf("size/length ratio = %f: ",
	//       results->targets[lasttarget].totalpix / results->targets[lasttarget].maxdist);
	

	/* approximate size based on distance (location in picture) 
	   rocks must be minimum number of pixels depending on distance 
	   at bottom of image, 8 pixels?, at top, 3 pixels? 
	   all targets must have more pixels that "minsize" to be kept */
	minsize = topPixSize + (int)((bottomPixSize - topPixSize) * (float)row / (float)numRows);
	
	if (results->targets[lasttarget].totalpix >= minsize) { // The target is big enough, but is it the right shape??
	  
	  /* check for long skinny targets - if they are sufficiently large,
	     these are probably shadow lines */
	  if (results->targets[lasttarget].totalpix <= 2 * minsize || (float)results->targets[lasttarget].totalpix /
	                                                                    results->targets[lasttarget].maxdist > LINERATIO) {

	    /* find pixel halfway between results->targets[lasttarget].firstPixel and results->targets[lasttarget].lastPixel */
	    cerr << "target #" << lasttarget << " firstPixel.row = " << results->targets[lasttarget].firstPixel.row 
		 << " firstPixel.col = " << results->targets[lasttarget].firstPixel.column
		 << " lastPixel.row = " << results->targets[lasttarget].lastPixel.row
		 << " lastPixel.column = " << results->targets[lasttarget].lastPixel.column << endl;
	    
	    results->targets[lasttarget].midpt.row = 
	      (int)(abs(results->targets[lasttarget].lastPixel.row - results->targets[lasttarget].firstPixel.row) / 2) +
	           MIN(results->targets[lasttarget].firstPixel.row, results->targets[lasttarget].lastPixel.row);

	    results->targets[lasttarget].midpt.column = 
	      (int)(abs(results->targets[lasttarget].lastPixel.column - results->targets[lasttarget].firstPixel.column) / 2) +
	           MIN(results->targets[lasttarget].firstPixel.column, results->targets[lasttarget].lastPixel.column);
	    
	    if(results->targets[lasttarget].midpt.row >= minRow && results->targets[lasttarget].midpt.row <= maxRow) {

	      if(robotPosition != NULL && robotPose != NULL) {
		float distanceEst;
		
		cerr << "** Target is at row = " << results->targets[lasttarget].midpt.row << ", col = " 
		     <<  results->targets[lasttarget].midpt.column << endl;
		
		// Do initial guess with height = 0 assumption
		get_gps(TARGET_ACQ_CAM_PAN, TARGET_ACQ_CAM_TILT + TARGET_ACQ_CAM_TILT_FUDGE, results->targets[lasttarget].midpt.row, 
			results->targets[lasttarget].midpt.column, focalLength(TARGET_ACQ_CAM_ZOOM), 
			robotPose->pitch, robotPose->roll, robotPose->yaw, 0.0,
			&results->targetPos[lasttarget].x, &results->targetPos[lasttarget].y, 
			&results->targetPos[lasttarget].z, &distanceEst);
		
		if(laser != NULL) {
		  // Now add laser data into the mix to improve flat earth assumption
		  int startRow, endRow;
		  get_laser_start_end(results->targetPos[lasttarget].x, results->targetPos[lasttarget].y, TARGET_ACQ_WIDTH, 
				      laser, &startRow, &endRow);
		  float realHeight = ground_height(laser, startRow, endRow);
		  
		  // Now realHeight contains the "approximate" height of the ground at the target.
		  get_gps(TARGET_ACQ_CAM_PAN, TARGET_ACQ_CAM_TILT + TARGET_ACQ_CAM_TILT_FUDGE, results->targets[lasttarget].midpt.row, 
			  results->targets[lasttarget].midpt.column, focalLength(TARGET_ACQ_CAM_ZOOM), 
			  robotPose->pitch, robotPose->roll, robotPose->yaw, realHeight - robotPosition->z,
			  &results->targetPos[lasttarget].x, &results->targetPos[lasttarget].y, 
			  &results->targetPos[lasttarget].z, &distanceEst);
		  
		  results->targetPos[lasttarget].x += robotPosition->x;
		  results->targetPos[lasttarget].y += robotPosition->y;
		  results->targetPos[lasttarget].z += robotPosition->z; /* laser calculations take care of projecting the beams 
									   to the ground,	
									   so DGPS_SENSOR_Z doesn't need to be considered */
		  
		} else {
		  results->targetPos[lasttarget].x += robotPosition->x;
		  results->targetPos[lasttarget].y += robotPosition->y;
		  results->targetPos[lasttarget].z += robotPosition->z - DGPS_SENSOR_Z;
		}
	      }
	      
	      /* only if we're keeping this target... */
	      if(++lasttarget >= MAX_TARGETS_IN_ACQ_DATA) {
		done = 1;
	      }
	    } else {
	      // target is outside acquisition range
	    }
	  } else {
	    // Target is too similar to a line -- throw it out
	  }
	} else {
	  // Target is too small -- throw it out
	}
      }
    }
  }

  for(int i=minRow; i < maxRow; i++) {
    for(int j=MIN_COL; j < MAX_COL; j++) {
      currPixel = i * numCols + j;
      mark = 0;
      for(int k=0; k < lasttarget; k++) {
	if(abs(results->targets[k].midpt.row - i) < 3 &&
	   abs(results->targets[k].midpt.column - j) < 3) {
	       mark = 1;
	}
      }
      if(red[currPixel] + blue[currPixel] + green[currPixel] != 0) {
	if(mark) {
	  red[currPixel] = 255;
	  green[currPixel] = blue[currPixel] = 0;
	} else if((red[currPixel] > redBlueCoeff * blue[currPixel]) && (red[currPixel] > redGreenCoeff * green[currPixel])) {
	  red[currPixel] = green[currPixel] = blue[currPixel] = 0;
	} else {
	  red[currPixel] = green[currPixel] = blue[currPixel] = 255;
	}
      }
    }
  }

  results->numTargets = lasttarget;

  delete(intensity);

  return(1);
}


