/***********************************************************************
 * get_gps : routine to calculate the 3D position of a point given
 *            as row and col in a monoscopic image. The routine assumes
 *            a flat world. It also assumes that the camera is mounted on
 *            a pan tilt head. This was written for the Nov 1998 trip to
 *            Antarctica for the hi-res science camera.
 *
 * INPUTS : pan - angle (in rad) of the pan motor on the PTU
 *          tilt - angle (in rad) of the tilt motor on the PTU
 *          row - row of the point in the camera image
 *          col - column of the point in the camera image
 *          f   - focal length of the camera. In mm.
 *          pitch - pitch of the robot when the image was acquired (rad)
 *          roll  - roll of the robot (rad)
 *          yaw   - yaw of the robot (rad)
 *          height- the height of the ground (in meters) near the target.
 *                  This height is with respect to the bottom of Nomad's
 *                  tires. Ground higher than the tires is positive.
 *                  Essentially height is the assumed pz coordinate of the
 *                  target in Frame 1.
 *
 * OUTPUTS : x - x coordinate of the point with respect to robot. Add to GPS
 *               x coordinate to get absolute coords. In meters.
 *           y - y coord wrt robot. In meters.
 *           z - z coord wrt robot. In meters.
 *           d - distance from the point to the front of the lens of the camera
 *               In meters.
 *
 * Written by Stewart Moorehead Oct 5, 1998.
 * Modified by Stewart Moorhead to accommodate ground height. Oct 21, 1999
 *****************************************************************************/

#include <math.h>
#include <iostream.h>
#include <stdio.h>
#include "gps_calc.h"
#include "hiResSensorDef.h"
#include "telemetryMsgs.h"

/*******
 Global Defines
 *******/
#define GPS_DEBUG 0     //set to one if you want lots of debugging info
                        //to be printed to cerr

void get_gps(float pan, float tilt, int row, int col, float f,
             float pitch, float roll, float yaw, float height,
             float *x, float *y, float *z, float *d)
{

float p_1[3],   /*coordinates of point in frame fixed to robot*/
       p_3[3],   /*coordinates of point in frame fixed to camera*/
       l[3],     /*coordinates of point on lens on p_3 ray in frame 3*/
       x_prime,  /*coordinates of CCD point in frame fixed to camera*/
       z_prime;  /*coordinates of CCD point in frame fixed to camera*/
float stilt, span, ctilt, cpan;
float spitch, sroll, syaw, cpitch, croll, cyaw;

/* standardize units */
f    /= 1000.0;

/*for efficiency compute all the sin's and cos's we will need*/
stilt = sin(tilt); span = sin(pan); ctilt = cos(tilt); cpan = cos(pan);
spitch = sin(pitch); sroll = sin(roll); syaw = sin(yaw);
cpitch = cos(pitch); croll = cos(roll); cyaw = cos(yaw);

/*calculate the coordinates of the row/col in a frame fixed to camera
  (I call this Frame 3 - if you had my notes it would all be clear). This
  gives the coordinates of the point on the CCD which corresponds to the
  sample of interest*/
x_prime = (col-MIDCOL)*(CCD_WIDTH/NUM_COL);
z_prime = (MIDROW-row)*(CCD_WIDTH/NUM_ROW);

/*Now we need to compute the coordinates of the point in Frame 3 of the
  actual object of interest. We can do this by assuming that in Frame 1,
  the frame attached to the robot, the point will have pz = height. Thus
  transforming the point from Frame 3 to Frame 1 and setting p_1[2] = height 
  we can solve for p_3[1]. From this we can calculate p_3[0] and p_3[2]
  using the standard projective geometry camera equations.
*/
p_3[Y] = -(stilt*(CCD_Y-f)+ctilt*CCD_Z+PTU_Z-height)/(stilt + ctilt*z_prime/f);
p_3[X] = p_3[Y]*x_prime/f;
p_3[Z] = p_3[Y]*z_prime/f;

/*Now compute the coordinates of the point in Frame 1*/
p_1[X] = cpan*p_3[X]-span*ctilt*p_3[Y]+span*stilt*p_3[Z]+cpan*CCD_X-
         span*ctilt*(CCD_Y-f)+span*stilt*CCD_Z+PTU_X;
p_1[Y] = span*p_3[X]+cpan*ctilt*p_3[Y]-cpan*stilt*p_3[Z]+span*CCD_X+
         cpan*ctilt*(CCD_Y-f)-cpan*stilt*CCD_Z+PTU_Y;
p_1[Z] = height;

/*finally transform p_1 to frame 0 which is aligned with the differential
  GPS frame but centered on the robot. Thus to get absolute coordinates
  you just need to add in the robot's current x,y,z as recorded by GPS*/
*x = cyaw*croll*p_1[X]+(-1.0*syaw*cpitch+cyaw*sroll*spitch)*p_1[Y]+
     (syaw*spitch+cyaw*sroll*cpitch)*p_1[Z];
*y = syaw*croll*p_1[X]+(cyaw*cpitch+syaw*sroll*spitch)*p_1[Y]+
     (syaw*sroll*cpitch-cyaw*spitch)*p_1[Z];
*z = -1.0*sroll*p_1[X]+croll*spitch*p_1[Y]+croll*cpitch*p_1[Z];

/*now, for Liam, I have to calculate the distance from the point to the front
 of the lens. Basically I calculate the hypotenuse from the focal point to the
 point and subtract the hypotenuse from the focal point to the lens
*/
/*
  l is the point on the lens that the ray from origin of Frame 3 to P
  intersects
*/
l[X] = LENS_LENGTH/p_3[Y]*p_3[X];
l[Y] = LENS_LENGTH;
l[Z] = LENS_LENGTH/p_3[Y]*p_3[Z];

*d = sqrt(p_3[X]*p_3[X]+p_3[Y]*p_3[Y]+p_3[Z]*p_3[Z]) -
     sqrt(l[X]*l[X]+l[Y]*l[Y]+l[Z]*l[Z]);

 cerr << "w/o pose, x = " << p_1[X] << ", y = " << p_1[Y] << ", z = " << p_1[Z] << endl;
 cerr << "x = " << *x << ", y = " << *y << ", z = " << *z << endl;
}

/***************************************************************************
 * ground_height : takes a pose stamped laser scan, and returns the average 
 *                 height of the data from start_laser to end_laser inclusive
 *                 Note we return the height in absolute DGS coordinates
 *                 and not relative to the robot's wheels. This is done
 *                 because the robot's wheels now might be at a different
 *                 altitude than they were when the laser scan was taken.
 *                 Granted this will be a small change and the error in
 * 		   DGPS z coordinate might be greater. Can easily be changed
 *                 if its a problem. 
 * Written by Stewart Moorehead, Oct 21, 1999.
 ***************************************************************************/
float ground_height(laserDerivedState *laser, int start, int end)
{
int i, count = 0;
float height = 0.0;
float z;

/*do some small error checking*/
if (start < 0)
	start = 0;
if (end > 179)
	end = 179;

/*go through each laser of interest, compute it's z component in the
  reference frame, Frame 1 (from get_gps above) - i.e. z=0 is the 
  bottom of the robot wheels, and compute the average.
*/
for (i=start; i<=end; i++)
	{
	z = (laser->laserHeight-laser->rangeMap[i]*cos(laser->laserTilt)*
                   cos(DEG_TO_RAD(i)));
	#if GPS_DEBUG
		cerr << z << " ";
	#endif
	height += z;
	count++;
	}
#if GPS_DEBUG
	cerr << "\n" << "sum = " << height << " count = " << count;
#endif

height = height/count + laser->z;    //return height in absolute DGS coords

#if GPS_DEBUG
	cerr << "\n" << "height = " << height << endl;
#endif

return(height);
}


/***************************************************************************
 * get_laser_start_end : given the x,y position of a target, returns the
 *                       start and end laser's which define a region
 *                       of width, width around that target. start and end
 *                       are intended to be used with ground_height().
 *
 * Inputs : x,y   : DGPS coordinates of target (absolute, not relative to
 *                                           robot) (meters)
 *          width : area of interest around the target (meters)
 *          *laser: pointer to most recent laser scan NDDS message
 *          
 * Outputs : start : number of first item in laser->rangeMap to use
 *           end   : number of last item in laser->rangeMap to use
 *
 * Written by Stewart Moorehead, Oct 21, 1999. 
 ***************************************************************************/


void get_laser_start_end(float x, float y, float width, 
			 laserDerivedState *laser, int *start, int *end)
{
double angle;
double angle_width;   //the angular spread needed to get width m
int middle; 
int spread;

angle = atan2(y - laser->y, x - laser->x);
middle = (int)(180/M_PI*angle);   //round off to nearest int & convert to deg

angle_width = width/laser->rangeMap[middle];
angle_width /= 2;  //get the half width
spread = (int)(180/M_PI*angle_width);

*start = middle - spread;
*end = middle + spread;

//now need to look at some error cases
if (*start < 0)
	*start = 0;

if (*end <= 0)
	*end = spread;

if (*end > 179)
	*end = 179;

if (*start >= 179)
	*start = 179 - spread;


}


