
/* FILE: acqHiResDriver.cpp
   CREATED: May 24, 1999
   AUTHOR: Michael Wagner
   DESCRIPTION: This file implements a target acqusition driver for Nomad's
     high resolution pan/tilt camera.
*/

#include <iostream.h>
#include <string.h>
#include "ndds/NDDS.h"
#include "acqHiResDriver.h"
#include "hiResSensorDef.h"
#include "telemetryMsgs.h"
#include "nddsMsgNames.h"
#include "SAS_Config.h"

// NDDS publish/subscribe messages.
RTIBool laserDerivedStateCallback(NDDSRecvInfo *issue) {
  if(issue->status == NDDS_FRESH_DATA) {
    laserDerivedState *laserDerivedStateMsg = (laserDerivedState *)issue->callBackRtnParam;
    memcpy(laserDerivedStateMsg, (laserDerivedState *)issue->instance, sizeof(laserDerivedState));
  }
  return(RTI_TRUE);
}

acqHiResDriver::acqHiResDriver(DGPS *currPosition, pose *currPose) : acqDriver(currPosition, currPose) {
  if(!getSAS_Config(TARGET_ACQ_START_ROW, this->startRow)) {
    cerr << "[acqHiResDriver] ERROR: Cannot read " << TARGET_ACQ_START_ROW << " from config file!" << endl;
    return;
  }
  if(!getSAS_Config(TARGET_ACQ_STOP_ROW, this->stopRow)) {
    cerr << "[acqHiResDriver] ERROR: Cannot read " << TARGET_ACQ_STOP_ROW << " from config file!" << endl;
    return;
  }
  this->currPosition = currPosition;
  this->currPose = currPose;
  strcpy(this->driverID, HI_RES_DRIVER_ID);
  sensorClientSetup();
  this->image = new ppmRockSeg(laserDerivedStateMsg);
#ifdef RED_TEST
  cerr << "[acqHiResDriver] Starting up using red threshold technique..." << endl;
#endif
}

acqHiResDriver::~acqHiResDriver() {
  delete(this->image);
}

int acqHiResDriver::setupAcq() {
  // *** Perform unDeploy ***
  strcpy(sensorRequestMsg->sensorID, HI_RES_SENSOR_ID);
  sensorRequestMsg->function = SENSOR_UNDEPLOY_FUNCTION;

  clientStatus = NddsClientCallAndWait(client, sensorReplyMsg, sensorRequestMsg,
				       SENSOR_MIN_WAIT_TIME, SENSOR_MAX_WAIT_TIME);
  if(clientStatus == NDDS_CLIENT_RECV_REPLY) {
    if(sensorReplyMsg->status != SENSOR_OK) {
      // Failure condition.
      // TBD: real diagnosis?
      cerr << "[" << this->driverID << "] ERROR: On setupAcq(), \"" << HI_RES_SENSOR_ID << "\" returned " << sensorReplyMsg->status << endl;
      return(TARGET_ACQ_FAILED);
    }
  } else {
    // Sensor did not respond to NDDS message
    cerr << "[" << this->driverID << "] ERROR: On setupAcq(), \"" << HI_RES_SENSOR_ID << "\" did not respond to request!" << endl;
    return(TARGET_ACQ_FAILED);
  }

  return(TARGET_ACQ_SUCCESSFUL);
}

int acqHiResDriver::examine(acqResults *results) {
  DGPS savedPosition;
  pose savedPose;

  // *** Perform DAQ ***

  strcpy(sensorRequestMsg->sensorID, HI_RES_SENSOR_ID);
  sensorRequestMsg->function = SENSOR_TARGET_ACQ_FUNCTION;

  // Save these at the beginning, when the image is actually taken.
  memcpy(&savedPosition, currPosition, sizeof(DGPS));
  memcpy(&savedPose, currPose, sizeof(pose));

  int sensorNum = 0;
  sensorRequestMsg->sensorNum = sensorNum;

  clientStatus = NddsClientCallAndWait(client, sensorReplyMsg, sensorRequestMsg,
				       SENSOR_MIN_WAIT_TIME, SENSOR_MAX_WAIT_TIME);
  if(clientStatus == NDDS_CLIENT_RECV_REPLY) {
    if(sensorReplyMsg->status != SENSOR_OK) {
      // Failure condition.
      // TBD: real diagnosis?
      cerr << "[" << this->driverID << "] ERROR: \"" << HI_RES_SENSOR_ID << "\" returned " << sensorReplyMsg->status << endl;
      return(TARGET_ACQ_FAILED);
    }
  } else {
    // Sensor did not respond to NDDS message
    cerr << "[" << this->driverID << "] ERROR: \"" << HI_RES_SENSOR_ID << "\" did not respond to request!" << endl;
    return(TARGET_ACQ_FAILED);
  }

  // *** Now use data in sensorReply->data ***

  int readingNum = sensorReplyMsg->data.sensorData[sensorNum].numReadings - 1;
  if(strcmp(sensorReplyMsg->data.sensorData[sensorNum].readings[readingNum].filename, "\0") == 0) {
    cerr << "[" << this->driverID << "] ERROR: Received no sensor data filename from sensor!" << endl;
    return(TARGET_ACQ_FAILED);
  }

  if(image->read(sensorReplyMsg->data.sensorData[sensorNum].readings[readingNum].filename)) {
#ifdef RED_TEST
    if(!image->redThreshhold(&savedPosition, &savedPose, 180, 300, results)) {
      return(TARGET_ACQ_FAILED);
    }
#else
    if(!image->kimThreshhold(&savedPosition, &savedPose, 180, 300, results)) {
      return(TARGET_ACQ_FAILED);
    }
#endif
    
    if(results->numTargets > 0) {
#ifdef DEBUG
      image->write("/tmp/segmentedImage.ppm");
#endif
    }
    return(TARGET_ACQ_SUCCESSFUL);
    
  } else {
    
    cerr << "[acqHiResDriver] ERROR: Could not read image \"" << sensorReplyMsg->data.sensorData[sensorNum].readings[readingNum].filename << "\"\n";
    return(TARGET_ACQ_FAILED);
  }
}

void acqHiResDriver::sensorClientSetup() {
  sensorRequestNddsRegister();
  sensorReplyNddsRegister();
  sensorRequestMsg = (sensorRequest *)calloc(1, sizeof(sensorRequest));
  sensorRequestMsg->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  sensorReplyMsg = (sensorReply *)calloc(1, sizeof(sensorReply));
  sensorReplyMsg->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  client = NddsClientCreate(HI_RES_SENSOR_ID, sensorReplyPublicationType, sensorRequestPublicationType);
  if(NddsClientServerWait(client, 2.0, 5, 1) == RTI_FALSE) { 
    cerr << "[" << this->driverID << "] Could not contact sensor \"" << HI_RES_SENSOR_ID << "\"!" << endl;
  } else {
    cerr << "[" << this->driverID << "] Created sensor \"" << HI_RES_SENSOR_ID << "\" client!" << endl;
  }

  // laserDerivedState messages will assist in creating DGPS estimates. 
  laserDerivedStateNddsRegister(); 
  laserDerivedStateMsg = (laserDerivedState *)calloc(1, sizeof(laserDerivedState));
  laserSub = NddsSubscriptionCreate(NDDS_SUBSCRIPTION_IMMEDIATE, /* TBD: If it's IMMEDIATE, the program crashes! */
				    LASER_DERIVED_STATE_MSG_NAME,
				    laserDerivedStatePublicationType,
				    laserDerivedStateMsg, TELEMETRY_DEADLINE,
				    TELEMETRY_MIN_SEPARATION,
				    laserDerivedStateCallback, laserDerivedStateMsg);
}  


