#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "navplan.h"
#include "SunMoonPos.h"
#include "solar.h"

/* Globals - robot configuration */
double PANEL_SIZE = 1.0; /* 1m x 1m square panels */
double POWER_GAIN = 100.0; /* power gain is Watts/square area */


/* Internal structures and function declarations */
typedef struct pixel_list_t{
  map_pixel pixel;
  struct pixel_list_t* next;
} pixel_list;

/* Subroutine and conversion functions */
double reduce(double value, double rangeMin, double rangeMax);
double absol(double vector[3]);
void vectdiff(const double one[3], const double two[3], double result[3]);
void matvectmult(double matrix[3][3], double vector[3], double result[3]);
double dot(double vector1[3], double vector2[3]);
double abscross(double vector1[3], double vector2[3]);
void Spherical1(double eps, double delta, double ra, double* lat,double* lon);
void Spherical2(double eps, double* delta, double* ra, double lat,double lon);
void Cartesian(double lat, double lon, double dist, double vector[]);

/* Sun position functions */
void getSunPos(double Jdate, double surflat, double surflon,
	       double *altitude, double *azimuth);

/* Terrain functions */
void findPos(MP_POSE *newpos, MP_POSE currpos, int dir);
void addPix(pixel_list *curr, map_pixel *pix);
int check_elev(MP_POSE robot, map_pixel pix, double sunElev,
	       double *theta);
double findPixels(MP_POSE robot, double angle, double sunElev,
		  planner_map map);


/****************************************************************************/
/* Subroutines & Conversions                                                */
/*   vectdiff                                                               */
/*   vectadd                                                                */
/*   matvectmult                                                            */
/*   dot                                                                    */
/*   abscross                                                               */
/*   Spherical1                                                             */
/*   Spherical2                                                             */
/*   Cartesian                                                              */
/****************************************************************************/

/****************************************************************************/
/* vectdiff: calculates the difference between two vectors                  */
/****************************************************************************/
void vectdiff(const double one[3], const double two[3], double result[3]) {
  int i;

  for (i=0; i<=2; ++i)
    result[i] = one[i] - two[i];

  return;
}

/****************************************************************************/
/* vectadd: calculates the sum of two vectors                               */
/****************************************************************************/
void vectadd(const double one[3], const double two[3], double result[3]) {
  int i;

  for (i=0; i<=2; ++i)
    result[i] = one[i] + two[i];

  return;
}

/****************************************************************************/
/* matvectmult: multiplies a 3x3 matrix with a 3x1 vector                   */
/****************************************************************************/
void matvectmult(double matrix[3][3], double vector[3], double result[3]) {
  int i;

  for (i=0; i<=2; ++i)
    result[i] = matrix[i][0]*vector[0] + matrix[i][1]*vector[1] +
      matrix[i][2]*vector[2];

  return;
}

/****************************************************************************/
/* dot: calculates the dot product of two vectors                           */
/****************************************************************************/
double dot(double vector1[3], double vector2[3]) {
  double retval;
 
  retval = vector1[0]*vector2[0] + vector1[1]*vector2[1] +
    vector1[2]*vector2[2];

  return(retval);
}

/****************************************************************************/
/* abscross: calculates the signed absolute value of the cross product of   */
/*           two vectors                                                    */
/****************************************************************************/
double abscross(double vector1[3], double vector2[3]) {
  double retval;
  double cross[3];
 
  cross[0] = vector1[1]*vector2[2] - vector1[2]*vector2[1];
  cross[1] = vector1[2]*vector2[0] - vector1[0]*vector2[2];
  cross[2] = vector1[0]*vector2[1] - vector1[1]*vector2[0];

  retval = absol(cross);

  return(retval);
}
 
/****************************************************************************/
/* Spherical1                                                               */
/*   Converts one set of spherical coordinates into another set of          */
/*   spherical coordinates.                                                 */
/*   eps is the angle between the two planes at the ascending node of       */
/*    plane 2 on plane 1                                                    */
/*   delta and ra are the latitude and longitude of plane 1                 */
/*    (the lower plane) in radians                                          */
/*   lat and lon are the latitude and longitude of plane 2 in radians       */
/*     (the unknowns being solved for)                                      */
/****************************************************************************/
void Spherical1(double eps, double delta, double ra, double* lat, double* lon){
  double a, b, c;

  a = sin(delta)*cos(eps) - cos(delta)*sin(eps)*sin(ra);
  b = cos(ra)*cos(delta);
  c = sin(delta)*sin(eps) + cos(delta)*cos(eps)*sin(ra);

  if ((delta == PI/2) | (delta == -PI/2)) *lon = PI/2;
  else *lon = atan2(c, b);

  *lat = asin(a);

  return;
}

/****************************************************************************/
/* Spherical2                                                               */
/*   Converts one set of spherical coordinates into another set of          */
/*   spherical coordinates.                                                 */
/*   eps is the angle between the two planes at the ascending node of       */
/*    plane 2 on plane 1                                                    */
/*   delta and ra are the latitude and longitude of plane 1                 */
/*    (the lower plane) in radians (the unknowns being solved for)          */
/*   lat and lon are the latitude and longitude of plane 2 in radians       */
/****************************************************************************/
void Spherical2(double eps, double* delta, double* ra, double lat, double lon){
  double a, b, c;

  a = sin(lat)*cos(eps) + cos(lat)*sin(eps)*sin(lon);
  b = cos(lat)*cos(lon);
  c = -sin(lat)*sin(eps) + cos(lat)*cos(eps)*sin(lon);

  if ((lat == PI/2) | (lat == -PI/2)) *ra = PI/2;
  else *ra = atan2(c, b);

  *delta = asin(a);

  return;
}

/****************************************************************************/
/* Cartesian                                                                */
/*   Converts the spherical geocentric equatorial coordinates output by     */
/*   SunMoonPos subroutine into cartesian geocentric equatorial coordinates */
/*    (x axis is in direction of vernal equinox, or Greenwich meridian)     */
/****************************************************************************/
void Cartesian(double lat, double lon, double dist, double vector[]) {

  vector[0] = dist * cos(lat) * cos(lon);
  vector[1] = dist * cos(lat) * sin(lon);
  vector[2] = dist * sin(lat);

  return;
}


/****************************************************************************/
/* Sun position from Earth                                                  */
/*   getSunPos                                                              */
/****************************************************************************/

/****************************************************************************/
/* This function takes the Julian date and the latitude and longitude of a  */
/* surface position (in radians), and returns the altitude and azimuth of   */
/* the sun (in radians) for that date and location.                         */
/****************************************************************************/
void getSunPos(double Jdate, double surflat, double surflon,
	       double *altitude, double *azimuth) {
  double sunRA, sunDec, sunDist;     /* Equatorial (geocentric) coordinates
					of center of sun                    */
  double foo;
  double H, GMST, T;                 /* hour angle, sidereal time           */

  /*  Get the sun's position for the date */
  SunMoonPos(Jdate, &sunRA, &sunDec, &sunDist, &foo, &foo, &foo);

  /* should add an int flag for Earth, Moon, Mars location... */

  /* convert RA to hour angle for this time and robot longitude */
  /* First, find sidereal time, GMST, in degrees */
  T = (Jdate - 2451545.0) / 36525.0;
  GMST = 280.46061837 + 360.98564736629 * (Jdate - 2451545.0) +
    0.000387933 * T * T - T * T * T / 38710000.0;
  GMST = GMST - ((int)(GMST / 360.0) * 360.0);
  if (GMST < 0.0) GMST = GMST + 360.0;

  /* convert GMST from degrees to radians */
  GMST = GMST * PI / 180.0;

  /* Next, find hour angle H (in radians) */
  H = GMST - surflon - sunRA;

  *azimuth = atan2(-1.0*sin(H)*cos(sunDec), cos(surflat)*sin(sunDec) -
		   sin(surflat)*cos(sunDec)*cos(H));
  *altitude = asin(sin(surflat)*sin(sunDec) +
		   cos(surflat)*cos(sunDec)*cos(H));

  /* Could consider elevation above perfectly smooth sphere, but at this
     distance from sun, difference is negligible:
     tan(altitude_elev) = tan(altitude_sealevel) -
     elevation / (distance of sun from surface, projected onto surface plane)
     This latter fraction is of the order 10E-9, even for 1km elevation */
}


/****************************************************************************/
/* Terrain map functions                                                    */
/*  findPos                                                                 */
/*  addPix                                                                  */
/*  check_elev                                                              */
/*  findPixels                                                              */
/****************************************************************************/

/****************************************************************************/
/* This function finds the new position one pixel away from currpos in      */
/* the direction "dir"                                                      */
/****************************************************************************/
void findPos(MP_POSE *newpos, MP_POSE currpos, int dir) {

  if (dir == UP) {
    newpos->x = currpos.x; newpos->y = currpos.y+PP_PIXEL_SIZE; }
  else if (dir == UPPERRIGHT) {
    newpos->x = currpos.x+PP_PIXEL_SIZE; newpos->y = currpos.y+PP_PIXEL_SIZE;}
  else if (dir == RIGHT) {
    newpos->x = currpos.x+PP_PIXEL_SIZE; newpos->y = currpos.y; }
  else if (dir == LOWERRIGHT) {
    newpos->x = currpos.x+PP_PIXEL_SIZE; newpos->y = currpos.y-PP_PIXEL_SIZE;}
  else if (dir == DOWN) {
    newpos->x = currpos.x; newpos->y = currpos.y-PP_PIXEL_SIZE; }
  else if (dir == LOWERLEFT) {
    newpos->x = currpos.x-PP_PIXEL_SIZE; newpos->y = currpos.y-PP_PIXEL_SIZE;}
  else if (dir == LEFT) {
    newpos->x = currpos.x-PP_PIXEL_SIZE; newpos->y = currpos.y; }
  else if (dir == UPPERLEFT) {
    newpos->x = currpos.x-PP_PIXEL_SIZE; newpos->y = currpos.y+PP_PIXEL_SIZE;}
}

/****************************************************************************/
/* This function adds the pixel pix to the list of pixels curr, setting it  */
/* to curr->next.                                                           */
/* curr->next is malloc'ed, and curr->next->next is set NULL.               */
/****************************************************************************/
void addPix(pixel_list *curr, map_pixel *pix) {
  curr->next = (pixel_list*)malloc(sizeof(pixel_list));
  curr->next->pixel.i = pix->i;
  curr->next->pixel.j = pix->j;
  curr->next->pixel.elevation = pix->elevation;
  curr->next->pixel.block = pix->block;
  curr->next->next = NULL;
}

/****************************************************************************/
/* This function checks the elevation of pix as seen from the robot's       */
/* location, comparing to the sun's elevation.  If the pix elevation blocks */
/* the sun, the function returns 1, otherwise 0.  The elevation is returned */
/* in the variable theta.                                                   */
/****************************************************************************/
int check_elev(MP_POSE robot, map_pixel pix, double sunElev, double *theta) {
  float dist;
  float alpha;
  float a, c;
  int quit = 0;
  MP_POSE currpos;

  /* includes curvature of Earth - non-negligible for distant objects */
  currpos = find_pixel_GPS(&pix);
  dist = sqrt(sqr(robot.x - currpos.x) + sqr(robot.y - currpos.y));
  if (dist == 0.0) {
    dist = PP_PIXEL_SIZE;
    printf("Warning:  distance between pixels approximated\n");
    /* pixels must be at least this far apart - be on the safe side */
    /* problem occurs when absolute positions are large enough for floats
       not to be big enough to hold a delta position of 0.50 meters - rarely
       happens since we use a small enough map */
  }

  /* should have an int flag for Earth, Moon, Mars... */

  alpha = dist / (EARTHRADIUS * 1000.0); /* in meters */

  c = (EARTHRADIUS*1000.0 + pix.elevation) * sin(alpha);
  a = (EARTHRADIUS*1000.0 + pix.elevation) * cos(alpha) -
    (EARTHRADIUS*1000.0 + robot.z);

  *theta = (double)atan2(a, c);
  if (*theta > sunElev) quit = 1;

  return(quit);
}


/****************************************************************************/
/* This function finds all the pixels in the local map which are in line    */
/* between the robot and the sun.  It returns the maximum elevation of the  */
/* terrain in that direction.                                               */
/****************************************************************************/
double findPixels(MP_POSE robot, double azimuth, double sunElev,
		  planner_map map) {
  pixel_list *Firstpix, *temp, *curr;
  map_pixel *newpix;
  MP_POSE currpos, newpos;
  double offset, deltax, deltay;
  double theta;
  int status;
  int quit = 0;
  double maxElev = -1.0 * PI;
  double angle;

  /* convert azimuth into angle from map's x-axis */
  /* (azimuth is angle from north towards the east) */
  /* --right now assume coordinates are the same-- */
  angle = azimuth;


  Firstpix = curr = (pixel_list *)malloc(sizeof(pixel_list));

  /* find pixel in which robot is sitting */
  newpix = find_pixel_addr(map, robot, &status);
  curr->pixel.i = newpix->i;
  curr->pixel.j = newpix->j;
  curr->pixel.elevation = newpix->elevation;
  curr->pixel.block = newpix->block;
  curr->next = NULL;

  /* find octant in which sun lies - make sure 0 <= angle < TWOPI */
  while (angle < 0.0) angle = angle + TWOPI;
  while (angle > TWOPI) angle = angle - TWOPI;

  if (angle <= PI / 4.0) {
    offset = 0.5 * PP_PIXEL_SIZE * (1.0 - tan(angle));
    deltax = PP_PIXEL_SIZE;
    deltay = deltax * tan(angle);
    while (!quit) {
      if (deltay >= offset) {
	currpos = find_pixel_GPS(newpix);
	findPos(&newpos, currpos, RIGHT);
	newpix = find_pixel_addr(map, newpos, &status);
	if (status == -1) quit = 1;
	else {
	  addPix(curr, newpix);
	  curr = curr->next;
	  quit = check_elev(robot, curr->pixel, sunElev, &theta);
	  if (theta > maxElev) maxElev = theta;
	  if (!quit) {
	    findPos(&newpos, currpos, UPPERRIGHT);
	    newpix = find_pixel_addr(map, newpos, &status);
	    if (status == -1) quit = 1;
	    else {
	      addPix(curr, newpix);
	      curr = curr->next;
	      quit = check_elev(robot, curr->pixel, sunElev, &theta);
	      if (theta > maxElev) maxElev = theta;
	      offset = PP_PIXEL_SIZE + offset - deltay; } } } }
      else {
	currpos = find_pixel_GPS(newpix);
	findPos(&newpos, currpos, RIGHT);
	newpix = find_pixel_addr(map, newpos, &status);
	if (status == -1) quit = 1;
	else {
	  addPix(curr, newpix);
	  curr = curr->next;
	  quit = check_elev(robot, curr->pixel, sunElev, &theta);
	  if (theta > maxElev) maxElev = theta;
	  offset = offset - deltay; }
      }
    }
  }
  else if (angle <= PI / 2.0) {
    offset = 0.5 * PP_PIXEL_SIZE * (1.0 - 1.0 / tan(angle));
    deltay = PP_PIXEL_SIZE;
    deltax = deltay / tan(angle);
    while (!quit) {
      if (deltax >= offset) {
	currpos = find_pixel_GPS(newpix);
	findPos(&newpos, currpos, UP);
	newpix = find_pixel_addr(map, newpos, &status);
	if (status == -1) quit = 1;
	else {
	  addPix(curr, newpix);
	  curr = curr->next;
	  quit = check_elev(robot, curr->pixel, sunElev, &theta);
	  if (theta > maxElev) maxElev = theta;
	  if (!quit) {
	    findPos(&newpos, currpos, UPPERRIGHT);
	    newpix = find_pixel_addr(map, newpos, &status);
	    if (status == -1) quit = 1;
	    else {
	      addPix(curr, newpix);
	      curr = curr->next;
	      quit = check_elev(robot, curr->pixel, sunElev, &theta);
	      if (theta > maxElev) maxElev = theta;
	      offset = PP_PIXEL_SIZE + offset - deltax; } } } }
      else {
	currpos = find_pixel_GPS(newpix);
	findPos(&newpos, currpos, UP);
	newpix = find_pixel_addr(map, newpos, &status);
	if (status == -1) quit = 1;
	else {
	  addPix(curr, newpix);
	  curr = curr->next;
	  quit = check_elev(robot, curr->pixel, sunElev, &theta);
	  if (theta > maxElev) maxElev = theta;
	  offset = offset - deltax; }
      }
    }
  }
  else if (angle <= 3.0 * PI / 4.0) {
    offset = -0.5 * PP_PIXEL_SIZE * (1.0 - 1.0 / tan(angle));
    deltay = PP_PIXEL_SIZE;
    deltax = deltay / tan(angle);
    while (!quit) {
      if (deltax <= offset) {
	currpos = find_pixel_GPS(newpix);
	findPos(&newpos, currpos, UP);
	newpix = find_pixel_addr(map, newpos, &status);
	if (status == -1) quit = 1;
	else {
	  addPix(curr, newpix);
	  curr = curr->next;
	  quit = check_elev(robot, curr->pixel, sunElev, &theta);
	  if (theta > maxElev) maxElev = theta;
	  if (!quit) {
	    findPos(&newpos, currpos, UPPERLEFT);
	    newpix = find_pixel_addr(map, newpos, &status);
	    if (status == -1) quit = 1;
	    else {
	      addPix(curr, newpix);
	      curr = curr->next;
	      quit = check_elev(robot, curr->pixel, sunElev, &theta);
	      if (theta > maxElev) maxElev = theta;
	      offset = -1.0 * PP_PIXEL_SIZE + offset - deltax; } } } }
      else {
	currpos = find_pixel_GPS(newpix);
	findPos(&newpos, currpos, UP);
	newpix = find_pixel_addr(map, newpos, &status);
	if (status == -1) quit = 1;
	else {
	  addPix(curr, newpix);
	  curr = curr->next;
	  quit = check_elev(robot, curr->pixel, sunElev, &theta);
	  if (theta > maxElev) maxElev = theta;
	  offset = offset - deltax; }
      }
    }
  }
  else if (angle <= PI) {
    offset = 0.5 * PP_PIXEL_SIZE * (1.0 + tan(angle));
    deltax = -1.0 * PP_PIXEL_SIZE;
    deltay = deltax * tan(angle);
    while (!quit) {
      if (deltay >= offset) {
	currpos = find_pixel_GPS(newpix);
	findPos(&newpos, currpos, LEFT);
	newpix = find_pixel_addr(map, newpos, &status);
	if (status == -1) quit = 1;
	else {
	  addPix(curr, newpix);
	  curr = curr->next;
	  quit = check_elev(robot, curr->pixel, sunElev, &theta);
	  if (theta > maxElev) maxElev = theta;
	  if (!quit) {
	    findPos(&newpos, currpos, UPPERLEFT);
	    newpix = find_pixel_addr(map, newpos, &status);
	    if (status == -1) quit = 1;
	    else {
	      addPix(curr, newpix);
	      curr = curr->next;
	      quit = check_elev(robot, curr->pixel, sunElev, &theta);
	      if (theta > maxElev) maxElev = theta;
	      offset = PP_PIXEL_SIZE + offset - deltay; } } } }
      else {
	currpos = find_pixel_GPS(newpix);
	findPos(&newpos, currpos, LEFT);
	newpix = find_pixel_addr(map, newpos, &status);
	if (status == -1) quit = 1;
	else {
	  addPix(curr, newpix);
	  curr = curr->next;
	  quit = check_elev(robot, curr->pixel, sunElev, &theta);
	  if (theta > maxElev) maxElev = theta;
	  offset = offset - deltay; }
      }
    }
  }
  else if (angle <= 5.0 * PI / 4.0) {
    offset = -0.5 * PP_PIXEL_SIZE * (1.0 - tan(angle));
    deltax = -1.0 * PP_PIXEL_SIZE;
    deltay = deltax * tan(angle);
    while (!quit) {
      if (deltay <= offset) {
	currpos = find_pixel_GPS(newpix);
	findPos(&newpos, currpos, LEFT);
	newpix = find_pixel_addr(map, newpos, &status);
	if (status == -1) quit = 1;
	else {
	  addPix(curr, newpix);
	  curr = curr->next;
	  quit = check_elev(robot, curr->pixel, sunElev, &theta);
	  if (theta > maxElev) maxElev = theta;
	  if (!quit) {
	    findPos(&newpos, currpos, LOWERLEFT);
	    newpix = find_pixel_addr(map, newpos, &status);
	    if (status == -1) quit = 1;
	    else {
	      addPix(curr, newpix);
	      curr = curr->next;
	      quit = check_elev(robot, curr->pixel, sunElev, &theta);
	      if (theta > maxElev) maxElev = theta;
	      offset = -1.0 * PP_PIXEL_SIZE + offset - deltay; } } } }
      else {
	currpos = find_pixel_GPS(newpix);
	findPos(&newpos, currpos, LEFT);
	newpix = find_pixel_addr(map, newpos, &status);
	if (status == -1) quit = 1;
	else {
	  addPix(curr, newpix);
	  curr = curr->next;
	  quit = check_elev(robot, curr->pixel, sunElev, &theta);
	  if (theta > maxElev) maxElev = theta;
	  offset = offset - deltay; }
      }
    }
  }
  else if (angle <= 3.0 * PI / 2.0) {
    offset = -0.5 * PP_PIXEL_SIZE * (1.0 - 1.0 / tan(angle));
    deltay = -1.0 * PP_PIXEL_SIZE;
    deltax = deltay / tan(angle);
    while (!quit) {
      if (deltax <= offset) {
	currpos = find_pixel_GPS(newpix);
	findPos(&newpos, currpos, DOWN);
	newpix = find_pixel_addr(map, newpos, &status);
	if (status == -1) quit = 1;
	else {
	  addPix(curr, newpix);
	  curr = curr->next;
	  quit = check_elev(robot, curr->pixel, sunElev, &theta);
	  if (theta > maxElev) maxElev = theta;
	  if (!quit) {
	    findPos(&newpos, currpos, LOWERLEFT);
	    newpix = find_pixel_addr(map, newpos, &status);
	    if (status == -1) quit = 1;
	    else {
	      addPix(curr, newpix);
	      curr = curr->next;
	      quit = check_elev(robot, curr->pixel, sunElev, &theta);
	      if (theta > maxElev) maxElev = theta;
	      offset = -1.0 * PP_PIXEL_SIZE + offset - deltax; } } } }
      else {
	currpos = find_pixel_GPS(newpix);
	findPos(&newpos, currpos, DOWN);
	newpix = find_pixel_addr(map, newpos, &status);
	if (status == -1) quit = 1;
	else {
	  addPix(curr, newpix);
	  curr = curr->next;
	  quit = check_elev(robot, curr->pixel, sunElev, &theta);
	  if (theta > maxElev) maxElev = theta;
	  offset = offset - deltax; }
      }
    }
  }
  else if (angle <= 7.0 * PI / 4.0) {
    offset = 0.5 * PP_PIXEL_SIZE * (1.0 + 1.0 / tan(angle));
    deltay = -1.0 * PP_PIXEL_SIZE;
    deltax = deltay / tan(angle);
    while (!quit) {
      if (deltax >= offset) {
	currpos = find_pixel_GPS(newpix);
	findPos(&newpos, currpos, DOWN);
	newpix = find_pixel_addr(map, newpos, &status);
	if (status == -1) quit = 1;
	else {
	  addPix(curr, newpix);
	  curr = curr->next;
	  quit = check_elev(robot, curr->pixel, sunElev, &theta);
	  if (theta > maxElev) maxElev = theta;
	  if (!quit) {
	    findPos(&newpos, currpos, LOWERRIGHT);
	    newpix = find_pixel_addr(map, newpos, &status);
	    if (status == -1) quit = 1;
	    else {
	      addPix(curr, newpix);
	      curr = curr->next;
	      quit = check_elev(robot, curr->pixel, sunElev, &theta);
	      if (theta > maxElev) maxElev = theta;
	      offset = PP_PIXEL_SIZE + offset - deltax; } } } }
      else {
	currpos = find_pixel_GPS(newpix);
	findPos(&newpos, currpos, DOWN);
	newpix = find_pixel_addr(map, newpos, &status);
	if (status == -1) quit = 1;
	else {
	  addPix(curr, newpix);
	  curr = curr->next;
	  quit = check_elev(robot, curr->pixel, sunElev, &theta);
	  if (theta > maxElev) maxElev = theta;
	  offset = offset - deltax; }
      }
    }
  }
  else {
    offset = -0.5 * PP_PIXEL_SIZE * (1.0 + tan(angle));
    deltax = PP_PIXEL_SIZE;
    deltay = deltax * tan(angle);
    while (!quit) {
      if (deltay <= offset) {
	currpos = find_pixel_GPS(newpix);
	findPos(&newpos, currpos, RIGHT);
	newpix = find_pixel_addr(map, newpos, &status);
	if (status == -1) quit = 1;
	else {
	  addPix(curr, newpix);
	  curr = curr->next;
	  quit = check_elev(robot, curr->pixel, sunElev, &theta);
	  if (theta > maxElev) maxElev = theta;
	  if (!quit) {
	    findPos(&newpos, currpos, LOWERRIGHT);
	    newpix = find_pixel_addr(map, newpos, &status);
	    if (status == -1) quit = 1;
	    else {
	      addPix(curr, newpix);
	      curr = curr->next;
	      quit = check_elev(robot, curr->pixel, sunElev, &theta);
	      if (theta > maxElev) maxElev = theta;
	      offset = -1.0 * PP_PIXEL_SIZE + offset - deltay; } } } }
      else {
	currpos = find_pixel_GPS(newpix);
	findPos(&newpos, currpos, RIGHT);
	newpix = find_pixel_addr(map, newpos, &status);
	if (status == -1) quit = 1;
	else {
	  addPix(curr, newpix);
	  curr = curr->next;
	  quit = check_elev(robot, curr->pixel, sunElev, &theta);
	  if (theta > maxElev) maxElev = theta;
	  offset = offset - deltay; }
      }
    }
  }

  /* Free Firstpix list */
  while (Firstpix != NULL) {
    temp = Firstpix->next;
    free(Firstpix);
    Firstpix = temp;
  }

  return(maxElev);
}


/****************************************************************************/
/* compute sun altitude, azimuth & visibility                               */
/****************************************************************************/
double solar(MP_POSE robot, float xaxis, float robot_lat,
	     float robot_lon, double Jdate, planner_map map,
	     char RecordSun[40]) {
  double altitude, azimuth;
  double maxElev;
  double sunpos[3];
  double rot[3][3];
  double temp[3], panel1[3], panel2[3];
  double panel3[3];  /* this is for a horizontal panel */
  double cosangle1, cosangle2, cosangle3;
  double power1, power2, power3;
  double lat, lon;
  FILE *Sunpos;
  map_pixel *newpix;
  int status;

  /* should have an int flag for Earth, Moon, Mars... */

  /* find lat/lon difference from DGPS base station */
  lat = (double)robot_lat + robot.y / (EARTHRADIUS * 1000.0);
  lon = (double)robot_lon - robot.x / (EARTHRADIUS * 1000.0);

  /* Make sure robot.z is filled in */
  newpix = find_pixel_addr(map, robot, &status);
  if (status == -1) printf("Error:  robot location is not in map\n");
  else robot.z = newpix->elevation;

  /* find sun altitude and azimuth at robot location */
  getSunPos(Jdate, lat, lon, &altitude, &azimuth);

  /* find maximum terrain elevation */
  maxElev = findPixels(robot, azimuth, altitude, map);

  if (maxElev >= altitude) {
    power1 = 0.0;
    power2 = 0.0;
    power3 = 0.0; }
  else {
    /* compute unit vector to sun */
    sunpos[0] = (float)(cos(PI/2.0 - azimuth) * cos(altitude));
    sunpos[1] = (float)(sin(PI/2.0 - azimuth) * cos(altitude));
    sunpos[2] = (float)sin(altitude);

    /* use robot roll, yaw, to calculate normals to solar panels */
    /* (robot.yaw of 0.0 means robot is pointed in +y direction) */
    /* confirm +- of roll, pitch... */
    rot[0][0] = cos(robot.yaw) * cos(robot.roll);
    rot[0][1] = cos(robot.yaw) * sin(robot.roll) * sin(robot.pitch) -
                sin(robot.yaw) * cos(robot.pitch);
    rot[0][2] = cos(robot.yaw) * sin(robot.roll) * cos(robot.pitch) +
                sin(robot.yaw) * sin(robot.pitch);
    rot[1][0] = sin(robot.yaw) * cos(robot.roll);
    rot[1][1] = sin(robot.yaw) * sin(robot.roll) * sin(robot.pitch) +
                cos(robot.yaw) * cos(robot.pitch);
    rot[1][2] = sin(robot.yaw) * sin(robot.roll) * cos(robot.pitch) -
                cos(robot.yaw) * sin(robot.pitch);
    rot[2][0] = -sin(robot.roll);
    rot[2][1] = cos(robot.roll) * sin(robot.pitch);
    rot[2][2] = cos(robot.roll) * cos(robot.pitch);

    /* This is where the robot configuration model comes in */
    temp[0] = 1.0;
    temp[1] = 0.0;
    temp[2] = 0.0;
    matvectmult(rot, temp, panel1);

    temp[0] = -1.0;
    temp[1] = 0.0;
    temp[2] = 0.0;
    matvectmult(rot, temp, panel2);

    temp[0] = 0.0;
    temp[1] = 0.0;
    temp[2] = 1.0;
    matvectmult(rot, temp, panel3);

    /* calculate angles between sun and solar panel normals */
    cosangle1 = dot(sunpos, panel1);
    cosangle2 = dot(sunpos, panel2);
    cosangle3 = dot(sunpos, panel3);

    /* calculate amount of solar power generation */
    /* power depends on square area and cos of incident light angle */
    if (cosangle1 > 0) power1 = POWER_GAIN * cosangle1 * PANEL_SIZE;
    else power1 = 0.0; /* sun not hitting panel */
    if (cosangle2 > 0) power2 = POWER_GAIN * cosangle2 * PANEL_SIZE;
    else power2 = 0.0;  /* sun not hitting panel */
    if (cosangle3 > 0) power3 = POWER_GAIN * cosangle3 * PANEL_SIZE;
    else power3 = 0.0; /* sun not hitting panel */
  }

  /* we want to record DGPS position of robot, power gained by each solar
     panel, position of sun relative to robot, date/time */
  if (strcmp(RecordSun, "none") != 0) {
    Sunpos = fopen(RecordSun, "a");
    /* robot.x, robot.y, robot.z, robot.yaw, power1, power2, altitude, azimuth, Jdate */
    fprintf(Sunpos, "%.3f %.3f %.3f %.3f %.3f %.3f %f %f %f %f\n", robot.x, robot.y, robot.z, robot.yaw, power1, power2, altitude, azimuth, Jdate, power3);
    fclose(Sunpos);
  }

  return(power1 + power2);
}


