/* FILE: sensorMain.cpp
   AUTHOR: Michael Wagner
   CREATED: Apr 15, 1999
   DESCRIPTION: This file links a sensor device driver to NDDS, 
     allowing communication with it. To compile, a
     sensor object file archive must first be created, and this
     archive must be included in the Makefile. Secondly, the file
     sensorMain.h must be edited to reflect the sensor's name and
     constructor method.
*/   

#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include "ndds/NDDS.h"
#include "currentSensor.h"
#include "currentSensorDef.h"
#include "sensorDef.h"
#include "sensorMsgs.h"
#include "SAS_ServiceNames.h"
#include "sensorMain.h"
#include "dbRecord.h"
#include "nddsMsgNames.h"
#include "SAS_Config.h"

void registerSensor(sensor *s) {
  sensorReg *msg = (sensorReg *)calloc(1, sizeof(sensorReg));
  sensorRegNddsRegister();
  NDDSPublication registerPub = NddsPublicationCreate(SENSOR_REG_MSG_NAME, sensorRegPublicationType,
						      msg, SENSOR_REG_PERSISTANCE, SENSOR_REG_STRENGTH);
  msg->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  strcpy(msg->sensorID, s->sensorID);
  msg->status = s->sensorStatus;
  NddsPublicationSend(registerPub);
}


/* Returns RTI_TRUE when a reply is to be sent, 
   RTI_FALSE if not. */
RTIBool ServerRoutine(NDDSObjectInstance reply, 
		      NDDSObjectInstance request,
		      void *userData)
{
  sensorReply   *itemReply   = (sensorReply *)reply;
  sensorRequest *itemRequest = (sensorRequest *)request;
  SENSOR_NAME   *sensor      = (SENSOR_NAME *)userData;
  
  /* Use the request here */

  switch(itemRequest->function) {
  case SENSOR_STARTUP_FUNCTION:
    cerr << "[" << SENSOR_NAME_STR << "] *** Intiating startup..." << endl;
    memcpy(itemReply, sensor->startup(itemRequest), sizeof(sensorReply));
    cerr << "[" << SENSOR_NAME_STR << "] *** done!" << endl;
    break;
  case SENSOR_CALIBRATE_FUNCTION:
    cerr << "[" << SENSOR_NAME_STR << "] *** Intiating calibrate..." << endl;
    memcpy(itemReply, sensor->calibration(itemRequest), sizeof(sensorReply));
    cerr << "[" << SENSOR_NAME_STR << "] *** done!" << endl;
    break;
  case SENSOR_DEPLOY_FUNCTION:
    cerr << "[" << SENSOR_NAME_STR << "] *** Intiating deploy..." << endl;
    memcpy(itemReply, sensor->deployment(itemRequest), sizeof(sensorReply));
    cerr << "[" << SENSOR_NAME_STR << "] *** done!" << endl;
    break;
  case SENSOR_UNDEPLOY_FUNCTION:
    cerr << "[" << SENSOR_NAME_STR << "] *** Intiating undeploy..." << endl;
    memcpy(itemReply, sensor->unDeployment(itemRequest), sizeof(sensorReply));
    cerr << "[" << SENSOR_NAME_STR << "] *** done!" << endl;
    break;
  case SENSOR_ACQUIRE_DATA_FUNCTION:
  case SENSOR_TARGET_ACQ_FUNCTION:
    cerr << "[" << SENSOR_NAME_STR << "] *** Intiating DAQ..." << endl;
    memcpy(itemReply, sensor->dataAcquisition(itemRequest), sizeof(sensorReply));
    cerr << "[" << SENSOR_NAME_STR << "] *** done!" << endl;
    break;
  case SENSOR_SHUTDOWN_FUNCTION:
    cerr << "[" << SENSOR_NAME_STR << "] *** Intiating shutdown..." << endl;
    memcpy(itemReply, sensor->shutdown(itemRequest), sizeof(sensorReply));
    cerr << "[" << SENSOR_NAME_STR << "] *** done!" << endl;
    break;
  case SENSOR_DIAGNOSTIC_FUNCTION:
    cerr << "[" << SENSOR_NAME_STR << "] *** Intiating diagnostic..." << endl;
    memcpy(itemReply, sensor->diagnostic(itemRequest), sizeof(sensorReply));
    cerr << "[" << SENSOR_NAME_STR << "] *** done!" << endl;
    break;
  case SENSOR_DEBUG_MODE_FUNCTION:
    cerr << "[" << SENSOR_NAME_STR << "] *** Debug mode function called..." << endl;
    memcpy(itemReply, sensor->toggleDebug(itemRequest), sizeof(sensorReply));
    cerr << "[" << SENSOR_NAME_STR << "] *** Done!" << endl;
    break;
  default:
    return RTI_FALSE;
  }

  return RTI_TRUE; 
}


void serverMain(SENSOR_NAME *sensor) {
  float         strength     = 1.0f;
  double        sleepTime    = 10.0f;
  NDDSServer    server       = NULL;
  sensorReply   *itemReply   = NULL;
  sensorRequest *itemRequest = NULL;
  
  sensorReplyNddsRegister();
  sensorRequestNddsRegister();
  
  itemReply   = (sensorReply *)calloc(1, sizeof(sensorReply));
  itemRequest = (sensorRequest *)calloc(1, sizeof(sensorRequest));
  
  NddsServerCreate(sensor->sensorID, NDDS_SERVER_IMMEDIATE, strength, 
		   itemReply, itemRequest, 
		   sensorReplyPublicationType, 
		   sensorRequestPublicationType, 
		   ServerRoutine, (void *)sensor);
  
  while (1) {
    NddsUtilitySleep(sleepTime);
  }
}

int main(int argc, char *argv[])
{
  if (argc != 2) {
    cerr << "Usage: [...]Sensor sensorID\n";
    return(-1);
  }

  SENSOR_NAME *sensor = new SENSOR_CONSTRUCTOR(argv[1]);

  int nddsDomain;
  if(!getSAS_Config(NDDS_DOMAIN, nddsDomain)) {
    cerr << "[" << argv[1] << "] ERROR: Could not read " << NDDS_DOMAIN << " from config file!" << endl;
    return(-1);
  }

  NddsInit(nddsDomain, NULL);
  
  cerr << "[" << argv[1] << "] Initialized NDDS" << endl;

  /* Startup sensor */
  sensorReply *startupReply = sensor->startup((sensorRequest *)calloc(1, sizeof(sensorRequest)));

  /* Initiate sensor registration stage */
  cerr << "[" << argv[1] << "] Publishing registration message...";
  registerSensor(sensor);
  cerr << "done!" << endl;

  /* Initiate server stage */
  cerr << "[" << argv[1] << "] Initiating server stage" << endl;
  serverMain(sensor);

  /* serverMain should not return! */
  cerr << "[" << argv[1] << "] serverMain should not return!" << endl;
  return(-1);
}

