/* FILE:    hiResSensor.h
   AUTHOR:  Michael Wagner
   CREATED: Apr 6, 1999

   DESCRIPTION: This file defines the high-resolution
     camera sensor driver class, hiResSensor. This class
     is inhereted from the virtual class sensor.h
*/

#ifndef __HI_RES_SENSOR_H__
#define __HI_RES_SENSOR_H__

#include "sensor.h"
#include "hiResPtu.h"
#include "lensController.h"
#include "frameGrabber.h"
#include "ppm.h"
#include "hiResSensorDef.h"
#include "telemetryMsgs.h"
#include "hiResSensorTelemetryThread.h"
#include "ndds/NDDS.h"

#define HI_RES_DEBUG_IMAGE_FILE "/usr/nomad/data/hiResImage.ppm"

class hiResSensor : public sensor {
 public:
  hiResSensor(char *_sensorID,
	      char *_DAQ_SaveFileNameBase,
	      frameGrabberConfig *fgConfig,
	      dgpsVector *ptuPosition);               /* The constructor should simply instantiate 
							 objects and set class variables. */
  ~hiResSensor();                                     /* The destructor should delete objects in 
							 the sensor driver class. Deleting the 
							 objects in this class should automatically 
							 call their respective uninit functions, to 
							 close serial ports and such. */
  
  // Implementing the sensor interface.
  sensorReply *startup(sensorRequest *req);         /* The startup function should initialize all 
						       subcomponents, and prepare the sensor for deployment. */
  sensorReply *calibration(sensorRequest *req);     /* The calibration function should be used to update 
						       internal state variables with correct values. */
  sensorReply *deployment(sensorRequest *req);      /* The deployment function should deploy the sensor to the 
						       target coordinates (request->targetPosition), given a 
						       current robot location and pose (request->robotPosition). */ 
  sensorReply *unDeployment(sensorRequest *req);    /* The unDeployment function should servo the sensor back 
						       to its startup state. */
  sensorReply *dataAcquisition(sensorRequest *req); /* The dataAcqusition function should simply capture data, 
						       calculate the appropriate file name in which to save this 
						       new data, and save it. */
  sensorReply *shutdown(sensorRequest *req);        /* The shutdown function should uninitialize the sensor but 
						       not delete any state information or internal variables. */ 
  sensorReply *diagnostic(sensorRequest *req);      /* The diagnostic function should run the sensor through a
						       number of tests to search for errors, and return the most
						       important error to the user. */
  sensorReply *toggleDebug(sensorRequest *req);     /* Switches sensor to debug mode. */

  sensorReply *getCost(sensorRequest *req);         /* Returns the sensor's costs */
  sensorReply *getWorkspace(sensorRequest *req);    /* Returns the sensor's workspace */
  sensorReply *getStatus(sensorRequest *req);       /* Just returns data in the status and extendedStatus fields of the reply, doesn't
						       take any action */


  // TBD: This is an ugly hack -- autoFocus should not be public, but it has to be so teleopCallback can 
  // call it, since I don't feel comfortable making an NDDS service routine also a class member function.
  int autoFocus();

 private:
  int ptuSerialPortNum, lensSerialPortNum;
  char *DAQ_SaveFileNameBase; 
  hiResPtu *ptu;
  lensController *lens;
  frameGrabber *fg;
  int debugMode;

  int autoFocus(frameGrabber *f, lensController *l); /* Returns 1 on success, 0 on failure */
  long calcAutoFocusScore(unsigned char *image, frameGrabber *fg);  /* Assumes higher score is better! */

  hiResSensorTelemetryThread *telThread;
};

RTIBool teleopCallback(NDDSObjectInstance reply, NDDSObjectInstance request, void *userData);

typedef struct {
  hiResSensor *sensor;
  lensController *lens;
  hiResPtu *ptu;
  frameGrabber *fg;
} teleopCallbackParam;



#endif /* __HI_RES_SENSOR_H__ */
