/* FILE:    sensor.h
   AUTHOR:  Michael Wagner
   CREATED: Apr 6, 1999

   DESCRIPTION: This file defines the virtual class
     sensor, to be extended by any sensor
     device driver.
*/

#ifndef __SENSOR_H__
#define __SENSOR_H__

#include "sensorMsgs.h"
#include "sensorDef.h"

class sensor {
 public:
  sensor() {};		                        /* The constructor should simply instantiate 
						   objects and set class variables. */
  virtual ~sensor() {};                           /* The destructor should delete objects in 
						     the sensor class. Deleting the 
						     objects in this class should automatically 
						     call their respective uninit functions, to 
						     close serial ports and such. */

  // The sensor interface.
  virtual sensorReply *startup(sensorRequest *req) = 0;         /* The startup function should initialize all 
								   subcomponents, and prepare the sensor for deployment. */
  virtual sensorReply *calibration(sensorRequest *req) = 0;     /* The calibration function should be used to update 
								   internal state variables with correct values. */
  virtual sensorReply *deployment(sensorRequest *req) = 0;      /* The deployment function should deploy the sensor to the 
								   target coordinates (request->targetPosition), given a 
								   current robot location and pose 
								   (request->robotPosition). */ 
  virtual sensorReply *unDeployment(sensorRequest *req) = 0;    /* The unDeployment function should servo the sensor back 
								   to its startup state. */
  virtual sensorReply *dataAcquisition(sensorRequest *req) = 0; /* The dataAcqusition function should simply capture data, 
								   calculate the appropriate file name in which to save this 
								   new data, and save it. */
  virtual sensorReply *shutdown(sensorRequest *req) = 0;        /* The shutdown function should uninitialize the sensor but 
								   not delete any state information or internal variables. */ 
  virtual sensorReply *diagnostic(sensorRequest *req) = 0;      /* The diagnostic function should run the sensor through a
								   number of tests to search for errors, and return the most
								   important error to the user. */
  virtual sensorReply *toggleDebug(sensorRequest *req) = 0;     /* Switches sensor to debug mode. */
  
  virtual sensorReply *getCost(sensorRequest *req) = 0;         /* Returns the sensor's costs */
  virtual sensorReply *getWorkspace(sensorRequest *req) = 0;    /* Returns the sensor's workspace */
  virtual sensorReply *getStatus(sensorRequest *req) = 0;       /* Just returns data in the status and extendedStatus fields of the reply, doesn't
								   take any action */

  char  sensorID[SENSOR_MAX_ID_LENGTH];
  int   sensorStatus; 
  int   sensorExtendedStatus;
};

#endif /* __SENSOR_H__ */


