/* FILE:    metalDetectorSensor.cpp
   AUTHOR:  Michael Wagner
   CREATED: Aug 26, 1999

   DESCRIPTION: This file implements the metal detector 
     sensor device driver class, metalDetectorSensor. This 
     class is inhereted from the virtual class sensor.h
*/

#include <iostream.h>
#include <fstream.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include "ndds/NDDS.h"
#include "metalDetectorSensor.h"
#include "metalDetectorSensorDef.h"
#include "daqCard.h"
#include "sensorDef.h"
#include "SAS_Config.h"

// Public functions

/* The constructor should simply instantiate objects and set class variables. */
metalDetectorSensor::metalDetectorSensor(char *sensorID, char *DAQ_SaveFileNameBase, float daqPeriod) : sensor() {
  this->DAQ_SaveFileNameBase = new char[strlen(DAQ_SaveFileNameBase)+1];
  strcpy(this->DAQ_SaveFileNameBase, DAQ_SaveFileNameBase);
  strcpy(this->sensorID, sensorID);
  this->daqPeriod = daqPeriod;
  debugMode = 0;
}

/* The destructor should delete objects in the sensor driver class. Deleting the objects in this class should 
   automatically call their respective uninit functions, to close serial ports and such. */
metalDetectorSensor::~metalDetectorSensor() {
  delete DAQ_SaveFileNameBase;
}

/* The startup function should initialize all subcomponents, and prepare the sensor for deployment. */
sensorReply *metalDetectorSensor::startup(sensorRequest *request) {
  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  strcpy(reply->sensorID, sensorID);
  reply->function = request->function;
  if(daq.open() == DAQ_CARD_OK) {
    if(daq.setChannel(METAL_DETECTOR_DAQ_CHANNEL)) {
      cerr << "[metalDetectorSensor] Successfully started metal detector!" << endl;
      reply->status = SENSOR_OK;
    } else {
      reply->status = SENSOR_MALFUNCTION_STATUS;
    }
  } else {
    cerr << "[metalDetectorSensor] ERROR: Metal detector could not startup!" << endl;
    reply->status = SENSOR_MALFUNCTION_STATUS;
  }
  return(reply);
}

/* The calibration function should be used to update internal state variables with correct values. */
sensorReply *metalDetectorSensor::calibration(sensorRequest *request) {
  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  strcpy(reply->sensorID, sensorID);
  reply->function = request->function;
  reply->status = SENSOR_OK;
  return(reply);
}

/* The deployment function should deploy the sensor to the target coordinates (request->targetPosition), 
   given a current robot location and pose (request->robotPosition). */ 
sensorReply *metalDetectorSensor::deployment(sensorRequest *request) {
  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  strcpy(reply->sensorID, sensorID);
  reply->function = request->function;
  reply->status = SENSOR_OK;
  return(reply);
}

/* The unDeployment function should servo the sensor back to its startup state. */
sensorReply *metalDetectorSensor::unDeployment(sensorRequest *request) {
  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  strcpy(reply->sensorID, sensorID);
  reply->function = request->function;
  reply->status = SENSOR_OK;
  return(reply);
}

/* The dataAcquisition function should simply capture data, calculate the appropriate file name 
   in which to save this new data, and save it. */
sensorReply *metalDetectorSensor::dataAcquisition(sensorRequest *request) {
  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  strcpy(reply->sensorID, sensorID);
  reply->function = request->function;
  
  char saveFileName[SENSOR_MAX_FILENAME_LENGTH];
  int retVal;

  time_t ltime;
  time(&ltime);
  sprintf(saveFileName, "%s%d.metal", DAQ_SaveFileNameBase, (long)ltime);
  retVal = daq.read(currentReading, METAL_DETECTOR_NUM_READINGS, daqPeriod, currentTimeBuff);
  if(retVal == DAQ_CARD_OK) {
    if(writeFile(saveFileName, currentReading)) {
      reply->status = SENSOR_OK;
    } else {
      reply->status = SENSOR_DAQ_FAILURE_STATUS;
    }
  } else {
    reply->status = SENSOR_MALFUNCTION_STATUS; 
  }

  return(reply);
}

/* The shutdown function should uninitialize the sensor but not delete any state information or
   internal variables. */ 
sensorReply *metalDetectorSensor::shutdown(sensorRequest *request) {
  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  strcpy(reply->sensorID, sensorID);
  reply->function = request->function;
  daq.close();
  reply->status = SENSOR_OK;
  return(reply);
}

/* The diagnostic function should run the sensor through a number of tests to search for errors, and
   return the most important error to the user. */
sensorReply *metalDetectorSensor::diagnostic(sensorRequest *request) {
  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  strcpy(reply->sensorID, sensorID);
  reply->function = request->function;
  reply->status = SENSOR_OK;
  return(reply);
}

/* Toggles sensor to/from debug mode. */
sensorReply *metalDetectorSensor::toggleDebug(sensorRequest *request) {
  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  strcpy(reply->sensorID, sensorID);
  reply->function = request->function;

  debugMode = (debugMode == 0) ? 1 : 0;

  if(debugMode) {
    cerr << "[metalDetectorSensor] Switching to debug mode.\n";
  } else {
    cerr << "[metalDetectorSensor] Switching out of debug mode.\n";
  }

  reply->status = SENSOR_OK;
  return(reply);
}

// Private methods

int metalDetectorSensor::writeFile(char *filename, float *dataBuff) {
  if(dataBuff == NULL || filename == NULL) {
    return(0);
  }

  fstream outfile;
  outfile.open(filename, ios::out);

  if(outfile.fail()) {
    cerr << "[metalDetectorSensor] ERROR: Cannot open " << filename << " for writing!" << endl;
    return(0);
  }
  
  for(int i=0; i < METAL_DETECTOR_NUM_READINGS; i++) {
    outfile << dataBuff[i] << endl;
  }

  outfile.close();

  return(1);
}

