/* FILE:    hiResPtu.h
   AUTHOR:  Michael Wagner
   CREATED: Apr 7. 1999

   DESCRIPTION: Defines the hiResPtu class, which encapsulates the 
     functionality of the Directed Perception PTU-46-70, on which 
     the high-res camera is mounted. 

     NOTE: This code assumes that the y-axis is pointing straight
     out from the front of the PTU.
*/

#ifndef __HI_RES_PTU_H__
#define __HI_RES_PTU_H__

#include "ptu.h"
#include "dgpsVector.h"
#include "ptuAngle.h"
#include "hiResSensorDef.h"
#include "dbRecord.h" /* for pose */

#define PTU_SPEED 500 /* pos / s */

// Definitions to augment the list in ptu.h
#define PTU_NOT_INITIALIZED (-2)   /* Returned when a command is made and 
				      the PTU hasn't been initialized. */
#define PTU_POSITION_MISMATCH (-3) /* Returned by servo commands when PTU is 
				      otherwise ok but resulting hardware 
				      angle doesn't match the commanded 
				      angle. */

class hiResPtu {
 public:
  hiResPtu(dgpsVector *ptuPosition);
  ~hiResPtu();

  int init(int serialPortNum);  /* Opens specified serial port, resets PTU.
				   Returns PTU_OK on success, status on 
				   failure. */
  void uninit();                /* Closes communications with serial port. */
  int reset();                  /* Returns status */
  int stop();                   /* Halts PTU until start() is called. Returns 
				   status of PTU halt command. */
  int start();                  /* Starts PTU after stop() is called. Returns 
				   1 if successful, 0 if not. */
  int rehome();                 /* Rehome PTU to straight ahead */
  ptuAngle *getLastCmdAngle();   /* Returns the last commanded angle. */
  ptuAngle *getHardwareAngle();  /* Returns the current hardware angle. */

  int servoToAngleAbs(ptuAngle *newAngle);            /* Servo to a new absolute 
							 angle, return status */
  int servoToAngleRel(ptuAngle *newAngle);            /* Servo to a new relative
							 angle, return status */
  int servoToVector(dgpsVector *current, dgpsVector *target, pose *currPose);
		                                      /* Servo to a DGPS
							 position, return
							 status. 
							 NOTE: roll, pitch, yaw must be in radians
							 NOTE: Positive pitch is CCW in the +x direction
							 NOTE: Positive roll is CCW in the +y direction
							 NOTE: Positive yaw is CCW in the +z direction
							 NOTE: This assumes that the y-axis is 
							 straight out the front of the PTU.
						      */

 private:
  int servoToAngle(int absolute, ptuAngle *newAngle); /* Called by both servoToAngle*
							 public functions. Absolute is
						         1 (true) or 0 (false). */

  ptuAngle *vectorToAngle(dgpsVector *target,
			  dgpsVector *current);  /* NOTE: This assumes that the y-axis is 
						straight out the front of the PTU. */ 
  ptuAngle *vectorToAngle(dgpsVector *target); /* NOTE: Assumes current vector is <0,0,0>, and that
					      the y-axis is straight out the front of
					      the PTU. */
  ptuAngle *lastCmdPtuAngle;
  double tiltRes, panRes;  // in radian / position
  int initOccurred;        // 1 if true, 0 if false
  int stopped;             // 1 if true, 0 if false
  portstream_fd serialStream;
  dgpsVector *ptuPosition; // This is a vector from the robot's DGPS sensor to the center of the PTU.
};  

#endif  // __HI_RES_PTU_H__






