/* FILE:    hiResPtu.cpp
   AUTHOR:  Michael Wagner
   CREATED: Apr 7, 1999

   DESCRIPTION: Implements the hiResPtu class, which encapsulates the 
     functionality of the Directed Perception PTU-46-70, on which 
     the high-res camera is mounted. 
*/

#include <iostream.h>
#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include <stdlib.h>
#include "ptu.h"
#include "hiResPtu.h"
#include "telemetryMsgs.h"

// The actual telemetry data structure.
//extern hiResDerivedState *p_hiResDerived;
static hiResDerivedState hiResDerived;
static hiResDerivedState *p_hiResDerived = &hiResDerived;

// Public functions

hiResPtu::hiResPtu(dgpsVector *ptuPosition) {
  this->ptuPosition = ptuPosition;
  lastCmdPtuAngle = new ptuAngle(0.0, 0.0);
  panRes = 0;
  tiltRes = 0;
  initOccurred = 0;
  stopped = 0;
  p_hiResDerived->pan = lastCmdPtuAngle->pan;
  p_hiResDerived->tilt = lastCmdPtuAngle->tilt;
#ifdef NO_PTU 
  cerr << "[hiResPtu] Starting up without PTU capability..." << endl;
#endif
}

hiResPtu::~hiResPtu() {
  if(initOccurred) {
    rehome();
    uninit();
    p_hiResDerived->pan = 0.0;
    p_hiResDerived->tilt = 0.0;
  }
  delete lastCmdPtuAngle;
}

int hiResPtu::init(int serialPortNum) {
#ifndef NO_PTU
  char serialPortName[256];
  int retVal;

#ifdef _UNIX
  sprintf(serialPortName, "/dev/cua%d", serialPortNum);
#else
  sprintf(serialPortName, "COM%d", serialPortNum);
#endif

  // Initialize serial port
  serialStream = open_host_port(serialPortName);
  
  if(serialStream == PORT_NOT_OPENED) {
    return(PORT_NOT_OPENED);
  }

  //  set_mode(DEFAULTS, RESTORE_FACTORY_SETTINGS);
  set_mode(DEFAULTS, RESTORE_SAVED_SETTINGS);

  retVal = reset_PTU_parser(5000);
  if(retVal != PTU_OK) {
    return(retVal);
  }

  int val = PTU_REG_POWER;
  set_desired(PAN, MOVE_POWER_LEVEL, (PTU_PARM_PTR *) &val, ABSOLUTE);
  if(val != get_desired(PAN, MOVE_POWER_LEVEL)) {
    cerr << "[hiResPtu] ERROR: Cannot set pan move power level!" << endl;
    return(PTU_NOT_INITIALIZED);
  }
  set_desired(TILT, MOVE_POWER_LEVEL, (PTU_PARM_PTR *) &val, ABSOLUTE);
  if(val != get_desired(TILT, MOVE_POWER_LEVEL)) {
    cerr << "[hiResPtu] ERROR: Cannot set tilt move power level!" << endl;
    return(PTU_NOT_INITIALIZED);
  }
  val = PTU_SPEED;
  set_desired(PAN, SPEED, (PTU_PARM_PTR *) &val, ABSOLUTE);
  if(val != get_desired(PAN, SPEED)) {
    cerr << "[hiResPtu] ERROR: Cannot set pan speed to " << PTU_SPEED << " pos/s!" << endl;
    return(PTU_NOT_INITIALIZED);
  }
  set_desired(TILT, SPEED, (PTU_PARM_PTR *) &val, ABSOLUTE);
  if(val != get_desired(TILT, SPEED)) {
    cerr << "[hiResPtu] ERROR: Cannot set tilt speed to " << PTU_SPEED << " pos/s!" << endl;
    return(PTU_NOT_INITIALIZED);
  }

  panRes = get_desired(PAN, RESOLUTION);   // Both resolutions in machine units
  tiltRes = get_desired(TILT, RESOLUTION);

  // Now convert to rad / positions
  panRes = (panRes * PI) / (3600 * 60 * 180);
  tiltRes = (tiltRes * PI) / (3600 * 60 * 180);

  cerr << "[hiResPtu] Pan resolution = " << panRes << ", tilt resolution = " << tiltRes << endl;

  initOccurred = 1;

#endif /* NO_PTU */
  
  return(PTU_OK);
}

void hiResPtu::uninit() {
#ifndef NO_PTU
  initOccurred = 0;
  close_host_port(serialStream);
#endif
}

int hiResPtu::reset() {
#ifndef NO_PTU
  if(!initOccurred) {
    return(PTU_NOT_INITIALIZED);
  } else {
    return(reset_ptu());
  }
#else
  return(PTU_OK);
#endif
}

int hiResPtu::stop() {
#ifndef NO_PTU
  if(!initOccurred) {
    return(PTU_NOT_INITIALIZED);
  } else {
    int retVal = halt(ALL);
    
    if(retVal == TRUE) {
      stopped = 1;
    }
    return(retVal);
  }
#else
  return(PTU_OK);
#endif
}

int hiResPtu::start() {
#ifndef NO_PTU
  if(!initOccurred) {
    return(PTU_NOT_INITIALIZED); 
  } else {
    if(!stopped) {
      return(0);
    } else {
      stopped = 0;
      return(1);
    }
  }
#else
  return(PTU_OK);
#endif
}
  
int hiResPtu::rehome() {
#ifndef NO_PTU
  if(!initOccurred) {
    return(PTU_NOT_INITIALIZED);
  } else {
    return(servoToAngleAbs(new ptuAngle(0.0, 0.0)));
  }
#else
  return(PTU_OK);
#endif
}

ptuAngle *hiResPtu::getLastCmdAngle() {
  return(lastCmdPtuAngle);
}

ptuAngle *hiResPtu::getHardwareAngle() {
#ifndef NO_PTU
  if(!initOccurred) {
    return(NULL);
  } else {
    long panPos, tiltPos;
    
    panPos = get_current(PAN, POSITION);
    tiltPos = get_current(TILT, POSITION);
    
    return(new ptuAngle((double)(panPos*panRes), (double)(tiltPos*tiltRes)));
  }
#else
  return(new ptuAngle(0.0, 0.0));
#endif
}

int hiResPtu::servoToAngleAbs(ptuAngle *newAngle) {
  return(servoToAngle(1, newAngle));
}

int hiResPtu::servoToAngleRel(ptuAngle *newAngle) {
  return(servoToAngle(0, newAngle));
}

int hiResPtu::servoToVector(dgpsVector *currPosition, dgpsVector *target, pose *currPose) {
#ifndef NO_PTU

  dgpsVector *result = new dgpsVector();
  
  cerr << "currPosition = " << currPosition->x << ", " << currPosition->y << ", " << currPosition->z << endl;
  cerr << "target = " << target->x << ", " << target->y << ", " << target->z << endl;
  cerr << "pose = " << currPose->roll << ", " << currPose->pitch << ", " << currPose->yaw << endl;

  result->x = target->x - currPosition->x;
  result->y = target->y - currPosition->y;
  result->z = target->z - currPosition->z;

  cerr << "Target in DGPS coordinates without pose: " << result->x << ", " << result->y << ", " << result->z << endl;
  
  result->rotate(-currPose->roll, -currPose->pitch, -currPose->yaw);

  cerr << "Target in DGPS coordinates with pose: " << result->x << ", " << result->y << ", " << result->z << endl;

  result->x -= PTU_X + CCD_X;
  result->y -= PTU_Y + CCD_Y;
  result->z -= PTU_Z - DGPS_SENSOR_Z + CCD_Z;

  cout << "Target in PTU coordinates with pose: " << result->x << ", " << result->y << ", " << result->z << endl;
  
  // alpha is the angle from the PTU axis of tilt rotation to the target. 
  ptuAngle *alpha = vectorToAngle(result);
  
  delete(result);

  // Now we must account for the fact that the CCD is not at the axis of tilt rotation by 
  // computing the ADDITIONAL angle, beta. The CCD platform and the ray from the PTU tilt axis
  // to the target make a leg and hypotenuse of a right triangle. The angle across from the
  // CCD platform leg is beta.
  double beta_tilt = asin(CCD_Z / sqrt(target->x * target->x + target->y * target->y + target->z * target->z));

  cout << "beta_tilt = " << beta_tilt << endl;

  return(servoToAngleAbs(new ptuAngle(alpha->pan, alpha->tilt + beta_tilt)));

#else
  
  return(PTU_OK);

#endif /* NO_PTU */
}


// Private functions

int hiResPtu::servoToAngle(int absolute, ptuAngle *newAngle) {
#ifndef NO_PTU

  if(!initOccurred) {
    return(PTU_NOT_INITIALIZED);
  } else {
    int retVal, panRetVal, tiltRetVal;
    long cmdPanPos, cmdTiltPos;
    long hardwarePanPos, hardwareTiltPos;
    
    cmdPanPos = (long)(newAngle->pan / panRes); /* positions */
    cmdTiltPos = (long)(newAngle->tilt / tiltRes); /* positions */

    cerr << "Angles: " << newAngle->pan << ", " << newAngle->tilt << endl;
    cerr << "Positions: " << cmdPanPos << ", " << cmdTiltPos << endl;

    if(absolute) {
      panRetVal = set_desired(PAN, POSITION, (PTU_PARM_PTR *) &cmdPanPos, 
			      ABSOLUTE);
      tiltRetVal = set_desired(TILT, POSITION, (PTU_PARM_PTR *) &cmdTiltPos, 
			       ABSOLUTE);
    } else {
      panRetVal = set_desired(PAN, POSITION, (PTU_PARM_PTR *) &cmdPanPos, 
			      RELATIVE);
      tiltRetVal = set_desired(TILT, POSITION, (PTU_PARM_PTR *) &cmdTiltPos, 
			       RELATIVE);
    }

    if((retVal = await_completion()) != PTU_OK) {
      	cerr << "[hiResPtu] ERROR: Failed waiting to complete PTU movement!" << endl;
      	return(retVal);
    } 
      
    hardwarePanPos = get_current(PAN, POSITION); /* positions */
    hardwareTiltPos = get_current(TILT, POSITION); /* positions */

    // Update telemetry
    p_hiResDerived->pan = hardwarePanPos * panRes; /* radians */
    p_hiResDerived->tilt = hardwareTiltPos * tiltRes; /* radians */
    
    if(panRetVal == tiltRetVal == PTU_OK) {
      lastCmdPtuAngle->copy(newAngle);
      
      if((hardwarePanPos != cmdPanPos) || (hardwareTiltPos != cmdTiltPos)) {
	cerr << "[hiResPtu] ERROR: PTU commanded / actual position mismatch!  ";
	cerr << "cmdPan = " << cmdPanPos << ", actPan = " << hardwarePanPos << "    cmdTilt = " << cmdTiltPos << ", actTilt = " << hardwareTiltPos << endl;
	return(PTU_POSITION_MISMATCH);
      }
	
    } else {
      return(tiltRetVal); // since it's the last returned value.
    }
  }
#else
  return(PTU_OK);
#endif /* NO_PTU */
}

ptuAngle *hiResPtu::vectorToAngle(dgpsVector *target, dgpsVector *current) {
  double r, rho;
  dgpsVector *normTarget = new dgpsVector(target->x - current->x,
				  target->y - current->y,
				  target->z - current->z);
  ptuAngle *newAngle = new ptuAngle(0.0, 0.0);

  r   = sqrt(normTarget->x * normTarget->x + normTarget->y * normTarget->y);
  rho = sqrt(r * r + normTarget->z * normTarget->z);
  
  newAngle->pan = acos(normTarget->x / r) - PI / 2; // x-axis is to the right of the PTU.
  newAngle->tilt = acos(r / rho);

  if(normTarget->y < 0) {
    newAngle->pan = -newAngle->pan;
  }
  if(normTarget->z < 0) {
    newAngle->tilt = -newAngle->tilt;
  }
  
  return(newAngle);
}

ptuAngle *hiResPtu::vectorToAngle(dgpsVector *target) {
  return(vectorToAngle(target, new dgpsVector(0.0, 0.0, 0.0)));
}

