/* FILE:    lensController.cpp
   AUTHOR:  Michael Wagner
   CREATED: Apr 8, 1999

   DESCRIPTION: Implements the lensController class, which is used
     to use Mike Montemerlo's Motorola HC16... microcontroller
     over a serial port. This device controls a Fujinon 
     H14x10.5B-Y41 lens.
*/

#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "lensController.h"
#include "ndds/NDDS.h"
#include <fstream.h>
#include "fujinonFOV.h"
#include "telemetryMsgs.h"

// The actual telemetry data structure.
//extern hiResDerivedState *p_hiResDerived;
static hiResDerivedState hiResDerived;
static hiResDerivedState *p_hiResDerived = &hiResDerived;

lensController::lensController() {
  initOccurred = 0;
  lastCmdZoom = lastCmdFocus = 0;
  p_hiResDerived->FOV = 0.0;
  p_hiResDerived->focalLength = 0.0;
#ifdef NO_LENS
  cerr << "[lensController] Starting up without lens capability..." << endl;
#endif
}

lensController::~lensController() {
  if(initOccurred) {
    uninit();
    p_hiResDerived->FOV = 0.0;
    p_hiResDerived->focalLength = 0.0;
  }
}

int lensController::init(int serialPortNum) {
#ifndef NO_LENS

#ifdef _UNIX
  sprintf(portName, "/dev/cua%d", serialPortNum);
#else
  sprintf(portName, "COM%d", serialPortNum);
#endif

  // Initialize serial port and streams
  serialStream = openserial(portName);

  if(serialStream == PORT_NOT_OPENED) {
    initOccurred = 0;
    return(LENS_CONTROLLER_NOT_INITIALIZED);
  } else {
    initOccurred = 1;
    lastCmdZoom = getHardwareZoom();
    lastCmdFocus = getHardwareFocus();
    return(LENS_OK);
  }
#else
  return(LENS_OK);
#endif /* NO_LENS */
}

void lensController::uninit() {
#ifndef NO_LENS
  closeserial(serialStream);
  initOccurred = 0;
#endif
}

int lensController::setZoomAbs(unsigned char zoomVal) {
#ifndef NO_LENS
  if(!initOccurred) {
    return(LENS_CONTROLLER_NOT_INITIALIZED);
  } else {    
    char buff[5];
    sprintf(buff, "z%d\n", zoomVal);
    fstream serialOut;
    serialOut.open(portName, ios::out);
    serialOut << buff << endl;
    serialOut.close();
    lastCmdZoom = zoomVal;
    
    // Update telemetry
    p_hiResDerived->FOV = fieldOfView(lastCmdZoom);
    p_hiResDerived->focalLength = focalLength(lastCmdZoom);
    return(LENS_OK);
  }
#else
  return(LENS_OK);
#endif
}

int lensController::setZoomAbsBlock(unsigned char zoomVal) {
  int retVal = setZoomAbs(zoomVal);
  if(retVal == LENS_OK) {
    return(blockOnZoom(zoomVal));
  } else {
    return(retVal);
  }
}

int lensController::setZoomRel(char zoomVal) { 
#ifndef NO_LENS
  if(!initOccurred) {
    return(LENS_CONTROLLER_NOT_INITIALIZED);
  } else {
    if((lastCmdZoom + zoomVal > ZOOM_OUT_FULL) || (lastCmdZoom + zoomVal < ZOOM_IN_FULL)) {
      return(LENS_CONTROLLER_CMD_OUT_OF_BOUNDS);
    } else {
      return(setZoomAbs(lastCmdZoom + zoomVal));
    }
  }
#else
  return(LENS_OK);
#endif
}

int lensController::setZoomRelBlock(char zoomVal) {
#ifndef NO_LENS
  if(!initOccurred) {
    return(LENS_CONTROLLER_NOT_INITIALIZED);
  } else {
    if((lastCmdZoom + zoomVal > ZOOM_OUT_FULL) || (lastCmdZoom + zoomVal < ZOOM_IN_FULL)) {
      return(LENS_CONTROLLER_CMD_OUT_OF_BOUNDS);
    } else {
      return(setZoomAbsBlock(lastCmdZoom + zoomVal));
    }
  }
#else 
  return(LENS_OK);
#endif
}

int lensController::setFocusAbs(unsigned char focusVal) {
#ifndef NO_LENS
  if(!initOccurred) {
    return(LENS_CONTROLLER_NOT_INITIALIZED);
  } else {
    char buff[20];
    sprintf(buff, "f%d\n", focusVal);
    fstream serialOut;
    serialOut.open(portName, ios::out);
    serialOut << buff << endl;
    serialOut.close();
    lastCmdFocus = focusVal;
    return(LENS_OK);
  }
#else
  return(LENS_OK);
#endif
}

int lensController::setFocusAbsBlock(unsigned char focusVal) {
#ifndef NO_LENS
  int retVal = setFocusAbs(focusVal);
  if(retVal == LENS_OK) {
    return(blockOnFocus(focusVal));
  } else {
    return(retVal);
  }
#else
  return(LENS_OK);
#endif
}

int lensController::setFocusRel(char focusVal) {
#ifndef NO_LENS
  if(!initOccurred) {
    return(LENS_CONTROLLER_NOT_INITIALIZED);
  } else {
    cout << "lens: lastCmdFocus + focusVal = " << (int)(lastCmdFocus + focusVal) << endl;
    if((lastCmdFocus + focusVal > FOCUS_IN_FULL) || (lastCmdFocus + focusVal < FOCUS_OUT_FULL)) {
      return(LENS_CONTROLLER_CMD_OUT_OF_BOUNDS);
    } else {
      return(setFocusAbs(lastCmdFocus + focusVal));
    }
  }
#else
  return(LENS_OK);
#endif
}

int lensController::setFocusRelBlock(char focusVal) {
#ifndef NO_LENS
  if(!initOccurred) {
    return(LENS_CONTROLLER_NOT_INITIALIZED);
  } else {
    cout << "lens: lastCmdFocus + focusVal = " << (int)(lastCmdFocus + focusVal) << endl;
    if((lastCmdFocus + focusVal > FOCUS_IN_FULL) || (lastCmdFocus + focusVal < FOCUS_OUT_FULL)) {
      return(LENS_CONTROLLER_CMD_OUT_OF_BOUNDS);
    } else {
      return(setFocusAbsBlock(lastCmdFocus + focusVal));
    }
  }
#else
  return(LENS_OK);
#endif
}


int lensController::getHardwareZoom() {
#ifndef NO_LENS
  if(!initOccurred) {
    return(LENS_CONTROLLER_NOT_INITIALIZED);
  } else {
    long z, f;
    fstream serialIn;
    serialIn.open(portName, ios::in);
    serialIn >> z >> f;
    serialIn.close();    
    return(z);
  }
#else
  return(0);
#endif
}

int lensController::getHardwareFocus() {
#ifndef NO_LENS
  if(!initOccurred) {
    return(LENS_CONTROLLER_NOT_INITIALIZED);
  } else {
    int z, f;
    fstream serialIn;
    serialIn.open(portName, ios::in);
    serialIn >> z >> f;
    serialIn.close();
    return(f);
  }
#else
  return(0);
#endif
}

int lensController::getLastCmdZoom() {
#ifndef NO_LENS
  if(!initOccurred) {
    return(LENS_CONTROLLER_NOT_INITIALIZED);
  } else {
    return((int)lastCmdZoom);
  }
#else
  return(0);
#endif
}

int lensController::getLastCmdFocus() {
#ifndef NO_LENS
  if(!initOccurred) {
    return(LENS_CONTROLLER_NOT_INITIALIZED);
  } else {
    return((int)lastCmdFocus);
  }
#else
  return(0);
#endif
}

int lensController::blockOnZoom(int zoomVal) {

#ifndef NO_LENS
  NddsUtilitySleep(0.01); // Wait to make sure we're not rushing the port.

  if(zoomVal < ZOOM_IN_FULL) {
    zoomVal = ZOOM_IN_FULL;
  } else if(zoomVal > ZOOM_OUT_FULL) {
    zoomVal = ZOOM_OUT_FULL;
  }

  int z;
  for(int i=0; i < TIMEOUT_TRIES; i++) {
    z = getHardwareZoom();
    if((z <= zoomVal + LENS_ACCURACY) && 
       (z >= zoomVal - LENS_ACCURACY)) {
      return(LENS_OK);
    } else {
      setZoomAbs(zoomVal); // Command again
      NddsUtilitySleep(0.01);
    }
  }

  // TBD: Add stuff to check to make sure it's really failing...
  // like something to check if the number has changed for a while...
  // maybe it's come up against a boundary?

  return(LENS_FAIL);
#else 
  return(LENS_OK);
#endif /* NO_LENS */
}

int lensController::blockOnFocus(int focusVal) {

#ifndef NO_LENS
  NddsUtilitySleep(0.001); // Wait to make sure we're not rushing the port.

  if(focusVal < FOCUS_OUT_FULL) {
    focusVal = FOCUS_OUT_FULL;
  } else if(focusVal > FOCUS_IN_FULL) {
    focusVal = FOCUS_IN_FULL;
  }

  int f;
  for(int i=0; i < TIMEOUT_TRIES; i++) {
    f = getHardwareFocus();
    if((f <= focusVal + LENS_ACCURACY) && 
       (f >= focusVal - LENS_ACCURACY)) {
      lastCmdFocus = focusVal;
      return(LENS_OK);
    } else {
      setFocusAbs(focusVal); // Command again
      NddsUtilitySleep(0.001);
    }
  }
  return(LENS_FAIL);
#else
  return(LENS_OK);
#endif /* NO_LENS */

}


// Static functions

float focalLength(unsigned char zoomVal) {
  return(147 - ((((float)zoomVal - 39) * (147 - 10.5)) / (220 - 39))); // + 10.5
}

float fieldOfView(unsigned char zoomVal) {
  return(fujinonFOV_Table[zoomVal - FOV_TABLE_OFFSET].FOV);
}

unsigned char fovToZoomByte(float FOV) {
  if(FOV <= fujinonFOV_Table[0].FOV) {
    return(fujinonFOV_Table[0].zoomVal); 
  } else if(FOV >= fujinonFOV_Table[ZOOM_OUT_FULL - ZOOM_IN_FULL].FOV) {
    return(fujinonFOV_Table[ZOOM_OUT_FULL - ZOOM_IN_FULL].zoomVal);
  } else {
    for(int i=1; i < ZOOM_OUT_FULL - ZOOM_IN_FULL + 1; i++) {
      if((fujinonFOV_Table[i-1].FOV <= FOV) && (fujinonFOV_Table[i].FOV >= FOV)) {
	return(fujinonFOV_Table[i].zoomVal);
      }
    }
  }
  // If it gets here, we've failed
  return(ZOOM_OUT_FULL);
}
