/* FILE: hiResSensorDef.h
   AUTHOR: Michael Wagner
   CREATED: Jun 3, 1999
   DESCRIPTION: This file holds definitions for the Sony XC-003 camera
     on the Nomad robot.
*/

#ifndef __HI_RES_SENSOR_DEF_H__
#define __HI_RES_SENSOR_DEF_H__

#include "lensController.h"

#ifndef PI
#define PI 3.14159265359
#endif


#define TARGET_MARGIN 1.5 /* The multiplier of a target's angular width, to make sure 
			     there's some margin around it for the classifier. */

#define PXC_HI_RES_CHANNEL 1  /* The PXC-200 framegrabber channel that the high-res camera is using */
#define PXC_ARM_CAM_CHANNEL 2  /* The PXC-200 framegrabber channel that the arm camera is using */

#define HI_RES_SENSOR_BASE_TIME_COST 60.0   /* sec TBD: Make these not arbitrary */
#define HI_RES_SENSOR_BASE_ENERGY_COST 1.0 /* Joules TBD: Make these not arbitrary */
#define HI_RES_SENSOR_BASE_STORAGE_COST 921617 /* Bytes */

#define HI_RES_SENSOR_MAX_DISTANCE_TOLERANCE 7.0 /* m */
#define HI_RES_SENSOR_MIN_DISTANCE_TOLERANCE 3.0 /* m */

#define HI_RES_SENSOR_ANGULAR_TOLERANCE (PI / 2) /* rad */

typedef enum {
  HI_RES_SENSOR_OK = 0,

  /* PTU errors */
  HI_RES_SENSOR_PTU_FAILED,            // General PTU error
  HI_RES_SENSOR_PTU_STARTUP_FAILED,    // General PTU init error
  HI_RES_SENSOR_PTU_POSITION_MISMATCH, // PTU position mismatch 
  HI_RES_SENSOR_PTU_NOT_INITIALIZED,   // PTU not initialized (properly?)
  HI_RES_SENSOR_PTU_PORT_NOT_OPENED,   // Serial port not opened properly
  HI_RES_SENSOR_PTU_ILLEGAL_COMMAND_ARGUMENT, // See ptu.h
  HI_RES_SENSOR_PTU_ILLEGAL_COMMAND,
  HI_RES_SENSOR_PTU_ILLEGAL_POSITION_ARGUMENT,
  HI_RES_SENSOR_PTU_ILLEGAL_SPEED_ARGUMENT,
  HI_RES_SENSOR_PTU_ACCEL_TABLE_EXCEEDED,
  HI_RES_SENSOR_PTU_DEFAULTS_EEPROM_FAULT,
  HI_RES_SENSOR_PTU_SAVED_DEFAULTS_CORRUPTED,
  HI_RES_SENSOR_PTU_LIMIT_HIT,
  HI_RES_SENSOR_PTU_CABLE_DISCONNECTED,
  HI_RES_SENSOR_PTU_ILLEGAL_UNIT_ID,
  HI_RES_SENSOR_PTU_ILLEGAL_POWER_MODE,
  HI_RES_SENSOR_PTU_RESET_FAILED,
  HI_RES_SENSOR_PTU_NOT_RESPONDING,
  HI_RES_SENSOR_PTU_FIRMWARE_VERSION_TOO_LOW,

  /* Lens controller errors */
  HI_RES_SENSOR_LENS_FAILED,
  HI_RES_SENSOR_LENS_ZOOM_FAILED,
  HI_RES_SENSOR_LENS_FOCUS_FAILED,
  HI_RES_SENSOR_LENS_STARTUP_FAILED,
  HI_RES_SENSOR_LENS_PORT_NOT_OPENED,
  HI_RES_SENSOR_LENS_NOT_INITIALIZED,
  HI_RES_SENSOR_LENS_ZOOM_CMD_OUT_OF_BOUNDS,
  HI_RES_SENSOR_LENS_FOCUS_CMD_OUT_OF_BOUNDS,

  /* Combintation errors (go with "SENSOR_MALFUNCTION") */
  HI_RES_SENSOR_AUTOFOCUS_FAILED,

  /* Framegrabber errors */
  HI_RES_SENSOR_FG_FAILED,
  HI_RES_SENSOR_FG_STARTUP_FAILED,

  /* Operational errors (go with "SENSOR_DEPLOYMENT_FAILURE") */
  HI_RES_SENSOR_LOST_TARGET,
  HI_RES_SENSOR_SEGMENTATION_FAILED

} HI_RES_SENSOR_EXTENDED_STATUS;

/* 7/2/99 MDW: NOTE -- The following measurements are basically as good as I can get them.
   As of this date, the only source of error in these measurements could be the 
   TARGET_ACQ_CAM_ZOOM, which contains a small fudge factor. Errors could come from incorrect
   placement of the DGPS sensor, the PTU, the camera...especially if the PTU is not mounted
   parallel to the robot's z-axis (i.e., it's not perpendicular to a flat ground). If the PTU is
   tilted too far back (towards the -y-axis), the target may end up too high in a zoomed in image
   (this seems unintuitive, but remember that the initial distance estimate will be too low!). 
   Vice versa could also be true. 
   8/17/99 MDW: Numbers have been updated to reflect John Hurst's PTU mount.
*/

// Definitions for the camera on the robot.
#define TARGET_ACQ_CAM_PAN  (-0.10471975512)  /* rad (currently -6 deg) */
#define TARGET_ACQ_CAM_TILT (-0.349065850399)   /* rad (currently -20 deg) */
#define TARGET_ACQ_CAM_TILT_FUDGE (0) /* rad (as of 9/8, 0 deg) */
#define TARGET_ACQ_CAM_ZOOM (ZOOM_OUT_FULL - 4) 
#define TARGET_ACQ_WIDTH 2.5 /* m */

#define CCD_WIDTH 0.0059862387		/* [m] */
#define NUM_COL   640
#define NUM_ROW   480
#define MIDCOL    NUM_COL/2
#define MIDROW    NUM_ROW/2

/* These values define the vector from the PTU axes to the CCD. */
#define CCD_X     0					/* [m] */
#define CCD_Y     -0.04
#define CCD_Z     0.11

/* This next value is the height of the DGPS sensor off the ground. Therefore, to transform
   a DGPS sensor centered frame to a frame centered on the ground point underneath the 
   sensor, simply subtract this from Z */
#ifndef DGPS_SENSOR_Z
#define DGPS_SENSOR_Z 1.84 /* [m] */
#endif

/* Definitions for PTU axes position from the point on the ground directly below the DGPS 
   sensor as of 8/17/99 (i.e. John Hurst's PTU mount) (in meters!)
*/
#ifndef PTU_X
#define PTU_X -0.68 /* [m] */
#endif

#ifndef PTU_Y
#define PTU_Y 0.79 /* [m] */
#endif

#ifndef PTU_Z
#define PTU_Z (DGPS_SENSOR_Z + 0.225 - 0.078)  /* [m] */ /* (MDW, 9/21/99) The 0.078 is the height of the spacers that are now removed */
#endif

#define LENS_LENGTH  0.17			/* [m] */

#endif /* __HI_RES_SENSOR_DEF_H__ */
