/* FILE:    hiResSensor.cpp
   AUTHOR:  Michael Wagner
   CREATED: Apr 6, 1999

   DESCRIPTION: This file implements the high-resolution
     camera sensor manager device driver class, 
     hiResSensor. This class is inhereted from the
     virtual class sensor.h
*/

#include <iostream.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include <sys/types.h>
#include <unistd.h>
#include <pthread.h>
#include "ndds/NDDS.h"
#include "hiResSensor.h"
#include "hiResSensorDef.h"
#include "hiResPtu.h"
#include "lensController.h"
#include "dgpsVector.h"
#include "ptuAngle.h"
#include "sensorDef.h"
#include "ppm.h"
#include "dbRecord.h"
#include "dbRecordDef.h"
#include "dbDef.h"
#include "dbClient.h"
#include "database.h"
#include "ppmRockSeg.h"
#include "SAS_Config.h"
#include "nddsMsgNames.h"
#include "telemetryMsgs.h"
#include "teleopMsgs.h"

// The actual telemetry data structure.
extern hiResDerivedState *p_hiResDerived;

#ifndef MAX
#define MAX(_a_, _b_) ((_a_ > _b_) ? _a_ : _b_)
#endif
#ifndef MIN
#define MIN(_a_, _b_) ((_a_ < _b_) ? _a_ : _b_)
#endif

// Public functions

/* The constructor should simply instantiate objects and set class variables. */
hiResSensor::hiResSensor(char *sensorID, char *DAQ_SaveFileNameBase, 
			 frameGrabberConfig *fgConfig, dgpsVector *ptuPosition) : sensor() {

  // Set up sensor parameters defined in sensor.h
  strcpy(this->sensorID, sensorID);
  sensorStatus = SENSOR_OFF_LINE;
  sensorExtendedStatus = HI_RES_SENSOR_OK;

  // Set up private objects and variables unique to hiResSensor
  ptu = new hiResPtu(ptuPosition);
  lens = new lensController();
  fg = new frameGrabber(fgConfig);
  if(!getSAS_Config(PTU_SERIAL_PORT, ptuSerialPortNum)) {
    cerr << "[hiResSensor] ERROR: Cannot read " << PTU_SERIAL_PORT << " from config file!" << endl;
    return;
  }
  if(!getSAS_Config(LENS_CONTROLLER_SERIAL_PORT, lensSerialPortNum)) {
    cerr << "[hiResSensor] ERROR: Cannot read " << LENS_CONTROLLER_SERIAL_PORT << " from config file!" << endl;
    return;
  }
  this->DAQ_SaveFileNameBase = new char[strlen(DAQ_SaveFileNameBase)+1];
  strcpy(this->DAQ_SaveFileNameBase, DAQ_SaveFileNameBase);
  debugMode = 0;
  int nddsDomain;
  if(!getSAS_Config(NDDS_DOMAIN, nddsDomain)) {
    cerr << "[hiResSensor] ERROR: Cannot read " << NDDS_DOMAIN << " from config file!" << endl;
    return;
  }    
  float period = HI_RES_TELEMETRY_DEFAULT_PERIOD;
  telThread = new hiResSensorTelemetryThread(nddsDomain, period);
  telThread->start();

#ifdef RED_TEST
  cerr << "[hiResSensor] Using red thresholding technique." << endl;
#endif

  // Set up teleop server
  hiResRequestNddsRegister();
  hiResReplyNddsRegister();
  hiResRequest *hiResRequestMsg = (hiResRequest *)calloc(1, sizeof(hiResRequest));
  hiResReply *hiResReplyMsg = (hiResReply *)calloc(1, sizeof(hiResReply));
  teleopCallbackParam *param = (teleopCallbackParam *)calloc(1, sizeof(teleopCallbackParam));
  param->ptu = ptu;
  param->lens = lens;
  param->fg = fg;
  param->sensor = this;
  NddsServerCreate(TELEOP_HI_RES_SERVICE, NDDS_SERVER_IMMEDIATE, 10.0f,
  		   hiResReplyMsg, hiResRequestMsg,
  		   hiResReplyPublicationType,
  		   hiResRequestPublicationType,
  		   teleopCallback, param);
  
}

/* The destructor should delete objects in the sensor driver class. Deleting the objects in this class should 
   automatically call their respective uninit functions, to close serial ports and such. */
hiResSensor::~hiResSensor() {
  telThread->stop();
  delete telThread;
  delete ptu;
  delete lens;
  delete fg;
  delete DAQ_SaveFileNameBase;
}

/* The startup function should initialize all subcomponents, and prepare the sensor for deployment. */
sensorReply *hiResSensor::startup(sensorRequest *request) {
  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));

  // Set up default values of reply.
  strcpy(reply->sensorID, sensorID);
  reply->function = request->function;
  reply->status = SENSOR_OK;
  reply->extendedStatus = HI_RES_SENSOR_OK;

  int ptuStatus;
  ptuAngle *newAngle;

  // Start up the PTU
  switch(ptu->init(ptuSerialPortNum)) {
  case PTU_OK:
    newAngle = new ptuAngle(TARGET_ACQ_CAM_PAN, TARGET_ACQ_CAM_TILT);
    ptuStatus = ptu->servoToAngleAbs(newAngle);
    delete(newAngle);
    switch(ptuStatus) {
    case PTU_OK:
      break;
    case PTU_POSITION_MISMATCH:
      cerr << "[hiResSensor] ERROR: PTU could not servo to target acquisition position (position mismatch)!" << endl;
      sensorStatus = reply->status = SENSOR_MALFUNCTION;
      sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_POSITION_MISMATCH;
      return(reply);
    case PTU_ILLEGAL_COMMAND_ARGUMENT:
      cerr << "[hiResSensor] ERROR: PTU could not servo to target acquisition position (illegal command argument)!" << endl;
      reply->status = SENSOR_MALFUNCTION;
      reply->extendedStatus = HI_RES_SENSOR_PTU_ILLEGAL_COMMAND_ARGUMENT;
      return(reply);
    case PTU_ILLEGAL_COMMAND:
      cerr << "[hiResSensor] ERROR: PTU could not servo to target acquisition position (illegal command)!" << endl;
      reply->status = SENSOR_MALFUNCTION;
      reply->extendedStatus = HI_RES_SENSOR_PTU_ILLEGAL_COMMAND;
      return(reply);
    case PTU_ILLEGAL_POSITION_ARGUMENT:
      cerr << "[hiResSensor] ERROR: PTU could not servo to target acquisition position (illegal position argument)!" << endl;
      reply->status = SENSOR_MALFUNCTION;
      reply->extendedStatus = HI_RES_SENSOR_PTU_ILLEGAL_POSITION_ARGUMENT;
      return(reply);
    case PTU_ILLEGAL_SPEED_ARGUMENT:
      cerr << "[hiResSensor] ERROR: PTU could not servo to target acquisition position (illegal speed argument)!" << endl;
      reply->status = SENSOR_MALFUNCTION;
      reply->extendedStatus = HI_RES_SENSOR_PTU_ILLEGAL_SPEED_ARGUMENT;
      return(reply);
    case PTU_ACCEL_TABLE_EXCEEDED:
      cerr << "[hiResSensor] ERROR: PTU could not servo to target acquisition position (acceleration table exceeded)!" << endl;
      sensorStatus = reply->status = SENSOR_MALFUNCTION;
      sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_ACCEL_TABLE_EXCEEDED;
      return(reply);
    case PTU_DEFAULTS_EEPROM_FAULT:
      cerr << "[hiResSensor] ERROR: PTU could not servo to target acquisition position (deafults eeprom fault)!" << endl;
      sensorStatus = reply->status = SENSOR_MALFUNCTION;
      sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_DEFAULTS_EEPROM_FAULT;
      return(reply);
    case PTU_SAVED_DEFAULTS_CORRUPTED:
      cerr << "[hiResSensor] ERROR: PTU could not servo to target acquisition position (saved defaults corrupted)!" << endl;
      sensorStatus = reply->status = SENSOR_MALFUNCTION;
      sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_SAVED_DEFAULTS_CORRUPTED;
      return(reply);
    case PTU_LIMIT_HIT:
      cerr << "[hiResSensor] ERROR: PTU could not servo to target acquisition position (limit switch hit)!" << endl;
      sensorStatus = reply->status = SENSOR_MALFUNCTION;
      sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_LIMIT_HIT;
      return(reply);
    case PTU_CABLE_DISCONNECTED:
      cerr << "[hiResSensor] ERROR: PTU could not servo to target acquisition position (cable disconnected)!" << endl;
      sensorStatus = reply->status = SENSOR_MALFUNCTION;
      sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_CABLE_DISCONNECTED;
      return(reply);
    case PTU_ILLEGAL_UNIT_ID:
      cerr << "[hiResSensor] ERROR: PTU could not servo to target acquisition position (illegal unit id)!" << endl;
      reply->status = SENSOR_MALFUNCTION;
      reply->extendedStatus = HI_RES_SENSOR_PTU_ILLEGAL_UNIT_ID;
      return(reply);
    case PTU_ILLEGAL_POWER_MODE:
      cerr << "[hiResSensor] ERROR: PTU could not servo to target acquisition position (illegal power mode)!" << endl;
      reply->status = SENSOR_MALFUNCTION;
      reply->extendedStatus = HI_RES_SENSOR_PTU_ILLEGAL_POWER_MODE;
      return(reply);
    case PTU_NOT_RESPONDING:
      cerr << "[hiResSensor] ERROR: PTU could not servo to target acquisition position (ptu not responding)!" << endl;
      sensorStatus = reply->status = SENSOR_MALFUNCTION;
      sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_NOT_RESPONDING;
      return(reply);
    default:
      cerr << "[hiResSensor] ERROR: PTU could not servo to target acquisition position!" << endl;
      reply->status = SENSOR_MALFUNCTION;
      reply->extendedStatus = HI_RES_SENSOR_PTU_FAILED;
      return(reply);
    }
    break;
 
  case PORT_NOT_OPENED:
    cerr << "[hiResSensor] ERROR: PTU could not startup (serial port not opened)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_PORT_NOT_OPENED;
    return(reply);
  case PTU_ACCEL_TABLE_EXCEEDED:
    cerr << "[hiResSensor] ERROR: PTU could not startup (acceleration table exceeded)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_ACCEL_TABLE_EXCEEDED;
    return(reply);
  case PTU_DEFAULTS_EEPROM_FAULT:
    cerr << "[hiResSensor] ERROR: PTU could not startup (defaults eeprom fault)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_DEFAULTS_EEPROM_FAULT;
    return(reply);
  case PTU_SAVED_DEFAULTS_CORRUPTED:
    cerr << "[hiResSensor] ERROR: PTU could not startup (saved defaults corrupted)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_SAVED_DEFAULTS_CORRUPTED;
    return(reply);
  case PTU_LIMIT_HIT:
    cerr << "[hiResSensor] ERROR: PTU could not startup (limit switch hit)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_LIMIT_HIT;
    return(reply);
  case PTU_CABLE_DISCONNECTED:
    cerr << "[hiResSensor] ERROR: PTU could not startup (cable disconnected)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_CABLE_DISCONNECTED;
    return(reply);
  case PTU_ILLEGAL_UNIT_ID:
    cerr << "[hiResSensor] ERROR: PTU could not startup (illegal unit id)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_ILLEGAL_UNIT_ID;
    return(reply);
  case PTU_ILLEGAL_POWER_MODE:
    cerr << "[hiResSensor] ERROR: PTU could not startup (illegal power mode)!" << endl;
    reply->status = SENSOR_MALFUNCTION;
    reply->extendedStatus = HI_RES_SENSOR_PTU_ILLEGAL_POWER_MODE;
    return(reply);
  case PTU_RESET_FAILED:
    cerr << "[hiResSensor] ERROR: PTU could not startup (reset failed)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_RESET_FAILED;
    return(reply);
  case PTU_NOT_RESPONDING:
    cerr << "[hiResSensor] ERROR: PTU could not startup (ptu not responding)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_NOT_RESPONDING;
    return(reply);
  case PTU_FIRMWARE_VERSION_TOO_LOW:
    cerr << "[hiResSensor] ERROR: PTU could not startup (ptu not responding)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_FIRMWARE_VERSION_TOO_LOW;
    return(reply);
  default:
    cerr << "[hiResSensor] ERROR: PTU could not startup!" << endl;
    reply->status = SENSOR_MALFUNCTION;
    reply->extendedStatus = HI_RES_SENSOR_PTU_STARTUP_FAILED;
    return(reply);
  }

  // Start up the lens controller
  switch(lens->init(lensSerialPortNum)) {
  case LENS_OK:
    switch(lens->setZoomAbsBlock(ZOOM_OUT_FULL)) {
    case LENS_OK:
      switch(lens->setFocusAbsBlock(FOCUS_OUT_FULL)) {
      case LENS_OK:
	break;
      case LENS_FAIL:
	cerr << "[hiResSensor] ERROR: Could not focus out fully!" << endl;
	sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_LENS_FOCUS_FAILED;
	sensorStatus = reply->status = SENSOR_MALFUNCTION;
	return(reply);
      case LENS_CONTROLLER_CMD_OUT_OF_BOUNDS:
	cerr << "[hiResSensor] ERROR: Could not focus out fully (command out of bounds)!" << endl;
	reply->extendedStatus = HI_RES_SENSOR_LENS_FOCUS_CMD_OUT_OF_BOUNDS;
	reply->status = SENSOR_MALFUNCTION;
	return(reply);
      default:
	cerr << "[hiResSensor] ERROR: Could not focus out fully!" << endl;
	reply->extendedStatus = HI_RES_SENSOR_LENS_FOCUS_FAILED;
	reply->status = SENSOR_MALFUNCTION;
	return(reply);
      }
      break;
    case LENS_FAIL:
      cerr << "[hiResSensor] ERROR: Could not zoom out fully!" << endl;
      sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_LENS_ZOOM_FAILED;
      sensorStatus = reply->status = SENSOR_MALFUNCTION;
      return(reply);
    case LENS_CONTROLLER_CMD_OUT_OF_BOUNDS:
      cerr << "[hiResSensor] ERROR: Could not zoom out fully (command out of bounds)!" << endl;
      reply->extendedStatus = HI_RES_SENSOR_LENS_ZOOM_CMD_OUT_OF_BOUNDS;
      reply->status = SENSOR_MALFUNCTION;
      return(reply);
    default:
      cerr << "[hiResSensor] ERROR: Could not zoom out fully!" << endl;
      reply->extendedStatus = HI_RES_SENSOR_LENS_ZOOM_FAILED;
      reply->status = SENSOR_MALFUNCTION;
      return(reply);
    }
    break;
  case PORT_NOT_OPENED:
    cerr << "[hiResSensor] ERROR: Lens controller startup failed (port not opened)!" << endl;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_LENS_PORT_NOT_OPENED;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    return(reply);
  case LENS_CONTROLLER_NOT_INITIALIZED:
    cerr << "[hiResSensor] ERROR: Lens controller startup failed!" << endl;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_LENS_STARTUP_FAILED;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    return(reply);
  default:
    cerr << "[hiResSensor] ERROR: Lens controller startup failed!" << endl;
    reply->extendedStatus = HI_RES_SENSOR_LENS_STARTUP_FAILED;
    reply->status = SENSOR_MALFUNCTION;
    return(reply);
  }  
}

/* The calibration function should be used to update internal state variables with correct values. */
sensorReply *hiResSensor::calibration(sensorRequest *request) {
  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  strcpy(reply->sensorID, sensorID);
  reply->function = request->function;

  switch(ptu->reset()) {
  case PTU_OK:
    if(sensorStatus == SENSOR_NEEDS_CALIBRATION) {
      sensorStatus = SENSOR_OK;
    }
    reply->extendedStatus = HI_RES_SENSOR_OK;
    return(reply);
  case PORT_NOT_OPENED:
    cerr << "[hiResSensor] ERROR: PTU failed on reset calibration (serial port not opened)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_PORT_NOT_OPENED;
    return(reply);
  case PTU_ACCEL_TABLE_EXCEEDED:
    cerr << "[hiResSensor] ERROR: PTU failed on reset calibration (acceleration table exceeded)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_ACCEL_TABLE_EXCEEDED;
    return(reply);
  case PTU_DEFAULTS_EEPROM_FAULT:
    cerr << "[hiResSensor] ERROR: PTU failed on reset calibration (defaults eeprom fault)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_DEFAULTS_EEPROM_FAULT;
    return(reply);
  case PTU_SAVED_DEFAULTS_CORRUPTED:
    cerr << "[hiResSensor] ERROR: PTU failed on reset calibration (saved defaults corrupted)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_SAVED_DEFAULTS_CORRUPTED;
    return(reply);
  case PTU_LIMIT_HIT:
    cerr << "[hiResSensor] ERROR: PTU failed on reset calibration (limit switch hit)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_LIMIT_HIT;
    return(reply);
  case PTU_CABLE_DISCONNECTED:
    cerr << "[hiResSensor] ERROR: PTU failed on reset calibration (cable disconnected)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_CABLE_DISCONNECTED;
    return(reply);
  case PTU_ILLEGAL_UNIT_ID:
    cerr << "[hiResSensor] ERROR: PTU failed on reset calibration (illegal unit id)!" << endl;
    reply->status = SENSOR_MALFUNCTION;
    reply->extendedStatus = HI_RES_SENSOR_PTU_ILLEGAL_UNIT_ID;
    return(reply);
  case PTU_ILLEGAL_POWER_MODE:
    cerr << "[hiResSensor] ERROR: PTU failed on reset calibration (illegal power mode)!" << endl;
    reply->status = SENSOR_MALFUNCTION;
    reply->extendedStatus = HI_RES_SENSOR_PTU_ILLEGAL_POWER_MODE;
    return(reply);
  case PTU_NOT_RESPONDING:
    cerr << "[hiResSensor] ERROR: PTU failed on reset calibration (ptu not responding)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_NOT_RESPONDING;
    return(reply);
  default:
    cerr << "[hiResSensor] ERROR: PTU failed on reset calibration!" << endl;
    reply->status = SENSOR_MALFUNCTION;
    reply->extendedStatus = HI_RES_SENSOR_PTU_STARTUP_FAILED;
    return(reply);
  }
}

/* The deployment function should deploy the sensor to the target coordinates (request->data.DGPS_coord), 
   given a current robot location and pose (request->robotPosition). */ 
sensorReply *hiResSensor::deployment(sensorRequest *request) {
  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));  
  strcpy(reply->sensorID, sensorID);
  reply->function = request->function;
  reply->status = SENSOR_OK;
  reply->extendedStatus = HI_RES_SENSOR_OK;

  dgpsVector *target = new dgpsVector(request->data.DGPS_coord.x,
				      request->data.DGPS_coord.y,
				      request->data.DGPS_coord.z);

  dgpsVector *current = new dgpsVector(request->robotPosition.x, 
				       request->robotPosition.y, 
				       request->robotPosition.z);

  /* Servo PTU to target. This will take care of pitch, roll, and yaw internally.
     Check out the return codes. Any important PTU status codes (defined in /usr/local/ptu-cpi/include/ptu.h)
     should map to a HI_RES_SENSOR_EXTENDED_STATUS. */
  int ptuStatus = ptu->servoToVector(current, target, &request->robotPose);
  delete(target); delete(current);

  switch(ptuStatus) {
  case PTU_OK:
    break;
  case PTU_POSITION_MISMATCH:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target position (position mismatch)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_POSITION_MISMATCH;
    return(reply);
  case PTU_ILLEGAL_COMMAND_ARGUMENT:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target position (illegal command argument)!" << endl;
    reply->status = SENSOR_MALFUNCTION;
    reply->extendedStatus = HI_RES_SENSOR_PTU_ILLEGAL_COMMAND_ARGUMENT;
    return(reply);
  case PTU_ILLEGAL_COMMAND:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target position (illegal command)!" << endl;
    reply->status = SENSOR_MALFUNCTION;
    reply->extendedStatus = HI_RES_SENSOR_PTU_ILLEGAL_COMMAND;
    return(reply);
  case PTU_ILLEGAL_POSITION_ARGUMENT:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target position (illegal position argument)!" << endl;
    reply->status = SENSOR_MALFUNCTION;
    reply->extendedStatus = HI_RES_SENSOR_PTU_ILLEGAL_POSITION_ARGUMENT;
    return(reply);
  case PTU_ILLEGAL_SPEED_ARGUMENT:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target position (illegal speed argument)!" << endl;
    reply->status = SENSOR_MALFUNCTION;
    reply->extendedStatus = HI_RES_SENSOR_PTU_ILLEGAL_SPEED_ARGUMENT;
    return(reply);
  case PTU_ACCEL_TABLE_EXCEEDED:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target position (acceleration table exceeded)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_ACCEL_TABLE_EXCEEDED;
    return(reply);
  case PTU_DEFAULTS_EEPROM_FAULT:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target position (defults eeprom fault)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_DEFAULTS_EEPROM_FAULT;
    return(reply);
  case PTU_SAVED_DEFAULTS_CORRUPTED:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target position (saved defaults corrupted)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_SAVED_DEFAULTS_CORRUPTED;
    return(reply);
  case PTU_LIMIT_HIT:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target position (limit switch hit)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_LIMIT_HIT;
    return(reply);
  case PTU_CABLE_DISCONNECTED:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target position (cable disconnected)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_CABLE_DISCONNECTED;
    return(reply);
  case PTU_ILLEGAL_UNIT_ID:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target position (illegal unit id)!" << endl;
    reply->status = SENSOR_MALFUNCTION;
    reply->extendedStatus = HI_RES_SENSOR_PTU_ILLEGAL_UNIT_ID;
    return(reply);
  case PTU_ILLEGAL_POWER_MODE:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target position (illegal power mode)!" << endl;
    reply->status = SENSOR_MALFUNCTION;
    reply->extendedStatus = HI_RES_SENSOR_PTU_ILLEGAL_POWER_MODE;
    return(reply);
  case PTU_NOT_RESPONDING:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target position (ptu not responding)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_NOT_RESPONDING;
    return(reply);
  default:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target position!" << endl;
    reply->status = SENSOR_MALFUNCTION;
    reply->extendedStatus = HI_RES_SENSOR_PTU_FAILED;
    return(reply);
  }

  switch(lens->setZoomAbsBlock(ZOOM_OUT_FULL)) {
  case LENS_OK:
    break;
  case LENS_FAIL:
    cerr << "[hiResSensor] ERROR: Could not zoom out fully!" << endl;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_LENS_ZOOM_FAILED;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    return(reply);
  case LENS_CONTROLLER_CMD_OUT_OF_BOUNDS:
    cerr << "[hiResSensor] ERROR: Could not zoom out fully (command out of bounds)!" << endl;
    reply->extendedStatus = HI_RES_SENSOR_LENS_ZOOM_CMD_OUT_OF_BOUNDS;
    reply->status = SENSOR_MALFUNCTION;
    return(reply);
  default:
    cerr << "[hiResSensor] ERROR: Could not zoom out fully!" << endl;
    reply->extendedStatus = HI_RES_SENSOR_LENS_ZOOM_FAILED;
    reply->status = SENSOR_MALFUNCTION;
    return(reply);
  }
  
  acqResults *results = new acqResults;
  ppmRockSeg *segmentor = new ppmRockSeg();

  /* Here's the deployment algorithm:
       1) Read in image
       2) Segment the image to find the shape and position of the target
       3) Find the target in the image closest to the center
       4) Calculate the angular size of the target
       5) Pan and tilt to center on the target's position
       6) Find the zoom encoder value that will increase the target's size in the image 
          until it almost fills the screen (there needs to be some margin for the classifier)
       7) Zoom in halfway to this target zoom value
       8) Perform autofocus
       9) Segment the image again to see if the target is still in the image
       10) Find the target in the image closest to the center
       11) Pan and tilt to center on the target's position
       12) Zoom in fully on the center of the target
       13) Perform autofocus
  */

  time_t startTime, endTime;

  time(&startTime);

  // 1) Read the image
  if(!segmentor->read(fg->captureHiResPPM(PXC_HI_RES_CHANNEL))) {
    cerr << "[hiResSensor] ERROR: Cannot read image!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_FG_FAILED;
    return(reply);
  }

  // 2) Segment the image
  // The segmentor parameters are NULL because we don't care about DGPS estimates
#ifdef RED_TEST 
  if(!segmentor->redThreshhold(NULL, NULL, 0, NUM_ROW, results)) {
#else
  if(!segmentor->kimThreshhold(NULL, NULL, 0, NUM_ROW, results)) { 
#endif
    cerr << "[hiResSensor] ERROR: Cannot segment image!" << endl;
    reply->status = SENSOR_DEPLOYMENT_FAILURE;
    reply->extendedStatus = HI_RES_SENSOR_SEGMENTATION_FAILED;
    return(reply);
  } 
  segmentor->write("/tmp/DAQ_segmentedImage.ppm");
      
  cerr << "***** Found " << results->numTargets << " target(s)" << endl;
  
  if(results->numTargets < 1) {
    cerr << "[hiResSensor] ERROR: Cannot find any targets in deployment image!" << endl;
    reply->status = SENSOR_DEPLOYMENT_FAILURE;
    reply->extendedStatus = HI_RES_SENSOR_LOST_TARGET;
    return(reply);
  }
  
  // 3) Find the target closest to the image center
  float minDistance = NUM_ROW * NUM_COL;
  int minDistanceIndex = -1;
  float tmpDistance;
  for(int i=0; i < results->numTargets; i++) {
    tmpDistance = sqrt((NUM_ROW/2 - results->targets[i].midpt.row) * (NUM_ROW/2 - results->targets[i].midpt.row) + 
		       (NUM_COL/2 - results->targets[i].midpt.column) * (NUM_COL/2 - results->targets[i].midpt.column));
    if(tmpDistance < minDistance) {
      minDistance = tmpDistance;
      minDistanceIndex = i;
    }
  }
      
  // 4) Find the angular size of the target
  cerr << "[hiResSensor] New target's upper left corner is at (" << results->targets[minDistanceIndex].firstPixel.row << ", " 
       << results->targets[minDistanceIndex].firstPixel.column << "), lower right corner is at (" 
       << results->targets[minDistanceIndex].lastPixel.row << ", " << results->targets[minDistanceIndex].lastPixel.column << "), and its center is at (" 
       << results->targets[minDistanceIndex].midpt.row << ", " << results->targets[minDistanceIndex].midpt.column << ")" << endl;
  
  float radiansPerRow = fieldOfView(ZOOM_OUT_FULL) / NUM_ROW;
  float radiansPerCol = fieldOfView(ZOOM_OUT_FULL) / NUM_COL;
  
  int rowOff = (int)(NUM_ROW / 2) - results->targets[minDistanceIndex].midpt.row;
  int colOff = (int)(NUM_COL / 2) - results->targets[minDistanceIndex].midpt.column;
  
  cerr << "[hiResSensor] Field of view is " << radiansPerRow*NUM_ROW << " radians" << endl;
  
  float height = fabs(results->targets[minDistanceIndex].firstPixel.row - results->targets[minDistanceIndex].lastPixel.row) * TARGET_MARGIN,
        width  = fabs(results->targets[minDistanceIndex].firstPixel.column - results->targets[minDistanceIndex].lastPixel.column) * TARGET_MARGIN;
    
  float targetAngularSize = (height >= width) ? height * radiansPerRow : width * radiansPerCol; // TBD: Take into account the margin!

  // 5) Center the target in the image
  switch(ptu->servoToAngleRel(new ptuAngle(radiansPerCol*colOff, radiansPerRow*rowOff))) {
  case PTU_OK:
    
    // 6) Find the correct zoom encoder value to zoom fully in on the target.
    cerr << "[hiResSensor] Target's angular size is " << targetAngularSize << " radians, so zooming to z = " << (int)fovToZoomByte(targetAngularSize) << endl;

    if(fovToZoomByte(targetAngularSize) * 2 < ZOOM_OUT_FULL) {

      // 7) Zoom in halfway on this target value
      cerr << "[hiResSensor] First we zoom into " << (int)fovToZoomByte(targetAngularSize) * 2 << endl;
      switch(lens->setZoomAbsBlock(fovToZoomByte(targetAngularSize) * 2)) {
      case LENS_OK:
	// 8) Do autofocus
	if(autoFocus(fg, lens)) {

	  // 9) Segment image again
#ifdef RED_TEST 
	  if(!segmentor->redThreshhold(NULL, NULL, 0, NUM_ROW, results)) {
#else
	  if(!segmentor->kimThreshhold(NULL, NULL, 0, NUM_ROW, results)) { 
#endif
	    cerr << "[hiResSensor] ERROR: Cannot segment halfway zoomed image!" << endl;
	    reply->status = SENSOR_DEPLOYMENT_FAILURE;
	    reply->extendedStatus = HI_RES_SENSOR_SEGMENTATION_FAILED;
	    return(reply);
	  } 
	  segmentor->write("/tmp/DAQ_segmentedImage.ppm");
	  
	  cerr << "***** Found " << results->numTargets << " target(s)" << endl;
	  
	  if(results->numTargets < 1) {
	    cerr << "[hiResSensor] ERROR: Cannot find any targets in halfway zoomed deployment image!" << endl;
	    reply->status = SENSOR_DEPLOYMENT_FAILURE;
	    reply->extendedStatus = HI_RES_SENSOR_LOST_TARGET;
	    return(reply);
	  }
  
	  // 10) Find the target in the image that's closest to the center
	  cerr << "[hiResSensor] New target's upper left corner is at (" << results->targets[minDistanceIndex].firstPixel.row << ", " 
	       << results->targets[minDistanceIndex].firstPixel.column << "), lower right corner is at (" 
	       << results->targets[minDistanceIndex].lastPixel.row << ", " << results->targets[minDistanceIndex].lastPixel.column << "), and its center is at (" 
	       << results->targets[minDistanceIndex].midpt.row << ", " << results->targets[minDistanceIndex].midpt.column << ")" << endl;
  
	  radiansPerRow = fieldOfView(lens->getHardwareZoom()) / NUM_ROW;
	  radiansPerCol = fieldOfView(lens->getHardwareZoom()) / NUM_COL;
	  
	  rowOff = (int)(NUM_ROW / 2) - results->targets[minDistanceIndex].midpt.row;
	  colOff = (int)(NUM_COL / 2) - results->targets[minDistanceIndex].midpt.column;
	  
	  cerr << "[hiResSensor] Field of view is " << radiansPerRow*NUM_ROW << " radians" << endl;
  
	  // 11) Pan and tilt to center on the target's position
	  switch(ptu->servoToAngleRel(new ptuAngle(radiansPerCol*colOff, radiansPerRow*rowOff))) {
	  case PTU_OK:
	    // 12) Zoom in all the way
	    cerr << "[hiResSensor] Now zooming in all the way to " << (int)fovToZoomByte(targetAngularSize)
		 << endl;
	    switch(lens->setZoomAbsBlock(fovToZoomByte(targetAngularSize))) {
	    case LENS_OK:
	      // 13) Do autofocus
	      if(autoFocus(fg, lens)) {  // TBD: This could probably be bounded to look only within +/- 10 of the last autofocus result.
		cerr << "[hiResSensor] Deployment successful!" << endl;
		time(&endTime);
		cerr << "[hiResSensor] *** The deployment operation took " << (long)endTime - (long)startTime << " seconds" << endl;
		return(reply);
	      } else {
		cerr << "[hiResSensor] ERROR: Autofocus failed!" << endl;
		reply->status = SENSOR_MALFUNCTION;
		reply->extendedStatus = HI_RES_SENSOR_AUTOFOCUS_FAILED;
		return(reply);
	      }
	    case LENS_FAIL:
	      cerr << "[hiResSensor] ERROR: Could not zoom fully in on target!" << endl;
	      sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_LENS_ZOOM_FAILED;
	      sensorStatus = reply->status = SENSOR_MALFUNCTION;
	      return(reply);
	    case LENS_CONTROLLER_CMD_OUT_OF_BOUNDS:
	      cerr << "[hiResSensor] ERROR: Could not zoom fully in on target (command out of bounds)!" << endl;
	      reply->extendedStatus = HI_RES_SENSOR_LENS_ZOOM_CMD_OUT_OF_BOUNDS;
	      reply->status = SENSOR_MALFUNCTION;
	      return(reply);
	    default:
	      cerr << "[hiResSensor] ERROR: Could not zoom fully in on target!" << endl;
	      reply->extendedStatus = HI_RES_SENSOR_LENS_ZOOM_FAILED;
	      reply->status = SENSOR_MALFUNCTION;
	      return(reply);
	    }
	  }
	} else {
	  cerr << "[hiResSensor] ERROR: Autofocus failed!" << endl;
	  reply->status = SENSOR_MALFUNCTION;
	  reply->extendedStatus = HI_RES_SENSOR_AUTOFOCUS_FAILED;
	  return(reply);
	}
      case LENS_FAIL:
	cerr << "[hiResSensor] ERROR: Could not zoom halfway in on target!" << endl;
	sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_LENS_ZOOM_FAILED;
	sensorStatus = reply->status = SENSOR_MALFUNCTION;
	return(reply);
      case LENS_CONTROLLER_CMD_OUT_OF_BOUNDS:
	cerr << "[hiResSensor] ERROR: Could not zoom halfway in on target (command out of bounds)!" << endl;
	reply->extendedStatus = HI_RES_SENSOR_LENS_ZOOM_CMD_OUT_OF_BOUNDS;
	reply->status = SENSOR_MALFUNCTION;
	return(reply);
      default:
	cerr << "[hiResSensor] ERROR: Could not zoom halfway in on target!" << endl;
	reply->extendedStatus = HI_RES_SENSOR_LENS_ZOOM_FAILED;
	reply->status = SENSOR_MALFUNCTION;
	return(reply);
      }
    } else {
      // Target is too big -- just go to step 12
      // 12) Zoom in all the way
      switch(lens->setZoomAbsBlock(fovToZoomByte(targetAngularSize))) {
      case LENS_OK:
	// 13) Do autofocus
	if(autoFocus(fg, lens)) {
	  cerr << "[hiResSensor] Deployment successful!" << endl;
	  time(&endTime);
	  cerr << "[hiResSensor] *** The deployment operation took " << (long)endTime - (long)startTime << " seconds" << endl;
	  return(reply);
	} else {
	  cerr << "[hiResSensor] ERROR: Autofocus failed!" << endl;
	  reply->status = SENSOR_MALFUNCTION;
	  reply->extendedStatus = HI_RES_SENSOR_AUTOFOCUS_FAILED;
	  return(reply);
	}
      case LENS_FAIL:
	cerr << "[hiResSensor] ERROR: Could not zoom fully in on target!" << endl;
	sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_LENS_ZOOM_FAILED;
	sensorStatus = reply->status = SENSOR_MALFUNCTION;
	return(reply);
      case LENS_CONTROLLER_CMD_OUT_OF_BOUNDS:
	cerr << "[hiResSensor] ERROR: Could not zoom fully in on target (command out of bounds)!" << endl;
	reply->extendedStatus = HI_RES_SENSOR_LENS_ZOOM_CMD_OUT_OF_BOUNDS;
	reply->status = SENSOR_MALFUNCTION;
	return(reply);
      default:
	cerr << "[hiResSensor] ERROR: Could not zoom fully in on target!" << endl;
	reply->extendedStatus = HI_RES_SENSOR_LENS_ZOOM_FAILED;
	reply->status = SENSOR_MALFUNCTION;
	return(reply);
      }
    }
  case PTU_POSITION_MISMATCH:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target's center (position mismatch)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_POSITION_MISMATCH;
    return(reply);
  case PTU_ILLEGAL_COMMAND_ARGUMENT:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target's center (illegal command argument)!" << endl;
    reply->status = SENSOR_MALFUNCTION;
    reply->extendedStatus = HI_RES_SENSOR_PTU_ILLEGAL_COMMAND_ARGUMENT;
    return(reply);
  case PTU_ILLEGAL_COMMAND:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target's center (illegal command)!" << endl;
    reply->status = SENSOR_MALFUNCTION;
    reply->extendedStatus = HI_RES_SENSOR_PTU_ILLEGAL_COMMAND;
    return(reply);
  case PTU_ILLEGAL_POSITION_ARGUMENT:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target's center (illegal position argument)!" << endl;
    reply->status = SENSOR_MALFUNCTION;
    reply->extendedStatus = HI_RES_SENSOR_PTU_ILLEGAL_POSITION_ARGUMENT;
    return(reply);
  case PTU_ILLEGAL_SPEED_ARGUMENT:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target's center (illegal speed argument)!" << endl;
    reply->status = SENSOR_MALFUNCTION;
    reply->extendedStatus = HI_RES_SENSOR_PTU_ILLEGAL_SPEED_ARGUMENT;
    return(reply);
  case PTU_ACCEL_TABLE_EXCEEDED:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target's center (acceleration table exceeded)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_ACCEL_TABLE_EXCEEDED;
    return(reply);
  case PTU_DEFAULTS_EEPROM_FAULT:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target's center (defults eeprom fault)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_DEFAULTS_EEPROM_FAULT;
    return(reply);
  case PTU_SAVED_DEFAULTS_CORRUPTED:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target's center (saved defaults corrupted)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_SAVED_DEFAULTS_CORRUPTED;
    return(reply);
  case PTU_LIMIT_HIT:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target's center (limit switch hit)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_LIMIT_HIT;
    return(reply);
  case PTU_CABLE_DISCONNECTED:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target's center (cable disconnected)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_CABLE_DISCONNECTED;
    return(reply);
  case PTU_ILLEGAL_UNIT_ID:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target's center (illegal unit id)!" << endl;
    reply->status = SENSOR_MALFUNCTION;
    reply->extendedStatus = HI_RES_SENSOR_PTU_ILLEGAL_UNIT_ID;
    return(reply);
  case PTU_ILLEGAL_POWER_MODE:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target's center (illegal power mode)!" << endl;
    reply->status = SENSOR_MALFUNCTION;
    reply->extendedStatus = HI_RES_SENSOR_PTU_ILLEGAL_POWER_MODE;
    return(reply);
  case PTU_NOT_RESPONDING:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target's center (ptu not responding)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_NOT_RESPONDING;
    return(reply);
  default:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target's center!" << endl;
    reply->status = SENSOR_MALFUNCTION;
    reply->extendedStatus = HI_RES_SENSOR_PTU_FAILED;
    return(reply);
  }
}

/* The unDeployment function should servo the sensor back to its startup state. */
sensorReply *hiResSensor::unDeployment(sensorRequest *request) {
  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  strcpy(reply->sensorID, sensorID);
  reply->function = request->function;
  reply->status = SENSOR_OK;
  reply->extendedStatus = HI_RES_SENSOR_OK;

  // Place PTU in target acq position
  switch(ptu->servoToAngleAbs(new ptuAngle(TARGET_ACQ_CAM_PAN, TARGET_ACQ_CAM_TILT))) {
  case PTU_OK:
    
    // Zoom out.
    switch(lens->setZoomAbsBlock(ZOOM_OUT_FULL)) {
    case LENS_OK:

      switch(lens->setFocusAbsBlock(FOCUS_OUT_FULL)) {
      case LENS_OK:
	return(reply);
      case LENS_FAIL:
	cerr << "[hiResSensor] ERROR: Could not focus out for undeployment!" << endl;
	sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_LENS_FOCUS_FAILED;
	sensorStatus = reply->status = SENSOR_MALFUNCTION;
	return(reply);
      case LENS_CONTROLLER_CMD_OUT_OF_BOUNDS:
	cerr << "[hiResSensor] ERROR: Could not focus out for undeployment (command out of bounds)!" << endl;
	reply->extendedStatus = HI_RES_SENSOR_LENS_FOCUS_CMD_OUT_OF_BOUNDS;
	reply->status = SENSOR_MALFUNCTION;
	return(reply);
      default:
	cerr << "[hiResSensor] ERROR: Could not focus out for undeployment!" << endl;
	reply->extendedStatus = HI_RES_SENSOR_LENS_FOCUS_FAILED;
	reply->status = SENSOR_MALFUNCTION;
	return(reply);
      }

    case LENS_FAIL:
      cerr << "[hiResSensor] ERROR: Could not zoom out for undeployment!" << endl;
      sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_LENS_ZOOM_FAILED;
      sensorStatus = reply->status = SENSOR_MALFUNCTION;
      return(reply);
    case LENS_CONTROLLER_CMD_OUT_OF_BOUNDS:
      cerr << "[hiResSensor] ERROR: Could not zoom out for undeployment (command out of bounds)!" << endl;
      reply->extendedStatus = HI_RES_SENSOR_LENS_ZOOM_CMD_OUT_OF_BOUNDS;
      reply->status = SENSOR_MALFUNCTION;
      return(reply);
    default:
      cerr << "[hiResSensor] ERROR: Could not zoom out for undeployment!" << endl;
      reply->extendedStatus = HI_RES_SENSOR_LENS_ZOOM_FAILED;
      reply->status = SENSOR_MALFUNCTION;
      return(reply);
    }
    
  case PTU_POSITION_MISMATCH:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target acquisition position (position mismatch)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_POSITION_MISMATCH;
    return(reply);
  case PTU_ILLEGAL_COMMAND_ARGUMENT:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target acquisition position (illegal command argument)!" << endl;
    reply->status = SENSOR_MALFUNCTION;
    reply->extendedStatus = HI_RES_SENSOR_PTU_ILLEGAL_COMMAND_ARGUMENT;
    return(reply);
  case PTU_ILLEGAL_COMMAND:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target acquisition position (illegal command)!" << endl;
    reply->status = SENSOR_MALFUNCTION;
    reply->extendedStatus = HI_RES_SENSOR_PTU_ILLEGAL_COMMAND;
    return(reply);
  case PTU_ILLEGAL_POSITION_ARGUMENT:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target acquisition position (illegal position argument)!" << endl;
    reply->status = SENSOR_MALFUNCTION;
    reply->extendedStatus = HI_RES_SENSOR_PTU_ILLEGAL_POSITION_ARGUMENT;
    return(reply);
  case PTU_ILLEGAL_SPEED_ARGUMENT:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target acquisition position (illegal speed argument)!" << endl;
    reply->status = SENSOR_MALFUNCTION;
    reply->extendedStatus = HI_RES_SENSOR_PTU_ILLEGAL_SPEED_ARGUMENT;
    return(reply);
  case PTU_ACCEL_TABLE_EXCEEDED:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target acquisition position (acceleration table exceeded)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_ACCEL_TABLE_EXCEEDED;
    return(reply);
  case PTU_DEFAULTS_EEPROM_FAULT:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target acquisition position (defults eeprom fault)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_DEFAULTS_EEPROM_FAULT;
    return(reply);
  case PTU_SAVED_DEFAULTS_CORRUPTED:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target acquisition position (saved defaults corrupted)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_SAVED_DEFAULTS_CORRUPTED;
    return(reply);
  case PTU_LIMIT_HIT:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target acquisition position (limit switch hit)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_LIMIT_HIT;
    return(reply);
  case PTU_CABLE_DISCONNECTED:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target acquisition position (cable disconnected)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_CABLE_DISCONNECTED;
    return(reply);
  case PTU_ILLEGAL_UNIT_ID:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target acquisition position (illegal unit id)!" << endl;
    reply->status = SENSOR_MALFUNCTION;
    reply->extendedStatus = HI_RES_SENSOR_PTU_ILLEGAL_UNIT_ID;
    return(reply);
  case PTU_ILLEGAL_POWER_MODE:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target acquisition position (illegal power mode)!" << endl;
    reply->status = SENSOR_MALFUNCTION;
    reply->extendedStatus = HI_RES_SENSOR_PTU_ILLEGAL_POWER_MODE;
    return(reply);
  case PTU_NOT_RESPONDING:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target acquisition position (ptu not responding)!" << endl;
    sensorStatus = reply->status = SENSOR_MALFUNCTION;
    sensorExtendedStatus = reply->extendedStatus = HI_RES_SENSOR_PTU_NOT_RESPONDING;
    return(reply);
  default:
    cerr << "[hiResSensor] ERROR: PTU could not servo to target acquisition position!" << endl;
    reply->status = SENSOR_MALFUNCTION;
    reply->extendedStatus = HI_RES_SENSOR_PTU_FAILED;
    return(reply);
  }
}

/* The dataAcquisition function should simply capture data, calculate the appropriate file name 
   in which to save this new data, and save it. */
sensorReply *hiResSensor::dataAcquisition(sensorRequest *request) {
  // This function simply calls the framegrabber's capture function, 
  // figures out what the filename should be, and copies the captured
  // file to the correct location.

  time_t startTime, endTime;
  time(&startTime);

  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  strcpy(reply->sensorID, sensorID);
  reply->function = request->function;
  reply->status = SENSOR_OK;
  reply->extendedStatus = HI_RES_SENSOR_OK;
  
  char saveFileName[SENSOR_MAX_FILENAME_LENGTH];
  char command[2*SENSOR_MAX_FILENAME_LENGTH + 3];

  int sensorNum = request->sensorNum;
  int readingNum = request->data.sensorData[sensorNum].numReadings;

  if(!debugMode) {
    if(request->function == SENSOR_TARGET_ACQ_FUNCTION) {
      // Don't copy the file somewhere, just return the raw captured file.
      reply->data.sensorData[sensorNum].numReadings++;
      strcpy(reply->data.sensorData[sensorNum].readings[readingNum].filename, fg->captureHiResPPM(PXC_HI_RES_CHANNEL));
      return(reply);
    } 

    // Otherwise, we're doing a real DAQ
    memcpy(&reply->data, &request->data, sizeof(dbRecord)); // Start with reply->data = request->data

    // Read in file
    ppmRockSeg *image = new ppmRockSeg();
    if(!image->read(fg->captureHiResPPM(PXC_HI_RES_CHANNEL))) {
      cerr << "[hiResSensor] ERROR: Cannot read framegrabber image!" << endl;
      reply->status = SENSOR_MALFUNCTION;
      reply->extendedStatus = HI_RES_SENSOR_FG_FAILED;
      return(reply);
    }
    
    // Write out file
    time_t ltime;
    time(&ltime);
    sprintf(saveFileName, "%s%d.ppm", DAQ_SaveFileNameBase, (long)ltime);
    image->write(saveFileName);
    
    acqResults results;

#ifdef RED_TEST
    if(!image->redThreshhold(NULL, NULL, 0, NUM_ROW, &results)) {
#else
    if(!image->kimThreshhold(NULL, NULL, 0, NUM_ROW, &results)) {
#endif
      cerr << "[hiResSensor] ERROR: Cannot segment image!" << endl;
      reply->status = SENSOR_DAQ_FAILURE;
      reply->extendedStatus = HI_RES_SENSOR_SEGMENTATION_FAILED;
      return(reply);
    } 

    if(results.numTargets < 1) {
      cerr << "[hiResSensor] ERROR: Cannot find any targets in DAQ image!" << endl;
      reply->status = SENSOR_DAQ_FAILURE;
      reply->extendedStatus = HI_RES_SENSOR_LOST_TARGET;
      return(reply);
    }

    // Find the target with a center closest to the center of the image
    float minDistance = NUM_ROW * NUM_COL;
    int minDistanceIndex = -1;
    float tmpDistance;
    for(int i=0; i < results.numTargets; i++) {
      tmpDistance = sqrt((NUM_ROW/2 - results.targets[i].midpt.row) * (NUM_ROW/2 - results.targets[i].midpt.row) + 
			 (NUM_COL/2 - results.targets[i].midpt.column) * (NUM_COL/2 - results.targets[i].midpt.column));
      if(tmpDistance < minDistance) {
	minDistance = tmpDistance;
	minDistanceIndex = i;
      }
    }

    // Select a pixel as being definitely rock
    unsigned char *row = image->getRow(results.targets[minDistanceIndex].midpt.row);
    unsigned char *col = image->getCol(results.targets[minDistanceIndex].midpt.column);

    // Search through the row
    int firstCol = -1, lastCol = image->numCols;
    for(int i=0; i < image->numCols; i++) {
      if(row[i] == 255) {
	// Found the first rock pixel
	firstCol = i;
	for(;i < image->numCols; i++) {
	  if(row[i] == 0) {
	    // Found the last connected rock pixel
	    lastCol = i;
	  }
	}
      }
    }
    // At this point, if firstCol still == -1, then no rock pixels were found.
    
    // Search through the column
    int firstRow = -1, lastRow = image->numRows;
    for(int i=0; i < image->numRows; i++) {
      if(col[i] == 255) {
	// Found the first rock pixel
	firstRow = i;
	for(;i < image->numRows; i++) {
	  if(col[i] == 0) {
	    // Found the last connected rock pixel
	    lastRow = i;
	  }
	}
      }
    }
    // At this point, if firstRow still == -1, then no rock pixels were found.
    
    if(firstRow == -1 && firstCol == -1) {
      // There's an error. Can't find a rock pixel in the image!
      reply->status = SENSOR_DAQ_FAILURE;
      reply->extendedStatus = HI_RES_SENSOR_LOST_TARGET;
      return(reply);
    } 
    
    if(lastRow - firstRow > lastCol - firstCol) {
      reply->data.sensorData[sensorNum].readings[readingNum].optionalData.optionalData[0] = (float)((int)((lastRow - firstRow) / 2)); // optionalData[0] holds the row
      reply->data.sensorData[sensorNum].readings[readingNum].optionalData.optionalData[1] = (float)((int)(image->numCols / 2));       // optionalData[1] holds the col
    } else {
      reply->data.sensorData[sensorNum].readings[readingNum].optionalData.optionalData[0] = (float)((int)(image->numRows / 2));       // optionalData[0] holds the row
      reply->data.sensorData[sensorNum].readings[readingNum].optionalData.optionalData[1] = (float)((int)((lastCol - firstCol) / 2)); // optionalData[1] holds the col
    }
      
    /* Now we must calculate the scaling of the rock in the image, in mm/pixel in row and column directions.
       To do this, we calculate the angular size of a pixel in radians, then multiply by the target's distance. */
    float imageFOV = fieldOfView(lens->getHardwareZoom());
    float rowAngularSize = imageFOV / image->numRows;
    float colAngularSize = imageFOV / image->numCols;
    float distance = sqrt(request->data.DGPS_coord.x * request->data.DGPS_coord.x +
			  request->data.DGPS_coord.y * request->data.DGPS_coord.y +
			  request->data.DGPS_coord.z * request->data.DGPS_coord.z);
    // optionalData[2] holds the row scaling (mm/pixel)
    reply->data.sensorData[sensorNum].readings[readingNum].optionalData.optionalData[2] = rowAngularSize * (distance * 1000.0 /* mm */);
    // optionalData[3] holds the col scaling (mm/pixel) 
    reply->data.sensorData[sensorNum].readings[readingNum].optionalData.optionalData[3] = colAngularSize * (distance * 1000.0 /* mm */);
    
    strcpy(reply->data.sensorData[sensorNum].readings[readingNum].filename, saveFileName);

    reply->data.sensorData[sensorNum].numReadings++;

    database db;
    db.createUnalteredBitMask(reply->changedFields);
    reply->changedFields.sensorData[sensorNum].numReadings = DB_ALTERED;
    reply->changedFields.sensorData[sensorNum].readings[readingNum].filename = DB_ALTERED;
    reply->changedFields.sensorData[sensorNum].readings[readingNum].optionalData = DB_ALTERED;

    cerr << "********* DEBUG ****************" << endl;
    dbSensorReadingBitMask(reply->changedFields.sensorData[sensorNum].readings[readingNum]);
    cerr << "Optional data for target " << reply->data.targetID << ", sensor " << sensorNum << ", reading " << readingNum << ": ";
    for(int z = 0; z < 10; z++ ) {
      cerr << " " << reply->data.sensorData[sensorNum].readings[readingNum].optionalData.optionalData[z];
    }
    cerr << endl;

    time(&endTime);
    cerr << "[hiResSensor] *** The DAQ operation took " << (long)endTime - (long)startTime << " seconds" << endl;
    return(reply);
    
  } else {
    // We're in debug mode, so just return a "dummy" file.
    strcpy(saveFileName, HI_RES_DEBUG_IMAGE_FILE);
    memcpy(&reply->data, &request->data, sizeof(dbRecord));
    strcpy(reply->data.sensorData[sensorNum].readings[readingNum].filename, saveFileName);

    reply->data.sensorData[sensorNum].numReadings++;

    database db;
    db.createUnalteredBitMask(reply->changedFields);
    reply->changedFields.sensorData[sensorNum].numReadings = DB_ALTERED;
    reply->changedFields.sensorData[sensorNum].readings[readingNum].filename = DB_ALTERED;
    reply->changedFields.sensorData[sensorNum].readings[readingNum].optionalData = DB_ALTERED;
    return(reply);
  } // if(!debugMode)
}

/* The shutdown function should uninitialize the sensor but not delete any state information or
   internal variables. */ 
sensorReply *hiResSensor::shutdown(sensorRequest *request) {
  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  strcpy(reply->sensorID, request->sensorID);
  reply->function = request->function;
  reply->status = SENSOR_OK;
  reply->extendedStatus = HI_RES_SENSOR_OK;
  
  ptu->uninit();
  lens->uninit();

  return(reply);
}

/* The diagnostic function should run the sensor through a number of tests to search for errors, and
   return the most important error to the user. */
sensorReply *hiResSensor::diagnostic(sensorRequest *request) {
  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  strcpy(reply->sensorID, sensorID);
  reply->function = request->function;
  reply->status = SENSOR_OK;
  reply->extendedStatus = HI_RES_SENSOR_OK;

  int status; 

  // TBD: Lens: get hardware z & f, get last commanded z & f. See if they match. Get focus and zoom, and check for serial port read failures.
  // TBD: hiResPtu: get hardware angle & last commanded angle. See if they match. Also perform a servo, and check its status.
  // TBD: Framegrabber: ?? Need to add more functionality to get fg diagnostics.

  return(reply);
}

/* Toggles sensor to/from debug mode. */
sensorReply *hiResSensor::toggleDebug(sensorRequest *request) {
  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  strcpy(reply->sensorID, sensorID);
  reply->function = request->function;
  reply->status = SENSOR_OK;
  reply->extendedStatus = HI_RES_SENSOR_OK;

  debugMode = (debugMode == 0) ? 1 : 0;

  if(debugMode) {
    cerr << "[hiResSensor] Switching to debug mode...using " << HI_RES_DEBUG_IMAGE_FILE << " for image data.\n";
  } else {
    cerr << "[hiResSensor] Switching out of debug mode.\n";
  }
  
  return(reply);
}
 
/* Returns the sensor's costs */
sensorReply *hiResSensor::getCost(sensorRequest *request) {
  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  strcpy(reply->sensorID, sensorID);
  reply->function = request->function;
  reply->status = SENSOR_OK;
  reply->extendedStatus = HI_RES_SENSOR_OK;

  reply->timeCost = HI_RES_SENSOR_BASE_TIME_COST;
  reply->energyCost = HI_RES_SENSOR_BASE_ENERGY_COST;
  reply->storageCost = HI_RES_SENSOR_BASE_STORAGE_COST;

  return(reply);
}

/* Returns the sensor's workspace */
sensorReply *hiResSensor::getWorkspace(sensorRequest *request) {
  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  strcpy(reply->sensorID, sensorID);
  reply->function = request->function;
  reply->status = SENSOR_OK;
  reply->extendedStatus = HI_RES_SENSOR_OK;

  reply->maxDistanceTolerance = HI_RES_SENSOR_MAX_DISTANCE_TOLERANCE;
  reply->minDistanceTolerance = HI_RES_SENSOR_MIN_DISTANCE_TOLERANCE;
  reply->angularTolerance = HI_RES_SENSOR_ANGULAR_TOLERANCE;

  return(reply);
}
  
/* Just returns data in the status and extendedStatus fields of the reply, doesn't take any action */
sensorReply *hiResSensor::getStatus(sensorRequest *request) {
  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  strcpy(reply->sensorID, sensorID);
  reply->function = request->function;
  reply->status = sensorStatus;
  reply->extendedStatus = sensorExtendedStatus;

  return(reply);
}
 
int hiResSensor::autoFocus() {
  return(this->autoFocus(fg, lens));
}


// Private methods

// This function assumes that calcAutoFocusScore will return a higher number for a more focused image. 
int hiResSensor::autoFocus(frameGrabber *f, lensController *l) {

  cerr << "[hiResSensor] Doing autofocus..." << endl;

  int upperBounds = MAX(FOCUS_OUT_FULL, FOCUS_IN_FULL);
  int lowerBounds = MIN(FOCUS_OUT_FULL, FOCUS_IN_FULL);
  int focusStep = 10;

  long currScore, lastScore;
  unsigned char image[fg->getNumLowResRows(PXC_HI_RES_CHANNEL) * fg->getNumLowResCols(PXC_HI_RES_CHANNEL) * fg->getNumRGB_Bytes(PXC_HI_RES_CHANNEL)];

  int max;
  int maxIndex;
  int lastFocus;
  int done = 0;
  int step = 0;

  while(!done) {
    cerr << "*** Step " << step << ": searching from " << lowerBounds << " to " << upperBounds << " with step = " << focusStep << endl;
    max = 0;
    maxIndex = -1;
    lastFocus = 0;
    lastScore = 0;
    if(step % 2 == 0) {
      // Go from lower number to higher number
      for(int i = lowerBounds; i <= upperBounds; i = i + focusStep) {
	if(l->setFocusAbsBlock(i) == LENS_FAIL) {
	  cerr << "couldn't set focus abs block to " << i << " goig from lower to higher number " << endl;
	  return(0);
	}
	if(lastFocus != l->getHardwareFocus()) {
	  lastFocus = l->getHardwareFocus();
	  if(fg->lowResMono(image, PXC_HI_RES_CHANNEL)) {
	    currScore = calcAutoFocusScore(image, fg);

	    if(currScore > max) {
	      max = currScore;
	      maxIndex = i;
	    } 
	    lastScore = currScore;
	    cerr << "step " << step << ": " << i << " " << currScore << endl;
	  }
	}
      }
    } else {
      // Go from higher number to lower number
      for(int i = upperBounds; i >= lowerBounds; i = i - focusStep) {
	if(l->setFocusAbsBlock(i) == LENS_FAIL) {
	  cerr << "couldn't set focus abs block to " << i << " goig from higher to lower number " << endl;
	  return(0);
	}
	if(lastFocus != l->getHardwareFocus()) {
	  lastFocus = l->getHardwareFocus();
	  if(fg->lowResMono(image, PXC_HI_RES_CHANNEL)) {
	    currScore = calcAutoFocusScore(image, fg);
	    
	    if(currScore > max) {
	      max = currScore;
	      maxIndex = i;
	    } 
	    lastScore = currScore;
	    cerr << "step " << step << ": " << i << " " << currScore << endl;
	  }
	}
      }
    }
    if(maxIndex == -1) {
      // Failure
      cerr << "first maxIndex == -1" << endl;
      return(0);
    } else {
      // Now we only need to check the previous focusStep focus settings
      upperBounds = MIN(maxIndex + focusStep, FOCUS_IN_FULL);
      lowerBounds = MAX(maxIndex - focusStep, FOCUS_OUT_FULL);
      focusStep = (int)(focusStep / 10);
      step++;
      if(focusStep == 0) {
	done = 1;
      }
    }
  }
     
  if(maxIndex == -1) {
    // Failure
    cerr << "maxindex == -1!" << endl;
    return(0);
  } else {
    if(l->setFocusAbsBlock(maxIndex) == LENS_OK) {
      cerr << "focused on " << maxIndex << endl;
      return(1);
    } else {
      cerr << "couldn't focus on " << maxIndex << endl;
      return(0);
    }
  }        
}

// Assumes images are an unsigned char arrays of low-res mono bytes
long hiResSensor::calcAutoFocusScore(unsigned char *image, frameGrabber *fg) {
  long score = 0;
  int max = -1, min = 10000;
  int rows = fg->getNumLowResRows(PXC_HI_RES_CHANNEL),
      cols = fg->getNumLowResCols(PXC_HI_RES_CHANNEL),
      colorBytes = fg->getNumMonoBytes(PXC_HI_RES_CHANNEL);

  for(int i=0; i < (rows * cols * colorBytes); i++) { 
    if(image[i] > max) max = image[i];
    if(image[i] < min) min = image[i];
    score += abs(image[i] - image[i+1]);
  }

  return(score);
}


// teleop service routine
 
RTIBool teleopCallback(NDDSObjectInstance reply, NDDSObjectInstance request, void *userData) {
  hiResRequest *serverRequest = (hiResRequest *)request;
  hiResReply *serverReply = (hiResReply *)reply;
  serverReply->filename = (char *)calloc(SENSOR_MAX_FILENAME_LENGTH, sizeof(char));
  teleopCallbackParam *param = (teleopCallbackParam *)userData;
  hiResSensor *sensor = param->sensor;
  lensController *lens = param->lens;
  frameGrabber *fg = param->fg;
  hiResPtu *ptu = param->ptu;
  sensorRequest *sensorRequestMsg = (sensorRequest *)calloc(1, sizeof(sensorRequest));
  sensorReply *sensorReplyMsg;

  switch(serverRequest->function) {
  case TELEOP_HI_RES_DEPLOY:
    // First pan & tilt
    if((serverReply->status = ptu->servoToAngleAbs(new ptuAngle(serverRequest->pan, serverRequest->tilt))) == PTU_OK) {
      // Then zoom the lens
      if((serverReply->status = lens->setZoomAbsBlock(fovToZoomByte(serverRequest->FOV))) == LENS_OK) {
	// Then autofocus, if necessary
	if(fovToZoomByte(serverRequest->FOV) != ZOOM_OUT_FULL) {
	  serverReply->status = sensor->autoFocus();
	}
      }
    } 
    return(RTI_TRUE);
    break;
  case TELEOP_HI_RES_DAQ:
    //    sensorRequestMsg->function = SENSOR_ACQUIRE_DATA_FUNCTION;
    //    sensorReplyMsg = sensor->dataAcquisition(sensorRequestMsg);
    //    serverReply->status = sensorReplyMsg->status;
    //    strncpy(serverReply->filename, sensorReplyMsg->data.sensorData.dataFilenames, SENSOR_MAX_FILENAME_LENGTH);
    //    return(RTI_TRUE);
    return(RTI_FALSE);
    break;
  case TELEOP_HI_RES_DEPLOY_TARGET_ACQ:
    return(RTI_FALSE);
    break;
  case TELEOP_HI_RES_RESET:
    serverReply->status = ptu->reset();
    return(RTI_TRUE);
    break;
  }

  return(RTI_FALSE);
}
 


