/* FILE:    frameGrabber.h
   AUTHOR:  Michael Wagner
   CREATED: Apr 9, 1999

   DESCRIPTION: Defines the frameGrabber class, encapsulates the functionality
     of the PXC-200/L framegrabber.
*/

#ifndef __FRAME_GRABBER_H__
#define __FRAME_GRABBER_H__

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/timeb.h>
#include <sys/ioctl.h>

#include "pxc200.h"

// Assume PXC driver >= 0.21
#define PXC_CMD_BUFF_LENGTH 1024

#ifdef ACQ_IMAGE
#define PXC_HI_RES_PPM_DEV "/tmp/acqImage.ppm"
#else
#define PXC_HI_RES_PPM_DEV "/dev/pxc0Hppm"
#endif

#define PXC_LOW_RES_PPM_DEV "/dev/pxc0ppm"
#define PXC_HI_RES_PGM_DEV "/dev/pxc0Hpgm"
#define PXC_LOW_RES_PGM_DEV "/dev/pxc0pgm"

#define PXC_LOW_RES_RGB_DEV "/dev/pxc0rgb"
#define PXC_HI_RES_RGB_DEV "/dev/pxc0Hrgb"
#define PXC_LOW_RES_MONO_DEV "/dev/pxc0"
#define PXC_HI_RES_MONO_DEV "/dev/pxc0H"

#define PXC_CTL_DEV "/dev/pxc0ctl"

#define PXC_HI_RES_IMAGE_ROWS 480
#define PXC_HI_RES_IMAGE_COLS 640
#define PXC_LOW_RES_IMAGE_ROWS 240
#define PXC_LOW_RES_IMAGE_COLS 320
#define PXC_RGB_IMAGE_BYTES 3
#define PXC_MONO_IMAGE_BYTES 1
#define PXC_DEFAULT_SLEEP_TIME 1

#ifndef PXC_TMP_PPM_FILE
#define PXC_TMP_PPM_FILE "/mnt/ramdisk/pxcTmp.ppm"
#endif

#ifndef PXC_TMP_PGM_FILE
#define PXC_TMP_PGM_FILE "/mnt/ramdisk/pxcTmp.pgm"
#endif

#ifndef PXC_DRV_LOCATION
#define PXC_DRV_LOCATION "/usr/local/pxc/pxc200"
#endif


// Definitions for the Sony XC-003 camera 
#ifndef PXC_SLEEP_TIME
#define PXC_SLEEP_TIME 2
#endif

#ifndef PXC_BRIGHTNESS
#define PXC_BRIGHTNESS 68
#endif

#ifndef PXC_CONTRAST
#define PXC_CONTRAST 108
#endif

#ifndef PXC_HUE
#define PXC_HUE -89
#endif

#ifndef PXC_SAT_U
#define PXC_SAT_U 201
#endif

#ifndef PXC_SAT_V
#define PXC_SAT_V 62
#endif


#define FG_OK 0
#define FG_FAIL -1

class frameGrabberConfig {
 public:
  frameGrabberConfig(int sleepTime, int brightness, int contrast, int hue, int satU, int satV);
  int getSleepTime();
  int getBrightness();
  int getContrast();
  int getHue();
  int getSatU();
  int getSatV();
  void setSleepTime(int);
  void setBrightness(int);
  void setContrast(int);
  void setHue(int);
  void setSatU(int);
  void setSatV(int);

 private:
  int sleepTime;  // This is how long the driver will stay open for each capture.
  int brightness;
  int contrast;
  int hue;
  int satU;
  int satV;
};

class frameGrabber {
 public:
  frameGrabber(frameGrabberConfig *config);
  void setConfig(frameGrabberConfig *config);
  frameGrabberConfig *getConfig();

  /* File-based functions */
  char *captureHiResPPM(int channel); /* Returns the filename of the high-res PPM file. */
  char *captureLowResPPM(int channel); /* Returns the filename of the low-res PPM file. */
  char *captureHiResPGM(int channel); /* Returns the filename of the high-res PGM file. */
  char *captureLowResPGM(int channel); /* Returns the filename of the low-res PGM file. */

  /* Memory based functions */
  unsigned char *hiResRGB(int channel); /* Returns a pointer to a copy of the
					   image in memory, NULL on failure */
  unsigned char *lowResRGB(int channel); 
  unsigned char *hiResMono(int channel); 
  int lowResMono(unsigned char *image, int channel); // Returns 0 on failure, 1 on success
  
  unsigned int getNumHiResRows(int channel) {
    return(PXC_HI_RES_IMAGE_ROWS);
  }
  unsigned int getNumHiResCols(int channel) {
    return(PXC_HI_RES_IMAGE_COLS);
  }
  unsigned int getNumLowResRows(int channel) {
    return(PXC_LOW_RES_IMAGE_ROWS);
  }
  unsigned int getNumLowResCols(int channel) {
    return(PXC_LOW_RES_IMAGE_COLS);
  }
  unsigned int getNumMonoBytes(int channel) {
    return(PXC_MONO_IMAGE_BYTES);
  }
  unsigned int getNumRGB_Bytes(int channel) {
    return(PXC_RGB_IMAGE_BYTES);
  }

 private:
  frameGrabberConfig *config;
  unsigned char *getImage(const char *device, int numRows, 
			  int numCols, int numColorBytes, int channel); /* Returns a pointer
									  to a copy of the
									  image in memory,
									  NULL on failure */
  int getImage(unsigned char * image, const char *device, int numRows, 
	       int numCols, int numColorBytes, int channel); /* Returns 0 on failure, 1 on success */
  
};

#endif // __FRAME_GRABBER_H__
