#include <iostream.h>
#include <fstream.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <tcl.h>
#include <tk.h>
#include "spectrometer.h"

spectrometer::spectrometer() {

  // Initialize spectrometer variables
  this->numSamplesAvg = 100;
  this->tolerance = 1;
  this->darkRefExists = 0;
  this->whiteRefExists = 0;
  this->rawSpectrumExists = 0;
  this->previousSpectrumExists = 0;

  bzero(&this->darkRef, sizeof(float) * SPEC_NUM_PIXELS);
  bzero(&this->whiteRef, sizeof(float) * SPEC_NUM_PIXELS);
  bzero(&this->rawSpectrum, sizeof(float) * SPEC_NUM_PIXELS);
  bzero(&this->previousSpectrum, sizeof(float) * SPEC_NUM_PIXELS);
  bzero(&this->reflectanceSpectrum, sizeof(float) * SPEC_NUM_PIXELS);

  this->dispersion = SPEC_NUM_PIXELS / (SPEC_MAX_WAVELENGTH - SPEC_MIN_WAVELENGTH);
  this->offset = this->dispersion * SPEC_MIN_WAVELENGTH;
  this->flashDelay = 0;    // FROM JOHN: "initialized but not used"
  this->samplingFreq = 10; // FROM JOHN: "integration time in ms"
  this->correctDark = 0;   // FROM JOHN: "correct for dark current"

  this->samplesSinceDarkRef = 0;
  this->samplesSinceWhiteRef = 0;
}

spectrometer::~spectrometer() {
  if(status == SPEC_OK) {
    status = SPEC_NOT_INITIALIZED;
  }
}

void spectrometer::init() {
  // Start the device 
  sendCommand("start");
  status = selectChannel(SPEC_ACTIVE_CHANNEL);
}

int spectrometer::uninit() {
  // Start the device 
  status = SPEC_NOT_INITIALIZED;
}

int spectrometer::doSampleSpectrum(char *filename) {
  int retVal;
  if((retVal = isCalibrated()) == SPEC_CALIBRATED) {
    samplesSinceWhiteRef++;
    samplesSinceDarkRef++;
    if(doRawSpectrum() == SPEC_OK) {
      calcReflectance();
      correctSpectrum(reflectanceSpectrum);
      return(writeSpectrum(reflectanceSpectrum, filename));
    } else {
      return(SPEC_FAILED);
    }
  } else {
    return(retVal);
  }
}

int spectrometer::doWhiteRef() {
  if(optimizeIntegrationTime() == SPEC_OK) {
    if(doSpectrum(whiteRef) == SPEC_OK) {
      whiteRefExists = 1;
      samplesSinceWhiteRef = 0;
      return(SPEC_OK);
    }
  }
  whiteRefExists = 0;
  return(SPEC_FAILED);
}

int spectrometer::doDarkRef() {
  if(doSpectrum(darkRef) == SPEC_OK) {
    darkRefExists = 1;
    samplesSinceDarkRef = 0;
    return(SPEC_OK);
  } else {
    darkRefExists = 0;
    return(SPEC_FAILED);
  }
}

int spectrometer::doRawSpectrum() {
  if(doSpectrum(rawSpectrum) == SPEC_OK) {
    rawSpectrumExists = 1;
    return(SPEC_OK);
  } else {
    rawSpectrumExists = 0;
    return(SPEC_FAILED);
  }
}

int spectrometer::doSpectrum(float *spectrum) {  
  if(status != SPEC_OK) {
    return(status);
  } else {
    bzero(spectrum, sizeof(float)*SPEC_NUM_PIXELS); // Initialize spectrum.
    float tmpVal;

    for(int i=0; i < numSamplesAvg; i++) {
      inSpecStream.open(SPEC_DEVICE, ios::in);
     
      if(inSpecStream.fail()) {
	cerr << "[spectrometer] ERROR: Cannot open " << SPEC_DEVICE << " for reading!" << endl;
	return(SPEC_FAILED);
      }
  
      for(int j=0; j < SPEC_NUM_PIXELS; j++) {
	inSpecStream >> tmpVal;
	spectrum[j] += tmpVal / numSamplesAvg;
	previousSpectrum[j] = tmpVal;
	if(inSpecStream.eof()) {
	  return(SPEC_FAILED);
	}
      }

      inSpecStream.close();

    }

    previousSpectrumExists = 1;
    return(SPEC_OK);
  }
}

int spectrometer::writeSpectrum(float *spectrum, char *filename) {
  if(status != SPEC_OK) {
    return(status);
  } 
  if(spectrum == NULL || filename == NULL) {
    return(SPEC_FAILED);
  }

  fstream fileOut(filename, ios::out);

  if(fileOut.fail()) {
    cerr << "[spectrometer] ERROR: Cannot open " << filename << " for writing!" << endl;
    return(SPEC_FAILED);
  }
  
#ifdef SPEC_INTERPOLATE
  for(int i=SPEC_MIN_WAVELENGTH; i <= SPEC_MAX_WAVELENGTH; i++) {
    fileOut << i << " " << spectrum[getIndex(i)] << endl;
  }
#else
  for(int i=0; i < SPEC_NUM_PIXELS; i++) {
    fileOut << spectrum[i] << endl;
  }
#endif
  fileOut.close();
  return(SPEC_OK);
}


void spectrometer::displayStatus() {
  specStatusStream.open(SPEC_STATUS_DEVICE, ios::in);
  char tmpLine[100];
  if(specStatusStream.fail()) {
    cerr << "[spectrometer] ERROR: Could not open ADC-500 status device!" << endl;
  } else {
    specStatusStream >> tmpLine;
    cout << tmpLine << endl;
  }
  specStatusStream.close();
}


int spectrometer::isSaturated(float *spectrum) {
  /* This function checks every wavelength ("channel") to check for saturation.
     Because random noise may cause saturation at a single wavelength,
     this function returns 0 only if a certain number of adjacent
     wavelengths are saturated. */
  int num_channels_sat = 0,
    max_num_adj_sat_ch = 0;
  int last_sat_channel = -1,
    num_sat_adj_ch = 0;

  for(int i=0; i < SPEC_NUM_PIXELS; i++) {
    if(spectrum[i] >= SPEC_SATURATION) {
      num_channels_sat++;
      if(last_sat_channel + 1 == i) 
	num_sat_adj_ch++;
      if(num_sat_adj_ch > max_num_adj_sat_ch)
	max_num_adj_sat_ch = num_sat_adj_ch;
    } 
    else
      num_sat_adj_ch = 0;
  }

  /* check saturation criteria */
  if(num_channels_sat > SPEC_MIN_SAT_CHANNELS ||
     max_num_adj_sat_ch > SPEC_MIN_ADJ_SAT_CHANNELS)
    return(1);

  return(0);
}

    
int spectrometer::isValidInput(float *spectrum) {
  if(isSaturated(spectrum)) {
    return(0);
  }

  float threshold; // FROM JOHN: "tolerance threshold, precalculated"
  threshold = SPEC_SATURATION * tolerance;

  // Check through all pixels for error cases
  for(int i=0; i < SPEC_NUM_PIXELS; i++) {
    if(spectrum[i] >= SPEC_SATURATION) {
      return(SPEC_SATURATED); // Detector saturated
    } 
    if(previousSpectrumExists) {
      if(fabs(spectrum[i] - previousSpectrum[i]) > threshold) {
	return(SPEC_UNSTABLE); // Unstable input
      }
    }
  }

  return(SPEC_OK);
}

void spectrometer::calcReflectance() {
  if(rawSpectrumExists && darkRefExists && whiteRefExists) {
    for(int i=0; i < SPEC_NUM_PIXELS; i++) {
      reflectanceSpectrum[i] = (rawSpectrum[i] - darkRef[i]) / (whiteRef[i] - darkRef[i]);
    }
  }
}

void spectrometer::correctSpectrum(float *spectrum) {
  // This method should remove any pathological measurements (beyond +/- SPEC_MAX_REFLECTANCE);
  for(int i=0; i < SPEC_NUM_PIXELS; i++) {
    if(spectrum[i] < SPEC_MIN_REFLECTANCE) {
      spectrum[i] = SPEC_MIN_REFLECTANCE;
    } else if(spectrum[i] > SPEC_MAX_REFLECTANCE) {
      spectrum[i] = SPEC_MAX_REFLECTANCE;
    }
  }
}

int spectrometer::optimizeIntegrationTime(void) {
  int min = SPEC_MIN_INTEGRATION_TIME;
  int max = SPEC_MAX_INTEGRATION_TIME;
  int delta, result, saturated;
  int intTime;
  float tmpSpec[SPEC_NUM_PIXELS];

  delta = (max - min) / 4;
  intTime = (max + min) / 2;
  setIntegrationTime(intTime);

  cerr << "[spectrometer] Optimizing integration time, starting at " << intTime << " ms" << endl;

  while(delta > 0) {
    cerr << "[spectrometer] Integration time = " << intTime << " ms" << endl;
    if(doSpectrum(tmpSpec) != SPEC_OK) {
      return(SPEC_FAILED);
    }
    if(isSaturated(tmpSpec)) {
      intTime -= delta;
    } else {
      intTime += delta;
    }
    setIntegrationTime(intTime);
    delta /= 2;
  }

  intTime = 7 * intTime / 8;
  setIntegrationTime(intTime);
  cerr << "[spectrometer] Integration time set to " << intTime << " ms" << endl;
  return(SPEC_OK);
}


int spectrometer::setIntegrationTime(int time) {
  if(time < SPEC_MIN_INTEGRATION_TIME || time > SPEC_MAX_INTEGRATION_TIME) {
    return(SPEC_FAILED);
  }
  // This is a hack, but I can't get fstream operations to work with this device.
  char cmd[10];
  sprintf(cmd, "%d", time);
  return(sendCommand(cmd));
}

int spectrometer::setNumAddedFrames(int num) {
  char cmd[20];
  sprintf(cmd, "add %d", num);
  return(sendCommand(cmd));
}  

int spectrometer::sendCommand(char *cmd) {
  // TBD: This is a hack! Should try to get fstream ops working instead
  char command[strlen(cmd) + strlen(SPEC_DEVICE) + 10];
  sprintf(command, "echo %s > %s", cmd, SPEC_DEVICE);
  if(system(command) == 0) {
    return(SPEC_OK);
  } else {
    return(SPEC_FAILED);
  }
  return(SPEC_OK);
}

int spectrometer::getIndex(int wavelength) {
  return((int)((dispersion * wavelength) - offset));
}

int spectrometer::isCalibrated() {
  if(!(whiteRefExists && samplesSinceWhiteRef < SPEC_MAX_SAMPLES_UNTIL_WHITE_CALIBRATION)) {
    return(SPEC_NO_WHITE_REF);
  }
  if(!(darkRefExists && samplesSinceDarkRef < SPEC_MAX_SAMPLES_UNTIL_DARK_CALIBRATION)) {
    return(SPEC_NO_DARK_REF);
  }
  return(SPEC_CALIBRATED);
}

int spectrometer::selectChannel(int channel) {
  if(channel < 0 || channel >= SPEC_NUM_CHANNELS) {
    return(SPEC_FAILED);
  }
  char cmd[20]; 
  sprintf(cmd, "channel %d", channel);
  return(sendCommand(cmd));
}

#ifdef MAIN

spectrometer spec; // The global spectrometer object used in main() and Tk_Main().

int main(int argc, char *argv[]) {
  spec.init();

  if(argc == 2) {
    if(strcmp(argv[1], "-nw") == 0) {
      char filename[1024];
      int t; 
      char c;
      cout << "Ready dark reference (hit 'c [enter]' to continue) ";
      cin >> c;
      if((t = spec.doDarkRef()) != SPEC_OK) {
	cerr << "[spectrometer] ERROR: doDarkRef returned " << t << endl;
	return(-1);
      }
      cout << "Ready white reference (hit 'c [enter]' to continue) ";
      cin >> c;
      if((t = spec.doWhiteRef()) != SPEC_OK) {
	cerr << "[spectrometer] ERROR: doWhiteRef returned " << t << endl;
	return(-1);
      }
      cout << "Enter sample filename: ";
      cin >> filename;
      cout << "Ready sample (hit 'c [enter]' to continue ";
      cin >> c;
      if((t = spec.doSampleSpectrum(filename)) != SPEC_OK) {
	cerr << "[spectrometer] ERROR: doSampleSpectrum returned " << t << endl;
	return(-1);
      } else {
	cerr << "[spectrometer] Success!" << endl;
	return(0);
      }
    } 
  }

  Tk_Main(argc, argv, (Tcl_AppInitProc *)Tcl_AppInit); /* Should never return */

  return(0);
}


/* All of these return the result of spec.do*() */
int TclCmd_doWhiteRef(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {
  sprintf(interp->result, "%d", spec.doWhiteRef());
  return TCL_OK;
}

int TclCmd_doDarkRef(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {
  sprintf(interp->result, "%d", spec.doDarkRef());
  return TCL_OK;
}

int TclCmd_doSampleSpectrum(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {
  if(argc != 2) {
    strcpy(interp->result, "Usage: doSampleSpectrum filename");
    return TCL_ERROR;
  } else {
    sprintf(interp->result, "%d", spec.doSampleSpectrum(argv[1]));
    return TCL_OK;
  }
}


Tcl_AppInit(Tcl_Interp *interp) {
  /*
   * Initialize package
   * Tcl_Init sets up the Tcl library facility.
   */
  if(Tcl_Init(interp) == TCL_ERROR) {
    return(TCL_ERROR);
  }

  if (Tk_Init(interp) == TCL_ERROR) {
    printf ("Tk_Init failed!\n");
    return TCL_ERROR;
  }

  Tcl_StaticPackage(interp, "Tk", Tk_Init, (Tcl_PackageInitProc *) NULL);

  /*
   * Register application-specific commands.
   */
  Tcl_CreateCommand(interp, "doWhiteRef", TclCmd_doWhiteRef,
		    (ClientData)NULL, (Tcl_CmdDeleteProc *)NULL);
  Tcl_CreateCommand(interp, "doDarkRef", TclCmd_doDarkRef,
		    (ClientData)NULL, (Tcl_CmdDeleteProc *)NULL);
  Tcl_CreateCommand(interp, "doSampleSpectrum", TclCmd_doSampleSpectrum,
		    (ClientData)NULL, (Tcl_CmdDeleteProc *)NULL);

  return TCL_OK;
}


#endif /* MAIN */
