#include "r3Vector.h"

InvalidIndexException iie;


r3Vector::r3Vector(double p0,double p1, double p2) {
  v[0]=p0;
  v[1]=p1;
  v[2]=p2;
}

double r3Vector::length(void) {
  return sqrt(v[0]*v[0]+v[1]*v[1] + v[2]*v[2]);
}

void r3Vector::normalize(void) {
  double l =length();
  if (l==0) 
    return;
  v[0]/=l;
  v[1]/=l;
  v[2]/=l;
}


void r3Vector::rotate(double roll, double pitch, double yaw) {
  double ca, sa,cb,sb,cg,sg;
  double nx,ny,nz;
  ca = cos(yaw);
  sa = sin(yaw);
  cb= cos(pitch);
  sb= sin(pitch);
  cg = cos(roll);
  sg= sin(roll);
  
  nx = ca*cb*v[0] + (ca*sb*sg - sa*cg)*v[1] + (ca*sb*cg + sa*sg)*v[2];
  ny = sa*cb*v[0] + (sa*sb*sg + ca*cg)*v[1] + (sa*sb*cg - ca*sg)*v[2];
  nz = -sb* v[0] + (cb*sg)*v[1] + (cb*cg)*v[2];
  v[0]=nx;
  v[1]=ny;
  v[2]=nz;
}

ostream &operator<<(ostream &str, r3Vector &v) {
  str << "[" << v.v[0] << " ,"  << v.v[1] << " ," << v.v[2] << "]";
  return str;
}

istream &operator>>(istream &s, r3Vector &in) {
  char c = 0;
  s >> c;
  if (c!='[') {
    s.clear(ios::badbit);
    return s;
  }
  s >> in[0];
  s >> c;
  if (c!=',') {
    s.clear(ios::badbit);
    return s;
  }
  s >> in[1];
  s >> c;
  if (c!=',') {
    s.clear(ios::badbit);
    return s;
  }
  s >> in[2];
  s >> c;
  if (c!=']') {
    s.clear(ios::badbit);
    return s;
  }
  return s;

}

r3Vector operator +(const r3Vector& rhs) {
  r3Vector result(rhs[0],rhs[1],rhs[2]);//v[0],v[1],v[2]);
  //  result[0]+=rhs[0];
  //  result[1]+=rhs[1];
  //  result[2]+=rhs[2];
  return result;
}

r3Vector operator -(const r3Vector& rhs) {
  r3Vector result(-rhs[0],-rhs[1],-rhs[2]);//v[0],v[1],v[2]);
  //  result[0]-=rhs[0];
  //  result[1]-=rhs[1];
  //  result[2]-=rhs[2];
  return result;
}

r3Vector & r3Vector::operator +=(const r3Vector& rhs) {
  v[0]+=rhs[0];
  v[1]+=rhs[1];
  v[2]+=rhs[2];
  return *this;
}

r3Vector & r3Vector::operator -=(const r3Vector& rhs) {
  v[0]-=rhs[0];
  v[1]-=rhs[1];
  v[2]-=rhs[2];
  return *this;
}

double& r3Vector::operator[](int i) {
  if (i>-1 && i<3)
    return v[i];
  else {
    cerr << "\nERROR: r3Vector index out of range: index = " << i << endl;
    throw iie;
  }
}

const double& r3Vector::operator[](int i) const {
  if (i>-1 && i<3)
    return v[i];
  else {
    cerr << "\nERROR: r3Vector index out of range: index = " << i << endl;
    throw iie;
  }
}

r3Vector operator+(const r3Vector & a, const r3Vector & b) {
  r3Vector result(a[0],a[1],a[2]);
  result[0]+=b[0];
  result[1]+=b[1];
  result[2]+=b[2];
  return result;
}

r3Vector operator-(const r3Vector & a, const r3Vector & b) {
  r3Vector result(a[0],a[1],a[2]);
  result[0]-=b[0];
  result[1]-=b[1];
  result[2]-=b[2];
  return result;
}

r3Vector operator*(const double a, const r3Vector &b) {
  r3Vector result(b[0],b[1],b[2]);
  result[0]*=a;
  result[1]*=a;
  result[2]*=a;
  return result;
}

r3Vector operator*(const r3Vector & a, const double &b) {
  r3Vector result(a[0],a[1],a[2]);
  result[0]*=b;
  result[1]*=b;
  result[2]*=b;
  return result;
}

r3Vector& r3Vector::operator=(const r3Vector rhs) {
  if (this ==&rhs) return *this;
  v[0]=rhs[0];
  v[1]=rhs[1];
  v[2]=rhs[2];
  return *this;
}

cylindricalCoord r3Vector::toCylindrical(double Ox,double Oy,double Oz) {
  double tx, ty,tz;
  cylindricalCoord ret;
  tx= v[0]-Ox;
  ty=v[1]-Oy;
  tz=v[2]-Oz;
  ret.z=tz;
  if (tx==0 && ty==0) {
    ret.theta =0;
  } else {
    ret.theta=atan2(ty,tx);
  }
  ret.r= sqrt(tx*tx + ty*ty);
  return ret;
}

void r3Vector::setFromCylindrical(double Ox,double Oy,double Oz, cylindricalCoord c) {
  double tx,ty,tz;
  tx = c.r*cos(c.theta);
  ty = c.r*sin(c.theta);
  tz = c.z;
  if (fabs(tx) < 1e-15) 
    tx=0;
  if (fabs(ty) < 1e-15)
    ty=0;
  v[0]=Ox + tx;
  v[1]=Oy + ty;
  v[2]=Oz + tz;
}

double  r3Vector::dot(r3Vector& on) {
  double ret;
  ret = on[0]*v[0] +on[1]*v[1] + on[2]*v[2] ;
  return ret;
}
