/* FILE:        daqCard.cpp
   AUTHOR:      Michael Wagner
   CREATED:     August 27, 1999
   DESCRIPTION: This file implements the daqCard class, which is used
                as an interface to an NI-DAQ series E card. These methods
		are designed to be fast, and that's why even though most of
		the methods are similar (except for buffer variable types)
		I don't do a lot of function calls to generic read methods
		and then typecast. I've also set up the >> operator to make
		usage very easy.
*/	

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <iostream.h>
#include <fcntl.h>
#include <asm/ioctl.h>
#include <asm/types.h>
#include <sys/ioctl.h>
#include "ndds/NDDS.h"
#include "nicts.h"
#include "daqCard.h"
#include "daqCardDef.h"

daqCard::daqCard() : ostream() {
  fd = -1;
  channel = 0;
}

int daqCard::open() {
  fd = ::open(DAQ_CARD_INPUT_DEVICE, O_RDWR); // Call the default stdio C version of open, not a class method
  if(fd < 0) {
    return(DAQ_CARD_FAIL);
  } else {
    return(DAQ_CARD_OK);
  }
}

void daqCard::close() {
  if(fd >= 0) {
    ::close(fd); // Call the default stdio C version of close, not a class method
  }
}

int daqCard::setChannel(int channel) {
  if(channel < 0 || channel >= DAQ_CARD_NUM_CHANNELS) {
    return(0);
  } else {
    this->channel = channel;
    return(1);
  }
}

ostream &daqCard::operator>>(float &f) {
  short binval;
  device_info dinfo;

  ioctl(fd, NICTL_DEVICE_INFO, &dinfo);
  ioctl(fd, NICTL_AI_RESET, 0);
  ioctl(fd, NICTL_AI_ADDCH, 0x10008000 | ((channel&0xF) <<16));  
  ioctl(fd, NICTL_AI_ONEPT_SETUP, 0);
  ioctl(fd, NICTL_AI_CONVERT, 0);

  if(::read(fd, &binval, 2) < 2) {
    return(*this);
  }

  if(dinfo.aires == 12) {
    f = binval * DAQ_CARD_SIGNAL_CONVERSION; 
  } else {
    if(dinfo.aires == 16) {
      f = binval * DAQ_CARD_SIGNAL_CONVERSION; 
    } else {
      return(*this);
    }
  }

  return(*this);
}

ostream &daqCard::operator>>(double &d) {
  short binval;
  device_info dinfo;

  ioctl(fd, NICTL_DEVICE_INFO, &dinfo);
  ioctl(fd, NICTL_AI_RESET, 0);
  ioctl(fd, NICTL_AI_ADDCH, 0x10008000 | ((channel&0xF) <<16));  
  ioctl(fd, NICTL_AI_ONEPT_SETUP, 0);
  ioctl(fd, NICTL_AI_CONVERT, 0);

  if(::read(fd, &binval, 2) < 2) {
    return(*this);
  }

  if(dinfo.aires == 12) {
    d = binval * DAQ_CARD_SIGNAL_CONVERSION; 
  } else {
    if(dinfo.aires == 16) {
      d = binval * DAQ_CARD_SIGNAL_CONVERSION; 
    } else {
      return(*this);
    }
  }

  return(*this);
}

ostream &daqCard::operator>>(int &i) {
  short binval;
  device_info dinfo;

  ioctl(fd, NICTL_DEVICE_INFO, &dinfo);
  ioctl(fd, NICTL_AI_RESET, 0);
  ioctl(fd, NICTL_AI_ADDCH, 0x10008000 | ((channel&0xF) <<16));  
  ioctl(fd, NICTL_AI_ONEPT_SETUP, 0);
  ioctl(fd, NICTL_AI_CONVERT, 0);

  if(::read(fd, &binval, 2) < 2) {
    return(*this);
  }

  if(dinfo.aires == 12) {
    i = (int)(binval * DAQ_CARD_SIGNAL_CONVERSION);
  } else {
    if(dinfo.aires == 16) {
      i = (int)(binval * DAQ_CARD_SIGNAL_CONVERSION); 
    } else {
      return(*this);
    }
  }

  return(*this);
}

ostream &daqCard::operator>>(short &s) {
  short binval;
  device_info dinfo;

  ioctl(fd, NICTL_DEVICE_INFO, &dinfo);
  ioctl(fd, NICTL_AI_RESET, 0);
  ioctl(fd, NICTL_AI_ADDCH, 0x10008000 | ((channel&0xF) <<16));  
  ioctl(fd, NICTL_AI_ONEPT_SETUP, 0);
  ioctl(fd, NICTL_AI_CONVERT, 0);

  if(::read(fd, &binval, 2) < 2) {
    return(*this);
  }

  if(dinfo.aires == 12) {
    s = (short)(binval * DAQ_CARD_SIGNAL_CONVERSION); 
  } else {
    if(dinfo.aires == 16) {
      s = (short)(binval * DAQ_CARD_SIGNAL_CONVERSION); 
    } else {
      return(*this);
    }
  }

  return(*this);
}

ostream &daqCard::operator>>(long &l) {
  short binval;
  device_info dinfo;

  ioctl(fd, NICTL_DEVICE_INFO, &dinfo);
  ioctl(fd, NICTL_AI_RESET, 0);
  ioctl(fd, NICTL_AI_ADDCH, 0x10008000 | ((channel&0xF) <<16));  
  ioctl(fd, NICTL_AI_ONEPT_SETUP, 0);
  ioctl(fd, NICTL_AI_CONVERT, 0);

  if(::read(fd, &binval, 2) < 2) {
    return(*this);
  }

  if(dinfo.aires == 12) {
    l = (long)(binval * DAQ_CARD_SIGNAL_CONVERSION); 
  } else {
    if(dinfo.aires == 16) {
      l = (long)(binval * DAQ_CARD_SIGNAL_CONVERSION); 
    } else {
      return(*this);
    }
  }

  return(*this);
}

#ifdef CANNOT_FIGURE_OUT_DAQ_DRIVER /* The ::read call never returns -- I don't know why!
				       TBD: Figure this out, because this is faster than
				       calling '>>' a bunch of times! */
int daqCard::read(float *f, int num) {
  if(f == NULL) {
    return(DAQ_CARD_NULL_BUFFER);
  } else {
    short binval[num];
    device_info dinfo;
    
    ioctl(fd, NICTL_DEVICE_INFO, &dinfo);
    ioctl(fd, NICTL_AI_RESET, 0);
    ioctl(fd, NICTL_AI_ADDCH, 0x10008000 | ((channel&0xF) <<16));  
    ioctl(fd, NICTL_AI_ONEPT_SETUP, 0);
    ioctl(fd, NICTL_AI_CONVERT, 0);
    
    if(::read(fd, &binval, num*2) < num*2) {
      return(DAQ_CARD_FAIL);
    }

    if(dinfo.aires == 12) {
      for(int i=0; i < num; i++) {
	f[i] = (float)(binval[i] * DAQ_CARD_SIGNAL_CONVERSION); 
      }
    } else {
      if(dinfo.aires == 16) {
	for(int i=0; i < num; i++) {
	  f[i] = (float)(binval[i] * DAQ_CARD_SIGNAL_CONVERSION); 
	}
      } else {
	return(DAQ_CARD_FAIL);
      }
    }
    
    return(DAQ_CARD_OK);
  }
}

int daqCard::read(double *d, int num) {
  if(d == NULL) {
    return(DAQ_CARD_NULL_BUFFER);
  } else {
    short binval[num];
    device_info dinfo;
    
    ioctl(fd, NICTL_DEVICE_INFO, &dinfo);
    ioctl(fd, NICTL_AI_RESET, 0);
    ioctl(fd, NICTL_AI_ADDCH, 0x10008000 | ((channel&0xF) <<16));  
    ioctl(fd, NICTL_AI_ONEPT_SETUP, 0);
    ioctl(fd, NICTL_AI_CONVERT, 0);
    
    if(::read(fd, &binval, num) < num) {
      return(DAQ_CARD_FAIL);
    }
    
    if(dinfo.aires == 12) {
      for(int i=0; i < num; i++) {
	d[i] = (double)(binval[i] * DAQ_CARD_SIGNAL_CONVERSION); 
      }
    } else {
      if(dinfo.aires == 16) {
	for(int i=0; i < num; i++) {
	  d[i] = (double)(binval[i] * DAQ_CARD_SIGNAL_CONVERSION); 
	}
      } else {
	return(DAQ_CARD_FAIL);
      }
    }
    
    return(DAQ_CARD_OK);
  }
}

int daqCard::read(int *i, int num) {
  if(i == NULL) {
    return(DAQ_CARD_NULL_BUFFER);
  } else {
    short binval[num];
    device_info dinfo;
    
    ioctl(fd, NICTL_DEVICE_INFO, &dinfo);
    ioctl(fd, NICTL_AI_RESET, 0);
    ioctl(fd, NICTL_AI_ADDCH, 0x10008000 | ((channel&0xF) <<16));  
    ioctl(fd, NICTL_AI_ONEPT_SETUP, 0);
    ioctl(fd, NICTL_AI_CONVERT, 0);
    
    if(::read(fd, &binval, num) < num) {
      return(DAQ_CARD_FAIL);
    }
    
    if(dinfo.aires == 12) {
      for(int j=0; j < num; j++) {
	i[j] = (int)(binval[j] * DAQ_CARD_SIGNAL_CONVERSION); 
      }
    } else {
      if(dinfo.aires == 16) {
	for(int j=0; j < num; j++) {
	  i[j] = (int)(binval[j] * DAQ_CARD_SIGNAL_CONVERSION); 
	}
      } else {
	return(DAQ_CARD_FAIL);
      }
    }
    
    return(DAQ_CARD_OK);
  }
}

int daqCard::read(long *l, int num) {
  if(l == NULL) {
    return(DAQ_CARD_NULL_BUFFER);
  } else {
    short binval[num];
    device_info dinfo;
    
    ioctl(fd, NICTL_DEVICE_INFO, &dinfo);
    ioctl(fd, NICTL_AI_RESET, 0);
    ioctl(fd, NICTL_AI_ADDCH, 0x10008000 | ((channel&0xF) <<16));  
    ioctl(fd, NICTL_AI_ONEPT_SETUP, 0);
    ioctl(fd, NICTL_AI_CONVERT, 0);
    
    if(::read(fd, &binval, num) < num) {
      return(DAQ_CARD_FAIL);
    }
    
    if(dinfo.aires == 12) {
      for(int i=0; i < num; i++) {
	l[i] = (long)(binval[i] * DAQ_CARD_SIGNAL_CONVERSION); 
      }
    } else {
      if(dinfo.aires == 16) {
	for(int i=0; i < num; i++) {
	  l[i] = (long)(binval[i] * DAQ_CARD_SIGNAL_CONVERSION); 
	}
      } else {
	return(DAQ_CARD_FAIL);
      }
    }
    
    return(DAQ_CARD_OK);
  }
}

int daqCard::read(short *s, int num) {
  if(s == NULL) {
    return(DAQ_CARD_NULL_BUFFER);
  } else {
    short binval[num];
    device_info dinfo;
    
    ioctl(fd, NICTL_DEVICE_INFO, &dinfo);
    ioctl(fd, NICTL_AI_RESET, 0);
    ioctl(fd, NICTL_AI_ADDCH, 0x10008000 | ((channel&0xF) <<16));  
    ioctl(fd, NICTL_AI_ONEPT_SETUP, 0);
    ioctl(fd, NICTL_AI_CONVERT, 0);
    
    if(::read(fd, &binval, num) < num) {
      return(DAQ_CARD_FAIL);
    }
    
    if(dinfo.aires == 12) {
      for(int i=0; i < num; i++) {
	s[i] = (short)(binval[i] * DAQ_CARD_SIGNAL_CONVERSION); 
      }
    } else {
      if(dinfo.aires == 16) {
	for(int i=0; i < num; i++) {
	  s[i] = (short)(binval[i] * DAQ_CARD_SIGNAL_CONVERSION); 
	}
      } else {
	return(DAQ_CARD_FAIL);
      }
    }
    
    return(DAQ_CARD_OK);
  }
}

#else /* CANNOT_FIGURE_OUT_DAQ_DRIVER */

int daqCard::read(float *f, int num) {
  if(f == NULL) {
    return(DAQ_CARD_NULL_BUFFER);
  } else {
    for(int i=0; i < num; i++) {
      *this >> f[i];
    }
    return(DAQ_CARD_OK);
  }
}

int daqCard::read(double *d, int num) {
  if(d == NULL) {
    return(DAQ_CARD_NULL_BUFFER);
  } else {
    for(int i=0; i < num; i++) {
      *this >> d[i];
    }
    return(DAQ_CARD_OK);
  }
}

int daqCard::read(int *i, int num) {
  if(i == NULL) {
    return(DAQ_CARD_NULL_BUFFER);
  } else {
    for(int j=0; j < num; j++) {
      *this >> i[j];
    }
    return(DAQ_CARD_OK);
  }
}

int daqCard::read(long *l, int num) {
  if(l == NULL) {
    return(DAQ_CARD_NULL_BUFFER);
  } else {
    for(int i=0; i < num; i++) {
      *this >> l[i];
    }
    return(DAQ_CARD_OK);
  }
}

int daqCard::read(short *s, int num) {
  if(s == NULL) {
    return(DAQ_CARD_NULL_BUFFER);
  } else {
    for(int i=0; i < num; i++) {
      *this >> s[i];
    }
    return(DAQ_CARD_OK);
  }
}

// TBD: Make these timed functions return execTime more precisely, right now
// the "time" function is only accurate to seconds!
int daqCard::read(float *f, int num, float daqPeriod, float *execTime) { 
  time_t startTime, endTime; 
  if(f == NULL || execTime == NULL) {
    return(DAQ_CARD_NULL_BUFFER);
  } else {
    for(int i=0; i < num; i++) {
      time(&startTime);
      *this >> f[i];
      NddsUtilitySleep(daqPeriod);
      time(&endTime);
      execTime[i] = endTime - startTime;
    }
    return(DAQ_CARD_OK);
  }
}

int daqCard::read(double *d, int num, float daqPeriod, float *execTime) {
  time_t startTime, endTime; 
  if(d == NULL || execTime == NULL) {
    return(DAQ_CARD_NULL_BUFFER);
  } else {
    for(int i=0; i < num; i++) {
      time(&startTime);
      *this >> d[i];
      NddsUtilitySleep(daqPeriod);
      time(&endTime);
      execTime[i] = endTime - startTime;
    }
    return(DAQ_CARD_OK);
  }
}

int daqCard::read(int *i, int num, float daqPeriod, float *execTime) {
  time_t startTime, endTime; 
  if(i == NULL || execTime == NULL) {
    return(DAQ_CARD_NULL_BUFFER);
  } else {
    for(int j=0; j < num; j++) {
      time(&startTime);
      *this >> i[j];
      NddsUtilitySleep(daqPeriod);
      time(&endTime);
      execTime[j] = endTime - startTime;
    }
    return(DAQ_CARD_OK);
  }
}

int daqCard::read(long *l, int num, float daqPeriod, float *execTime) {
  time_t startTime, endTime; 
  if(l == NULL || execTime == NULL) {
    return(DAQ_CARD_NULL_BUFFER);
  } else {
    for(int i=0; i < num; i++) {
      time(&startTime);
      *this >> l[i];
      NddsUtilitySleep(daqPeriod);
      time(&endTime);
      execTime[i] = endTime - startTime;
    }
    return(DAQ_CARD_OK);
  }
}

int daqCard::read(short *s, int num, float daqPeriod, float *execTime) {
  time_t startTime, endTime; 
  if(s == NULL || execTime == NULL) {
    return(DAQ_CARD_NULL_BUFFER);
  } else {
    execTime = (float *)calloc(num, sizeof(float));
    for(int i=0; i < num; i++) {
      time(&startTime);
      *this >> s[i];
      NddsUtilitySleep(daqPeriod);
      time(&endTime);
      execTime[i] = endTime - startTime;
    }
    return(DAQ_CARD_OK);
  }
}

#endif /* CANNOT_FIGURE_OUT_DAQ_DRIVER */

