/* FILE:    armSensor.cpp
   AUTHOR:  Michael Wagner
   CREATED: Aug 19, 1999

   DESCRIPTION: This file defines the sensor driver for
     Nomad's manipulator arm, armSensor. This class
     is inhereted from the virtual class sensor.h
*/

#include <iostream.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include <sys/types.h>
#include <unistd.h>
#include <pthread.h>
#include "ndds/NDDS.h"
#include "armSensor.h"
#include "sensorDef.h"
#include "database.h"
#include "SAS_Config.h"
#include "nddsMsgNames.h"
#include "telemetryMsgs.h"
#include "teleopMsgs.h"
#include "armSensorTelemetryThread.h"


// Public functions

/* The constructors should simply instantiate objects and set class variables. */
armSensor::armSensor(char *sensorID, int boxSize, int pixSize, char *DAQ_SaveFileNameBase) : sensor() {
  cerr << "[armSensor] Starting up with boxSize = " << boxSize << ", pixSize = " << pixSize << endl;
#ifdef ARM_NO_WRIST
  cerr << "[armSensor] Starting up without wrist manipulation code" << endl;
#endif
  strcpy(this->sensorID, sensorID);
  debugMode = 0;
  ims = new armCamImageScanner(boxSize, pixSize);
  arm = new MeteoriteArmController(ims);
  spectro = new spectrometer();
  metal = new metalDetector();

  this->DAQ_SaveFileNameBase = new char[strlen(DAQ_SaveFileNameBase)+1];
  strcpy(this->DAQ_SaveFileNameBase, DAQ_SaveFileNameBase);
  
  int nddsDomain;
  if(!getSAS_Config(NDDS_DOMAIN, nddsDomain)) {
    cerr << "[armSensor] ERROR: Cannot read " << NDDS_DOMAIN << " from config file!" << endl;
    return;
  }    
  float period = ARM_TELEMETRY_DEFAULT_PERIOD;
  telThread = new armSensorTelemetryThread(nddsDomain, period);
  telThread->start();

  // Set up teleop server
  armRequestNddsRegister();
  armReplyNddsRegister();
  armRequest *armRequestMsg = (armRequest *)calloc(1, sizeof(armRequest));
  armReply *armReplyMsg = (armReply *)calloc(1, sizeof(armReply));
  NddsServerCreate(TELEOP_ARM_SERVICE, NDDS_SERVER_IMMEDIATE, 10.0f,
  		   armReplyMsg, armRequestMsg,
  		   armReplyPublicationType,
  		   armRequestPublicationType,
  		   teleopCallback, arm);  
}

armSensor::armSensor(char *sensorID, char *DAQ_SaveFileNameBase) {
  cerr << "[armSensor] Starting up with default segmentation parameters" << endl;
#ifdef ARM_NO_WRIST
  cerr << "[armSensor] Starting up without wrist manipulation code" << endl;
#endif
#ifdef ARM_NO_ELECTRONICS_CONTROLLER
  cerr << "[armSensor] Starting up without electronics controller" << endl;
#endif
  strcpy(this->sensorID, sensorID);
  debugMode = 0;
  ims = new armCamImageScanner();
  arm = new MeteoriteArmController(ims);
  spectro = new spectrometer();
  metal = new metalDetector();

  this->DAQ_SaveFileNameBase = new char[strlen(DAQ_SaveFileNameBase)+1];
  strcpy(this->DAQ_SaveFileNameBase, DAQ_SaveFileNameBase);
  
  int nddsDomain;
  if(!getSAS_Config(NDDS_DOMAIN, nddsDomain)) {
    cerr << "[armSensor] ERROR: Cannot read " << NDDS_DOMAIN << " from config file!" << endl;
    return;
  }    
  float period = ARM_TELEMETRY_DEFAULT_PERIOD;
  telThread = new armSensorTelemetryThread(nddsDomain, period);
  telThread->start();
}

/* The destructor should delete objects in the sensor driver class. Deleting the objects in this class should 
   automatically call their respective uninit functions, to close serial ports and such. */
armSensor::~armSensor() {
  telThread->stop();
  delete(telThread);
  delete(ims);
  delete(arm);
  delete(spectro);
  delete(metal);
}

/* The startup function should initialize all subcomponents, and prepare the sensor for deployment. */
sensorReply *armSensor::startup(sensorRequest *request) {
  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  strcpy(reply->sensorID, sensorID);
  reply->status = SENSOR_OK;
  reply->function = request->function;

#ifdef ARM_NO_ELECTRONICS_CONTROLLER
  cerr << "[armSensor] NOT initializing electronics controller" << endl;
#else
  cerr << "[armSensor] Initializing electronics controller..." << endl;
  arm->init();
  cerr << "[armSensor] Done!" << endl;
#endif

  cerr << "[armSensor] Initializing spectrometer..." << endl;
  spectro->init();
  cerr << "[armSensor] Done!" << endl;

  cerr << "[armSensor] Initializing metal detector..." << endl;
  metal->init();
  cerr << "[armSensor] Done!" << endl;

  return(reply);
}

/* The calibration function should be used to update internal state variables with correct values. */
sensorReply *armSensor::calibration(sensorRequest *request) {
  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  strcpy(reply->sensorID, sensorID);
  reply->function = request->function;
  reply->status = SENSOR_OK;

  // TBD: Stow arm, do spectrometer calib?

  return(reply);
}

/* The deployment function should deploy the sensor to the target coordinates (request->data.DGPS_coord), 
   given a current robot location and pose (request->robotPosition). */ 
sensorReply *armSensor::deployment(sensorRequest *request) {
  cerr << "[armSensor] Target DGPS: " << endl;
  DGPSPrint(&request->data.DGPS_coord);
  cerr << "[armSensor] Robot DGPS: " << endl;
  DGPSPrint(&request->robotPosition);
  cerr << "[armSensor] Robot pose: " << endl;
  posePrint(&request->robotPose);

  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  strcpy(reply->sensorID, sensorID);
  reply->function = request->function;

#ifndef ARM_NO_ELECTRONICS_CONTROLLER

  // Move arm into position above target
  int result = arm->deployArm(request);
  if(result == ARM_OK) {

    // Get metal detector reference reading
    if(metal->getReference() != METAL_OK) {
      cerr << "[armSensor] WARNING: Could not take metal detector reference reading!" << endl;
    }

    // Move wrist down right on top of sample
    arm->deployDownFully();
      
    reply->status = SENSOR_OK;

  } else if(result == ARM_OUT_OF_WORKSPACE_ERROR) {
    cerr << "[armSensor] ERROR: target out of arm's workspace!" << endl;
    reply->status = SENSOR_OUT_OF_WORKSPACE;

  } else {
    cerr << "[armSensor] ERROR: arm->deployArm failed! Returned " << result << endl;
    reply->status = SENSOR_DEPLOYMENT_FAILURE;
  }
#endif /* ~ARM_NO_ELECTRONICS_CONTROLLER */

  cerr << "[armSensor] Done!" << endl;

  return(reply);
}

/* The unDeployment function should servo the sensor back to its startup state. */
sensorReply *armSensor::unDeployment(sensorRequest *request) {
  cerr << "[armSensor] Stowing arm..." << endl;
  
  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  strcpy(reply->sensorID, sensorID);
  reply->function = request->function;

  int result;

#ifndef ARM_NO_ELECTRONICS_CONTROLLER
  if (!arm->checkIfStowed()) {
    if((result = arm->stow()) == ARM_OK) {
      reply->status = SENSOR_OK;
    } else {
      cerr << "[armSensor] ERROR: arm->stow failed! Returned " << result << endl;
      reply->status = SENSOR_MALFUNCTION;
    }
  }
#endif

  cerr << "[armSensor] Done!" << endl;

  return(reply);
}

/* The dataAcquisition function should simply capture data, calculate the appropriate file name 
   in which to save this new data, and save it. */
sensorReply *armSensor::dataAcquisition(sensorRequest *request) {
  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  strcpy(reply->sensorID, sensorID);
  reply->function = request->function;
  reply->status = SENSOR_OK;

  char specSaveFileName[SENSOR_MAX_FILENAME_LENGTH];
  char metalSaveFileName[SENSOR_MAX_FILENAME_LENGTH];
  char saveFileName[SENSOR_MAX_FILENAME_LENGTH];
  time_t ltime;

  int result;
  int metalStatus = SENSOR_OK;
  int specStatus = SENSOR_OK;

  // Get metal detector signal reading
  if(metal->getSignal() != METAL_OK) {
    cerr << "[armSensor] WARNING: Could not take metal detector signal reading!" << endl;
    metalStatus = SENSOR_MALFUNCTION;
  } else {
    time(&ltime);
    sprintf(metalSaveFileName, "%s/metal/%d.metal", DAQ_SaveFileNameBase, (long)ltime);
    metal->writeAllData(metalSaveFileName); /* This method actually writes both the reference and signal to two different files.
					       These two filenames will be listed on two lines in the file named saveFileName. */
  }

  // Get spectrometer reading
  sprintf(specSaveFileName, "%s/spec/%d.spec", DAQ_SaveFileNameBase, (long)ltime);
  result = spectro->doSampleSpectrum(specSaveFileName);
  if(result == SPEC_NO_WHITE_REF) {
    cerr << "[armSensor] ERROR: No spectrometer white reference taken!" << endl;
    specStatus = SENSOR_NEEDS_CALIBRATION;
  } else if(result == SPEC_NO_DARK_REF) {
    cerr << "[armSensor] ERROR: No spectrometer white reference taken!" << endl;
    specStatus = SENSOR_NEEDS_CALIBRATION;
  } else if(result != SPEC_OK) {
    cerr << "[armSensor] WARNING: Could not take spectrometer reading! Returned: " << result << endl;
    specStatus = SENSOR_MALFUNCTION;
  }

  // Only report that it's broken if BOTH metal detector and spectrometer readings fail.
  if(metalStatus == SENSOR_NEEDS_CALIBRATION || specStatus == SENSOR_NEEDS_CALIBRATION) {
    reply->status = SENSOR_NEEDS_CALIBRATION;
  } else if(metalStatus != SENSOR_OK && specStatus != SENSOR_OK) {
    reply->status = SENSOR_MALFUNCTION;

  } else {

    /* Now we must construct a file that contains both the spectrometer and metal detector DAQ save file names
       on two different lines. The name of this new file will be inserted into the database. */
    sprintf(saveFileName, "%s%d.manipulator", DAQ_SaveFileNameBase, (long)ltime);    

    fstream outFile(saveFileName, ios::out);
    if(outFile.fail()) {
      cerr << "[armSensor] ERROR: Could not write manipulator data file \"" << saveFileName << "\"!" << endl;
      reply->status = SENSOR_MALFUNCTION;
      return(reply);
    }
    outFile << specSaveFileName << endl << metalSaveFileName << endl;
    outFile.close();    

    int sensorNum = request->sensorNum;
    int readingNum = request->data.sensorData[sensorNum].numReadings;

    strcpy(reply->data.sensorData[sensorNum].readings[readingNum].filename, saveFileName);
    
    reply->data.sensorData[sensorNum].numReadings++;
    
    database db;
    db.createUnalteredBitMask(reply->changedFields);
    reply->changedFields.sensorData[sensorNum].numReadings = DB_ALTERED;
    reply->changedFields.sensorData[sensorNum].readings[readingNum].filename = DB_ALTERED;

  }

  return(reply);
}

/* The shutdown function should uninitialize the sensor but not delete any state information or
   internal variables. */ 
sensorReply *armSensor::shutdown(sensorRequest *request) {
  // TBD: Do something here??
  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  strcpy(reply->sensorID, sensorID);
  reply->function = request->function;
  reply->status = SENSOR_OK;
  return(reply);
}

/* The diagnostic function should run the sensor through a number of tests to search for errors, and
   return the most important error to the user. */
sensorReply *armSensor::diagnostic(sensorRequest *request) {
  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  strcpy(reply->sensorID, sensorID);
  reply->function = request->function;

  if(arm->calibrate() == ARM_OK) {
    reply->status = SENSOR_OK;
  } else {
    reply->status = SENSOR_MALFUNCTION;
  }

  return(reply);
}


/* Toggles sensor to/from debug mode. */
sensorReply *armSensor::toggleDebug(sensorRequest *request) {
  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  strcpy(reply->sensorID, sensorID);
  reply->function = request->function;

  debugMode = (debugMode == 0) ? 1 : 0;

  if(debugMode) {
    cerr << "[armSensor] Switching to debug mode.\n";
  } else {
    cerr << "[armSensor] Switching out of debug mode.\n";
  }

  reply->status = SENSOR_OK;
  return(reply);
}

/* Returns the sensor's costs */
sensorReply *armSensor::getCost(sensorRequest *request) {
  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  strcpy(reply->sensorID, sensorID);
  reply->function = request->function;
  reply->status = SENSOR_OK;
  reply->extendedStatus = ARM_SENSOR_OK;

  reply->timeCost = ARM_SENSOR_BASE_TIME_COST;
  reply->energyCost = ARM_SENSOR_BASE_ENERGY_COST;
  reply->storageCost = ARM_SENSOR_BASE_STORAGE_COST;

  return(reply);
}

/* Returns the sensor's workspace */
sensorReply *armSensor::getWorkspace(sensorRequest *request) {
  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  strcpy(reply->sensorID, sensorID);
  reply->function = request->function;
  reply->status = SENSOR_OK;
  reply->extendedStatus = ARM_SENSOR_OK;

  reply->maxDistanceTolerance = ARM_SENSOR_MAX_DISTANCE_TOLERANCE;
  reply->minDistanceTolerance = ARM_SENSOR_MIN_DISTANCE_TOLERANCE;
  reply->angularTolerance = ARM_SENSOR_ANGULAR_TOLERANCE;

  return(reply);
}
  
/* Just returns data in the status and extendedStatus fields of the reply, doesn't take any action */
sensorReply *armSensor::getStatus(sensorRequest *request) {
  sensorReply *reply = (sensorReply *)calloc(1, sizeof(sensorReply));
  reply->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  strcpy(reply->sensorID, sensorID);
  reply->function = request->function;
  reply->status = sensorStatus;
  reply->extendedStatus = sensorExtendedStatus;

  return(reply);
}
 

// teleop service routine
 
RTIBool teleopCallback(NDDSObjectInstance reply, NDDSObjectInstance request, void *userData) {
  armRequest *serverRequest = (armRequest *)request;
  armReply *serverReply = (armReply *)reply;
  MeteoriteArmController *arm = (MeteoriteArmController *)userData;

  switch(serverRequest->function) {
  case TELEOP_ARM_MOVE:
    {
      if(arm->moveToPosition(arm->estimateRockPosition(serverRequest->row, serverRequest->col)) == ARM_OK) {
	serverReply->status = SENSOR_OK;
      } else {
	serverReply->status = SENSOR_MALFUNCTION;
      }
    } 
    return(RTI_TRUE);
    break;
  case TELEOP_ARM_STOW:
    {
      arm->stow();
      serverReply->status = SENSOR_OK;
    }
    return(RTI_TRUE);
    break;
  }

  return(RTI_FALSE);
}
 


