#include <iostream.h>
#include "armCamImageScanner.h"
#include "ppmRockSeg.h"
#include "hiResSensorDef.h"

r3Vector armCamImageScanner::findRock(frameGrabber *fg, int hres, int vres) {
  ppmRockSeg segmentor(boxSize, boxSize, pixSize, pixSize,NULL); /* There aren't top and bottom differences
							       because camera is facing ground at 90 deg */
  r3Vector *retVal = new r3Vector;
  acqResults segResults;

  if(!segmentor.read(fg->captureHiResPPM(PXC_ARM_CAM_CHANNEL))) {
    cerr << "[armCamImageScanner] ERROR: Cannot read image from arm camera!" << endl;
    (*retVal)[0] = (*retVal)[1] = -1;
    return *retVal;
  } 

#ifdef VERBOSE_SCANNER
  cerr << "[armCamImageScanner] DEBUG: writing initial image to initial.ppm" << endl;
#endif
  segmentor.write("initial.ppm");
	
#ifdef RED_TEST
  if(!segmentor.redThreshhold(NULL, NULL, 0, vres, &segResults)) { /* NOTE: These ppmRockSeg methods
								      will ignore hres and vres, because
								      they know the dimensions of the PPM
								      image. This would probably cause 
								      unexpected behavior if hres and vres
								      do not reflect the true PPM image 
								      dimensions! */
#else
  if(!segmentor.kimThreshhold(NULL, NULL, 0, vres, &segResults)) {
#endif

    cerr << "[armCamImageScanner] ERROR: Cannot segment image!" << endl;
    (*retVal)[0] = (*retVal)[1] = -1;
    (*retVal)[2] = 0;
    return *retVal;
  }
  if(segResults.numTargets < 1) {
    cerr << "[armCamImageScanner] ERROR: Cannot find any targets in image!" << endl;
    (*retVal)[0] = (*retVal)[1] = -1;
    (*retVal)[2] = 0;
    return *retVal;
  }
#ifdef VERBOSE_SCANNER  
  cerr << "[armCamImageScanner] Found " << segResults.numTargets << " target(s) in image!" << endl;
#endif
  float minDistance = vres * hres;
  int minDistanceIndex = -1;
  float tmpDistance;
  for(int i=0; i < segResults.numTargets; i++) {
    tmpDistance = sqrt((vres/2 - segResults.targets[i].midpt.row) * (vres/2 - segResults.targets[i].midpt.row) + 
		       (hres/2 - segResults.targets[i].midpt.column) * (hres/2 - segResults.targets[i].midpt.column));
    if(tmpDistance < minDistance) {
      minDistance = tmpDistance;
      minDistanceIndex = i;
    }
  }

#ifdef VERBOSE_SCANNER
  cerr << "[armCamImageScanner] DEBUG: writing results to segmented.ppm" << endl;
 #endif	
  segmentor.write("segmented.ppm");

  (*retVal)[0] = segResults.targets[minDistanceIndex].midpt.column;
  (*retVal)[1] = segResults.targets[minDistanceIndex].midpt.row;
  (*retVal)[2] = 0; 
   return *retVal;
}

#ifdef MAIN
int main(int argc, char *argv[]) {
  armCamImageScanner *cam;

  if(argc == 1) {
    cam = new armCamImageScanner();
  } else if(strcmp(argv[1], "-h") == 0) {
    cerr << "Usage: armCamImageScanner [-h] [boxSize pixSize]" << endl;
    cerr << "  boxSize - Increasing this number allows less dense clumps of" << endl;
    cerr << "            non-connected pixels to be considered one target." << endl;
    cerr << "            It is the maximum pixel distance away that two " << endl;
    cerr << "            rock pixels can be considered one target." << endl;
    cerr << "            The arm camera's default is 5" << endl;
    cerr << "  pixSize - Decreasing this number allows smaller clumps of rock" << endl;
    cerr << "            pixels to be considered targets. Increasing it means" << endl;
    cerr << "            that only bigger clumps are considered real targets." << endl;
    cerr << "            The arm camera's default is 200" << endl;
    return(0);
  } else if(argc == 3) {
    cam = new armCamImageScanner(atoi(argv[1]), atoi(argv[2]));
  } else {
    cerr << "Usage: armCamImageScanner [-h] [boxSize pixSize]" << endl;
    cerr << "  boxSize - Increasing this number allows less dense clumps of" << endl;
    cerr << "            non-connected pixels to be considered one target." << endl;
    cerr << "            It is the maximum pixel distance away that two " << endl;
    cerr << "            rock pixels can be considered one target." << endl;
    cerr << "            The arm camera's default is 5" << endl;
    cerr << "  pixSize - Decreasing this number allows smaller clumps of rock" << endl;
    cerr << "            pixels to be considered targets. Increasing it means" << endl;
    cerr << "            that only bigger clumps are considered real targets." << endl;
    cerr << "            The arm camera's default is 200" << endl;
    return(-1);
  }

#ifdef RED_TEST
  cerr << "[armCamImageScanner] Attempting to take an image using red thresholding..." << endl;
#else
  cerr << "[armCamImageScanner] Attempting to take an image using kim's thresholding..." << endl;
#endif
  frameGrabber *fg = new frameGrabber(new frameGrabberConfig(PXC_SLEEP_TIME, PXC_BRIGHTNESS, PXC_CONTRAST, PXC_HUE, PXC_SAT_U, PXC_SAT_V));

  cerr << "[armCamImageScanner] Using box size = " << cam->boxSize << ", pix size = " << cam->pixSize << endl;

  r3Vector r = cam->findRock(fg, 640, 480);
  cout << "Target is at: " << r << endl;
  delete(fg);
  return(0);
}
#endif /* MAIN */
