#include <math.h>
#include <fstream.h>
#include "r3Vector.h"
#include "r3Line.h"
#include "JointLimit.h"
#include "dbRecord.h"
#include "sensorMsgs.h"
#include "ElectronicsInterface.h"
#include "frameGrabber.h"
#include "ImageScanner.h"
#ifndef __METEORITE_ARM_CONTROLLER
#define __METEORITE_ARM_CONTROLLER

/*This code was written by Chris Urmson                                     *
 *please contact me with any problems, quesetions, random insults...        *
 *curmson@ri.cmu.edu                                                        *
 *or at 268-3978.                                                           *
 *                                                                          */

/*This file provides the interface to the arm controller for NOMAD.
  All units are in meters and degrees.
  The arm frame of reference is centered at the center of rotation of the first
  joint but on the flat plane that NOMAD is on.  The frame of reference 
  is right handed, Z points up through the axis of the joint, Y points 
  straight forward in front of the robot and X points to the right (when 
  looking in the direction of travel of the robot).
  
  The joints are numbered from proximal to distal, ie joint 1 is the joint 
  attached to NOMAD, joint 2 is in the middle of the SCARA part of the arm...
  Joints are indexed from 1 to N.

  */


/*the following parameters are the length of the first and second links of the
  NOMAD arm, the wrist offset is the distance from the centerline of the second
  link to the the camera centerline.  Lengths are from the centerpoint of 
  rotation to the following center point of rotation or camera axis.
  */
//#define NO_JOINT_3 1
//#define NO_JOINT_4 1


#define ARM_LINK_1_LENGTH 0.89376 // 89.376 cm
#define ARM_LINK_2_LENGTH 0.89281 // 89.757 cm
#define ARM_LINK_3_LENGTH 0.381 // 15 inches
#define WRIST_OFFSET 0.155 //0.20
#define FULL_LINK_2_LENGTH (sqrt(SQ(ARM_LINK_2_LENGTH)+SQ(WRIST_OFFSET)))

#define WRIST_ANGLE_ADJUST (dATAN2(WRIST_OFFSET,ARM_LINK_2_LENGTH))

#define CONTROLLER_PORT "/dev/ttyS1"

#define ARM_HEIGHT_ABOVE_FLAT_EARTH 1.2036 //1.1836//1.05  <- use this number when not on stands.
#define VERTICAL_OFFSET_BETWEEN_HORIZONTAL_LINKS 0.2286 //9.0 inches
#define WRIST_TOP_TO_CAMERA 0.275 //0.20 /* Should be CAMERA_BOTTOM_TO_WRIST_TOP, so it's positive (see DEFAULT_CAMERA_HEIGHT) */

#define DEFAULT_CAMERA_HEIGHT (ARM_HEIGHT_ABOVE_FLAT_EARTH - WRIST_TOP_TO_CAMERA - VERTICAL_OFFSET_BETWEEN_HORIZONTAL_LINKS)

/*these next parameters are the offsets of the base of the arm from the GPS 
  receiever, they are used to transform the GPS estimate of the target to the
  arms frame
  */ 
/* these need to be measured from the sensor mast to ground below arm joint 1  */
#define ARM_X_OFFSET 0.43  
#define ARM_Y_OFFSET 1.68
#define ARM_Z_OFFSET -1.84 /* Keep up to date with DGPS_SENSOR_Z */


/* the following two functions define the horizontal and vertical field of view
   of the camera.  These two definitions have a direct affect on the 
   metersPerVerticalPixel and metersPerHorizontalPixel functions
   */
#define H_FOV 92.0
#define V_FOV 69.0

/* the following definitions determine how big the captured image from the 
   camera is. These values directly affect metersPerVerticalPixel and 
   metersPerHorizontalPixel
   */
#define CAMERA_VERTICAL_RESOLUTION 480
#define CAMERA_HORIZONTAL_RESOLUTION 640

/*the following define limits for the different joints in the arm (in degrees)*/
#define LIMIT_LOWER_1 20.0
#define LIMIT_UPPER_1 180.0
#define LIMIT_LOWER_2 0.0
#define LIMIT_UPPER_2 180.0
#define LIMIT_LOWER_3 0.0
#define LIMIT_UPPER_3 89.0
#define LIMIT_LOWER_4 -60.0
#define LIMIT_UPPER_4 60.0

/*the following are settings for the arm camera's framegrabber*/
#define ARM_CAM_SLEEPTIME 1
#define ARM_CAM_BRIGHTNESS 1
#define ARM_CAM_CONTRAST 1
#define ARM_CAM_HUE 1
#define ARM_CAM_SATU 1
#define ARM_CAM_SATV 1
#define ARM_CAMERA_CHANNEL 1

/*the following set the angles that the arm joints should be set to
  when stowed and on power up*/
#define JOINT_1_STOW_ANGLE 180.0
#define JOINT_2_STOW_ANGLE 180.0
#define JOINT_3_STOW_ANGLE 3.0
#define JOINT_4_STOW_ANGLE 0.0

#define JOINT_1_SAFE_ANGLE 90.0

#ifdef __METEORITE_ARM_CONTROLLER_C
/*these global variables set the count for a zero angle at each joint
  of the arm and for the number of counts per degree for each joint of the
  arm, for unused axes leave the values 0 and 1 appropriately.*/
double GaxesZeroPositions[MAXIMUM_AXES]={4.6567,-4.2282,0,0,0,0,0,0};
double GcountsPerDegree[MAXIMUM_AXES]={1777.778,1777.778,5925.689,1,1,1,1,1};
double GanalogCountsPerDegree[MAXIMUM_AXES]= {0.0555,0.05585,0.0555,1,1,1,1,1};

#else 
extern double  *GaxesZeroPositions;
extern double *GcountsPerDegree;
extern double *GanalogCountsPerDegree;
#endif

typedef enum {
  ARM_OK,
  ARM_FAIL,
  ARM_DEPLOY_ERROR,
  ARM_NO_TARGETS_FOUND_ERROR,
  ARM_OUT_OF_WORKSPACE_ERROR,
  ARM_IK_ERROR
} ARM_STATUS;

class MeteoriteArmController {
protected:
  /*the following four varaibles contain the value that the controller wants
    theta1, theta2, theta3 and theta4 to be.*/
  double desTheta[4];

  /* the limits provide a means for delimitering the bounds of motion for the
     different axises (i know that isn't the correct spelling, but hey?)
     */
  JointLimit * limits[4];
  
  

  r3Vector rockPositionFromGPS; /*this is the estimate of the rock position 
				  from the GPS coordinate given it*/

  /*the next four variables are used to estimate the position of the rock when
    performing the visual servoing*/

  /*the following two variables are used to talk to the controller. They
    are initialized in the constructor for this class.
    */
  ElectronicsInterface *controller;
  fstream* controllerPort;

  /*the grabber is used to get a picture from the little camera and then
    figure out where the rock is in the image.*/
  frameGrabber *grabber;
  frameGrabberConfig *fgc;
  
  /*the constructor passes in an appropriate ImageScanner (or inherited base
    class to the MeteoriteArmController constructor.*/
  ImageScanner *scanner;

  /* setJointPosition???- this will set the given joint to the given theta 
     (either an absolute or relative theta)
     returns:
     0 if theta is in the valid range and the desired theta has been set
     nonzero if the joint number or the theta is not valid.
  */
  int setJointAngleAbs(int joint, double theta);
  int setJointAngleRel(int joint, double theta);
  
    
  /* getJointAngle-
     this queries the controller and asks for angle of the given joint
     returns:
     the angle of the given joint (in degrees of course)
     */
  double getJointAngle(int joint);
  


  /* transFormWorldToArmCoord-
     this takes a world coordinate and transforms it into an arm workspace 
     coordinate given the pose of the robot
     side effects:
     changes rockPositionFromGPS to the new arm frame of reference 
     version of the rock's position.
     */
  void transformWorldToArmCoord(sensorRequest *request);

  /* inverseKinematics-
     this determines the required joint angles so that the arm end effector
     will be at the X,Y coords of the given point
     side effects:
     changes desTheta[0],desTheta[1] to reflect the appropriate position
     returns:
     ARM_OK if the point is not within the workspace of the arm
     Another ARM_STATUS if the point is within the workspace of the arm
     */
  int inverseKinematics(r3Vector point); // line 10


  /* estimateRockPosition-
     this function makes a guess at were the rock would be based on the FOV 
     of the camera.  The rock is assumed to be on a flat plane at height 0.
     this function is used to generate the paralax (sp) lines that are used 
     to estimate the exact position of the rock
     returns:
     a point in the XY plane (ie Z is always 0) where the rock would be if it
     were on a flat earth.
     this should throw a rock not found exception and then we should do 
     something.
     */
  r3Vector estimateRockPosition(void); // line 162

  /* metersPerVerticalPixel-
     metersPerHorizontalPixel-
     these functions return the distance at 0 height that a pixel in either
     the vertical or horizontal direction represents.
     returns: 
     the number of meters a pixel represents at 0 height
     */
  double metersPerVerticalPixel(void);  // line 145
  double metersPerHorizontalPixel(void); // line 153


  /* determineHeight-
     this function determines the height of the end effector camera above the 
     flat plane NOMAD is on (in the arm coordinate space)
     returns:
     height based on the value of theta3
     */
  double determineHeight(void); //line 210
  

  /* scanImage-
     this function talks to the image grabber hardware, and gets the coordinate
     of the rock in that image.
     returns:
     a point in the XY plane which is where the rock is in the image.
     */
  r3Vector scanImage(void);


  /*determineThetaForHeight-
    this function figures the required theta to move the end of the arm to the given
    height.
    returns:
    the angle for joint 3 (in degrees)
    */
  double determineThetaForHeight(double height);

public:
  /*checiIfStowed-
    this is a routine to quickly check if the arm is properly stowed
    returns 1 if the arm is stowed, 
            0 otherwise
    */
  int checkIfStowed(void) {

    return !((fabs(getJointAngle(3)-JOINT_3_STOW_ANGLE)>5.0)|| 
	    (fabs(getJointAngle(2)-JOINT_2_STOW_ANGLE)>5.0) ||
	     (fabs(getJointAngle(1)-JOINT_1_STOW_ANGLE)>5.0));
	     //	    (fabs(getJointAngle(1)-JOINT_1_STOW_ANGLE)>5.0));
      }

  /* getCurrentPosition-
     this performs the forward kinematics needed to determine the position
     of the camera
     returns:
     r3Vector containing the current position of the camera
     */
  r3Vector getCurrentPosition(void);


  /* MeteoriteArmController-
     this is a public no argument constructor for the class. It opens the serial port
     initializes the controller etc.
     */
  MeteoriteArmController(ImageScanner *ims);

  /*~MeteoriteArmController-
    the public destructor frees up the serial port and closes the controller.
    */
  ~MeteoriteArmController();

  /* init-
     This method actually starts communication with the controller initializes it, etc.
     This method allows the class constructor to be called without needing a connection
     to the electronics controller.
  */
  void init();


  /* deployArm- 
     this moves the arm to a DGPS location and lowers the wrist 
     _halfway down to the target_
     returns:
     ARM_OK if all is ok 
     !ARM_OK if fault.
     */
  int deployArm(sensorRequest *request);

  /*deployPractice-
    this is the same as deploy arm except that the position is given
    in the coordinate space of the arm.
    returns:
    ARM_OK if all is ok 
    !ARM_OK if fault.
    */
  int deployPractice(double x, double y);

  /*deployDownFully-
    Once deployPractice or deployArm has been called, this method sends the wrist
    down all the way until the touch sensor trips a kill switch.
  */
  void deployDownFully();
    
  /*putArmAt-
    this is a testing routine that puts the arm at a given x,y coordinate
    in the arm coordinate space. 
    returns:
    ARM_OK if all is ok
    !ARM_OK if some fault has occured.
    */
  int putArmAt(double x, double y);

  /*setArmHeight
    puts the camera at the given height in the workspace*/
  void setArmHeight(double z);
  
  /* stow- 
     this puts the arm into a stored position
     returns:
     1 (one) if all is ok  <-------- TBD (MDW 11/2): does it return ANYTHING?
     non-one if some fault has occured.
     */
  int stow(void);

  /* calibrate- 
     this runs the calibration routine
     returns:
     0 (zero) if succesful
     nonzero if aborted or faulted.
     */
  /*unstow-
    this moves the arm to a safe position where it can move to other */
  int unstow(void);


  int calibrate(void);

  /* isWithinWorkspace-
     this runs the inverse kinematics to see if the
     target is within the workspace of the arm
     returns:
     0 (zero) if the point is not in the workspace of the arm
     nonzero if the point is in the workspace.
  */
  int isWithinWorkspace(sensorRequest *req);

  /* estimateRockPosition-
     this function makes a guess at were the rock would be based on the FOV 
     of the camera and a user-defined rol and column.  The rock is assumed 
     to be on a flat plane at height 0. this function is used to generate
     the paralax (sp) lines that are used to estimate the exact position of the rock.
     this is just about the same as the private version of estimateRockPosition(void).
     returns:
     a point in the XY plane (ie Z is always 0) where the rock would be if it
     were on a flat earth.
     */
  r3Vector estimateRockPosition(int row, int col); // line 162

  /* moveToPosition-
     this function moves the arm to the x,y location given in the argument
     returns:
     ARM_OK if all is ok
     !ARM_OK if there is a fault
     */
  int moveToPosition(r3Vector pos); // line 240

};

class RockNotFoundException {
};


#endif //__METEORITE_ARM_CONTROLLER
