/*This code was written by Chris Urmson                                     *
 *please contact me with any problems, quesetions, random insults...        *
 *curmson@ri.cmu.edu                                                        *
 *or at 268-8919.                                                           *
 *                                                                          */
#define __METEORITE_ARM_CONTROLLER_C
#include "ChrisDefaultInclude.h"
#include "MeteoriteArmController.h"


#ifdef ARM_PUBLISH_TELEMETRY
// The telemetry structure
extern armDerivedState *p_armDerived;

#ifndef
#define DEG_TO_RAD(_x_) (_x_ * 3.1415927 / 180.0)
#endif

#endif // ARM_PUBLISH_TELEMETRY

int MeteoriteArmController::inverseKinematics(r3Vector point) {
  double c2plus;/* terms with plus in them are angles with the 
		   l3 offset included into the triangle - to get the
		   true joint angle atan2(l3,l2) needs to be removed from the
		   angle.*/
  double s2plusa, s2plusb;
  
  /* terms ending in a or b are the two possibilities for solutions as there
     are for most points (in the workspace)two possible ways of reaching them.
  */
  /* terms beginning with 'c' are cosines, terms beginning with 's' are sines
     'k's are substitution terms similar to those in Introduction to Robotics 
     by Craig pgs 124, 125*/
  double c2,s2a, s2b; 
  double theta2a, theta2b;
  double theta1a, theta1b;
  double k1a,k1b, k2a, k2b;
  
  double x,y; /*local holders so the code looks pretty*/
  double l1, l2, l3;
  
  x = point[0];
  y = point[1];
  
  l1 = ARM_LINK_1_LENGTH;
  l2 = FULL_LINK_2_LENGTH;//ARM_LINK_2_LENGTH;
  //  l3 = WRIST_OFFSET;
  
  c2plus = (SQ(x) +SQ(y) - SQ(l1) - SQ(l2))/(2*l1 * l2);
  
  if (fabs(c2plus)>1.0) {
    /* if this is the case then we are not within the workspace of the arm...
       dope.
    */    
    cerr << "Target out of reach!\n";
    cerr << "in MeteoriteArmController.cpp - inverseKinematics\n";
    return ARM_OUT_OF_WORKSPACE_ERROR;
  }

  /* get both possible theta2s by solving for the + and - sqrt of c^2+s^2 =1 */
  s2plusa = sqrt(1-SQ(c2plus));
  s2plusb = -sqrt(1-SQ(c2plus));
  
  /* now actually get the two possibilities for theta2 by subtracting off the 
     error produced by the offset*/
  theta2a = dATAN2(s2plusa,c2plus);//- WRIST_ANGLE_ADJUST;
  theta2b = dATAN2(s2plusb,c2plus);// - WRIST_ANGLE_ADJUST;
 
  c2 = dCOS(theta2a);
  s2a = dSIN(theta2a);
  s2b = dSIN(theta2b);

  k1a = l1 + l2 * c2;// l3 * s2a;
  k1b = l1 + l2 * c2;// + l3 * s2b;

  k2a = l2 * s2a;// - l3 * c2;
  k2b = l2 * s2b;// - l3 * c2;

  theta1a = dATAN2(y,x) - dATAN2(k2a,k1a);// this used to look like a deliberate sign change.
  theta1b = dATAN2(y,x) - dATAN2(k2b,k1b);


  theta2a-=WRIST_ANGLE_ADJUST;
  theta2b-=WRIST_ANGLE_ADJUST;
  /* we now have two possible solutions- {[theta1a,theta2a], [theta1b,theta2b]}
     all we have to do now is pick the more reasonable of the two...
     */

  if (limits[0]->inRange(theta1a) && limits[0]->inRange(theta1b)) {
    /* both theta1s are good...*/

    if (limits[1]->inRange(theta2a) && limits[1]->inRange(theta2b)) {
      /*all of the thetas are valid- now just pick the best one...*/
      double disa, disb;
      double cur1, cur2;
      cur1 = getJointAngle(1);
      cur2 = getJointAngle(2);
      disa = fabs(theta1a-cur1)+fabs(theta2a-cur2);
      disb = fabs(theta1b-cur1)+fabs(theta2b-cur2);
      
      if (disa<disb) {
	desTheta[0] = theta1a;
	desTheta[1] = theta2a;
      } else {
	desTheta[0] = theta1b;
	desTheta[1] = theta2b;
      }


    } else if (limits[1]->inRange(theta2a)) {
      /* soln 1 is good. */
      desTheta[0] = theta1a;
      desTheta[1] = theta2a;

    } else if (limits[1]->inRange(theta2b)) {
      /* soln 2 is good */
      desTheta[0]= theta1b;
      desTheta[1] = theta2b;

    } else {
      /*neither soln is good */
      if (!limits[1]->inRange(theta2a)) {
	cerr << "Unable to find solution to IK, 1a&b good but 2's both bad!\n";
	cerr << "in MeteoriteArmController.cpp - inverseKinematics\n";
	return ARM_IK_ERROR;
      }
    }
    /* this is where the if statement for both soln's being valid ends*/
  } else if (limits[0]->inRange(theta1a)) {
    /* the first solution is valid */
    if (!limits[1]->inRange(theta2a)) {
      cerr << "Unable to find solution to IK, 1a good but 2a bad!\n";
      cerr << "in MeteoriteArmController.cpp - inverseKinematics\n";
      return ARM_IK_ERROR;
    }
    desTheta[0] = theta1a;
    desTheta[1] = theta2a;

  } else if (limits[0]->inRange(theta1b)) {
    /* the second solution is valid */
    if (!limits[1]->inRange(theta2b)) {
      cerr << "Unable to find solution to IK, 1b good but 2b bad!\n";
      cerr << "in MeteoriteArmController.cpp - inverseKinematics\n";
      return ARM_IK_ERROR;
    }
    desTheta[0] = theta1b;
    desTheta[1] = theta2b;

  } else {
    /*neither solution is valid */
    cerr << "Neither solution to the IK is valid!\n";
    cerr << "in MeteoriteArmController.cpp - inverseKinematics\n";
    return ARM_IK_ERROR;    
  }

  return ARM_OK;
}

double MeteoriteArmController::metersPerVerticalPixel(void) {
  double height= determineHeight();
  double mpp,base;
  base = 2 * height * dTAN(V_FOV/2.0);
  mpp = base/CAMERA_VERTICAL_RESOLUTION;
  return mpp/2;
}

double MeteoriteArmController::metersPerHorizontalPixel(void) {
  double height= determineHeight();
  double mpp,base;
  base = 2 * height * dTAN(H_FOV/2.0);
  mpp = base/CAMERA_HORIZONTAL_RESOLUTION;
  return mpp/2;
}

//#define DEBUG_ROCK_ESTIMATE
r3Vector MeteoriteArmController::estimateRockPosition(void) {
  r3Vector r;
  double mpvp, mphp;
  double camTheta;
  double rockX,rockY;
  double armOffsetX, armOffsetY;
  double c,s;
  double estRockX, estRockY;
  r3Vector rockPosInImage;
  r3Vector curPosition;
  mpvp = metersPerVerticalPixel();
  mphp = metersPerHorizontalPixel();
#ifdef DEBUG_ROCK_ESTIMATE
  cerr << "mphp: " << mphp << " mpvp: "<< mpvp << "\n";
#endif 
  
  
  rockPosInImage = scanImage(); 
  if (rockPosInImage[0]==-1 && rockPosInImage[1]==-1) {
    /*we couldn't see the rock in the image
      rotate the camera and try again.*/
    double step = 15.0;
    int noRock=1;
#ifndef NO_JOINT_4
    while (limits[3]->inRange(getJointAngle(4)+step) && noRock) {
      /*step out rotations around nominal till we get what we want*/
      setJointAngleAbs(4,getJointAngle(4)+step);
      step*=-2;
      rockPosInImage = scanImage();
      if (rockPosInImage[0]!=-1) {
	noRock=0;
      }
    }
#endif
    if (noRock==1) {
      /*even by rotating the camera we couldn't see the rock...dope*/
      RockNotFoundException *rnfe=new RockNotFoundException();
      throw *rnfe;
    }	
  }
#ifndef NO_JOINT_4
  camTheta = getJointAngle(1) + getJointAngle(2) +getJointAngle(4);    
#else 
  camTheta = getJointAngle(1) + getJointAngle(2);
#endif

#ifdef DEBUG_ROCK_ESTIMATE
  cerr << "Rock in image at absolute " << rockPosInImage << "\n";
#endif
  /*transform the given rock position so that it is an offset from the 
    center of the image.*/
  rockX = rockPosInImage[0] - CAMERA_HORIZONTAL_RESOLUTION/2.0;
  /*negative is to account for 0,0 being top left not bottom left.*/
  rockY = -(rockPosInImage[1] - CAMERA_VERTICAL_RESOLUTION/2.0);


#ifdef DEBUG_ROCK_ESTIMATE
  cerr << "Rock in image at " << rockX << " " << rockY << "\n";
#endif

  /*now tranform to meters from pixels*/
  rockX = mphp * rockX;
  rockY = mpvp * rockY;

#ifdef DEBUG_ROCK_ESTIMATE
  cerr << "Rock in real offset " << rockX << " " << rockY << "\n";
#endif
  /* now we tranform to arm space coordinates*/
  
  c = dCOS(camTheta+90.0); // trust me on the 90... Something to do with 
  s = dSIN(camTheta+90.0); // with the camera being a left handed frame
  // I think.... (it worked in simulation.)

  armOffsetX = c *rockX -s*rockY;
  armOffsetY = s* rockX +c*rockY;
  
  /* coordinates so far are relevant to where the camera was located, now
     just add them to our current position*/
  curPosition = getCurrentPosition();
  estRockX = curPosition[0] - armOffsetX;
  estRockY = curPosition[1] - armOffsetY;



  r[0] = estRockX;
  r[1] = estRockY;
  r[2]= 0.0;
  //#ifdef DEBUG_ROCK_ESTIMATE
  cerr << "currently at: " << curPosition << "\n";
  cerr << "Should move to : " << r << "\n";
  
  //#endif

  return r;
}

r3Vector MeteoriteArmController::estimateRockPosition(int row, int col) {
  r3Vector r;
  double mpvp, mphp;
  double camTheta;
  double rockX,rockY;
  double armOffsetX, armOffsetY;
  double c,s;
  double estRockX, estRockY;
  r3Vector rockPosInImage;
  r3Vector curPosition;
  mpvp = metersPerVerticalPixel();
  mphp = metersPerHorizontalPixel();
#ifdef DEBUG_ROCK_ESTIMATE
  cerr << "mphp: " << mphp << " mpvp: "<< mpvp << "\n";
#endif 
  
  
  // Don't get pixels from image -- the user is passing them in
  rockPosInImage[0] = col;
  rockPosInImage[1] = row;
  rockPosInImage[2] = 0;

#ifndef NO_JOINT_4
  camTheta = getJointAngle(1) + getJointAngle(2) +getJointAngle(4);    
#else 
  camTheta = getJointAngle(1) + getJointAngle(2);
#endif

#ifdef DEBUG_ROCK_ESTIMATE
  cerr << "Rock in image at absolute " << rockPosInImage << "\n";
#endif
  /*transform the given rock position so that it is an offset from the 
    center of the image.*/
  rockX = rockPosInImage[0] - CAMERA_HORIZONTAL_RESOLUTION/2.0;
  /*negative is to account for 0,0 being top left not bottom left.*/
  rockY = -(rockPosInImage[1] - CAMERA_VERTICAL_RESOLUTION/2.0);


#ifdef DEBUG_ROCK_ESTIMATE
  cerr << "Rock in image at " << rockX << " " << rockY << "\n";
#endif

  /*now tranform to meters from pixels*/
  rockX = mphp * rockX;
  rockY = mpvp * rockY;

#ifdef DEBUG_ROCK_ESTIMATE
  cerr << "Rock in real offset " << rockX << " " << rockY << "\n";
#endif
  /* now we tranform to arm space coordinates*/
  
  c = dCOS(camTheta+90.0); // trust me on the 90... Something to do with 
  s = dSIN(camTheta+90.0); // with the camera being a left handed frame
  // I think.... (it worked in simulation.)

  armOffsetX = c *rockX -s*rockY;
  armOffsetY = s* rockX +c*rockY;
  
  /* coordinates so far are relevant to where the camera was located, now
     just add them to our current position*/
  curPosition = getCurrentPosition();
  estRockX = curPosition[0] - armOffsetX;
  estRockY = curPosition[1] - armOffsetY;



  r[0] = estRockX;
  r[1] = estRockY;
  r[2]= 0.0;
  //#ifdef DEBUG_ROCK_ESTIMATE
  cerr << "currently at: " << curPosition << "\n";
  cerr << "Should move to : " << r << "\n";
  
  //#endif

  return r;
}

double MeteoriteArmController::determineHeight(void) {
  double height;
  double t;
#ifdef NO_JOINT_3
  return DEFAULT_CAMERA_HEIGHT;
#else
  t = getJointAngle(3);
  height = DEFAULT_CAMERA_HEIGHT- 2*dSIN(t)*ARM_LINK_3_LENGTH;
  return height;
#endif
}

double MeteoriteArmController::determineThetaForHeight(double height) {
  if (height > DEFAULT_CAMERA_HEIGHT) {
    height = DEFAULT_CAMERA_HEIGHT;
  } else if (height < (DEFAULT_CAMERA_HEIGHT- 2*ARM_LINK_3_LENGTH)) {
    height = DEFAULT_CAMERA_HEIGHT-2*ARM_LINK_3_LENGTH;
  }
  return dASIN((height-DEFAULT_CAMERA_HEIGHT)/(-2*ARM_LINK_3_LENGTH));
}

r3Vector MeteoriteArmController::getCurrentPosition(void) {
  r3Vector r;
  double t1, t2;
  double c1, s1, c12,s12;
  double l1,l2,l3;

  t1 = getJointAngle(1);
  t2 = getJointAngle(2);
  l1 = ARM_LINK_1_LENGTH;
  l2 = FULL_LINK_2_LENGTH;//ARM_LINK_2_LENGTH;
  //  l3 = WRIST_OFFSET;
  c1 = dCOS(t1);
  s1 = dSIN(t1);
  c12 = dCOS(t1+t2+WRIST_ANGLE_ADJUST);
  s12 = dSIN(t1+t2+WRIST_ANGLE_ADJUST);
  r[0] = l1 *c1 + l2 *c12; //-l3 *s12;
  r[1] = l1 *s1 + l2*s12; //-l3 *c12;
  r[2] = determineHeight();
  return r;
}

int MeteoriteArmController::moveToPosition(r3Vector pos) {
  int result;
  if ((result = inverseKinematics(pos)) != ARM_OK) {
    /* we have some kind of error*/
    cerr << "Unable to move to the given position " << pos << "!\n";
    cerr << "in MeteoriteController.cpp- moveToPosition\n";
    return result;
  } 
  //  cerr << "setting Joint final joint angles to: " << desTheta[0] << ",";
  //
  cerr << desTheta[1] << "\n";
  //  return 1;
  setJointAngleAbs(2,desTheta[1]);
  setJointAngleAbs(1,desTheta[0]);
  return ARM_OK;
}

r3Vector MeteoriteArmController::scanImage(void) {
  return scanner->findRock(grabber,CAMERA_HORIZONTAL_RESOLUTION,
			   CAMERA_VERTICAL_RESOLUTION);

} 



void MeteoriteArmController::transformWorldToArmCoord(sensorRequest *request) {
  r3Vector mastPos; 
  /*first step is to move the frame of reference to a frame of reference at the DGPS receiver is at.*/
  rockPositionFromGPS[0] = request->data.DGPS_coord.x;
  rockPositionFromGPS[1] = request->data.DGPS_coord.y;
  rockPositionFromGPS[2] = request->data.DGPS_coord.z;

  mastPos[0] = request->robotPosition.x;
  mastPos[1] = request->robotPosition.y; 
  mastPos[2] = request->robotPosition.z;

  rockPositionFromGPS=rockPositionFromGPS-mastPos;
  rockPositionFromGPS.rotate(request->robotPose.roll,
			     request->robotPose.pitch,request->robotPose.yaw);
  rockPositionFromGPS[0] = rockPositionFromGPS[0]-ARM_X_OFFSET;
  rockPositionFromGPS[1] = rockPositionFromGPS[1]-ARM_Y_OFFSET;
  rockPositionFromGPS[2] = rockPositionFromGPS[2]-ARM_Z_OFFSET;

  cerr << "transformed coordinates " << rockPositionFromGPS << endl;
}


MeteoriteArmController::MeteoriteArmController(ImageScanner *ims) {
  int i;
  scanner= ims;

  /*please set the appropriate settings for the next line*/
  fgc = new frameGrabberConfig(ARM_CAM_SLEEPTIME,ARM_CAM_BRIGHTNESS,
			      ARM_CAM_CONTRAST,ARM_CAM_HUE,
			      ARM_CAM_SATU,ARM_CAM_SATV);
  controllerPort = new fstream(CONTROLLER_PORT,ios::in|ios::out);
  controller = new ElectronicsInterface(controllerPort,controllerPort,
					GaxesZeroPositions,GcountsPerDegree,
					GanalogCountsPerDegree);
  limits[0] = new JointLimit(LIMIT_LOWER_1,LIMIT_UPPER_1);
  limits[1] = new JointLimit(LIMIT_LOWER_2,LIMIT_UPPER_2);
  limits[2] = new JointLimit(LIMIT_LOWER_3,LIMIT_UPPER_3);
  limits[3] = new JointLimit(LIMIT_LOWER_4,LIMIT_UPPER_4);
  grabber = new frameGrabber(fgc);
  /* this is to deal with the robot loses power, when it starts up,
     it will stow itself*/
}


MeteoriteArmController::~MeteoriteArmController(void) {
  int i;
  controllerPort->close();
  for (i=0;i<4;i++) {
    delete limits[i];
  }
  delete controllerPort;
  delete controller;
  delete grabber;
  delete fgc;
}

void MeteoriteArmController::init(void) {
  controller->initialize();
  if (!checkIfStowed()) {
    cout << "[MeteoriteArmController] NEED TO STOW MYSELF" << endl;
    stow(); // put me back in.
  }
}

int MeteoriteArmController::setJointAngleAbs(int joint, double theta) {
  /*the controller uses zero indexed axes*/
  if (limits[joint-1]->inRange(theta)) {
    cout << "telling interface- " << joint-1 << " " << theta << "degrees\n";
    controller->moveToAbsolutePosition(joint-1,theta);
    return ARM_OK;
  } else {
    return ARM_FAIL;
  }
}

int MeteoriteArmController::setJointAngleRel(int joint, double theta) {
  /*the controller uses zero indexed axes*/
  if (limits[joint-1]->inRange(theta +this->getJointAngle(joint))) {
    controller->moveToAndWait(joint-1,theta+this->getJointAngle(joint));
    return ARM_OK;
  } else {
    return ARM_FAIL;
  }
}

double MeteoriteArmController::getJointAngle(int joint) {
  /*the controller uses zero indexed axes*/
  double angle = controller->queryPosition(joint-1);

#ifdef ARM_PUBLISH_TELEMETRY
  switch(joint-1) {
  case 0:
    p_armDerived->jointAngle0 = DEG_TO_RAD(angle);
    break;
  case 1:
    p_armDerived->jointAngle1 = DEG_TO_RAD(angle);
    break;
  case 2:
    p_armDerived->jointAngle2 = DEG_TO_RAD(angle);
    break;
  case 3:
    p_armDerived->jointAngle3 = DEG_TO_RAD(angle);
    break;
  }
#endif

  return angle;
}



int MeteoriteArmController::unstow(void) {
  setJointAngleAbs(1,JOINT_1_SAFE_ANGLE);
}

int MeteoriteArmController::stow(void) {
#ifndef NO_JOINT_3
  if (fabs(getJointAngle(3)-JOINT_3_STOW_ANGLE)>5.0) {
  cout << "moving joint 3 to stow\n";
  setJointAngleAbs(3,JOINT_3_STOW_ANGLE);
  }
#endif
#ifndef NO_JOINT_4
  if (fabs(getJointAngle(4)-JOINT_4_STOW_ANGLE)>5.0) {
    cout << "moving joint 4 to stow\n";
    setJointAngleAbs(4,JOINT_4_STOW_ANGLE);
  }
  
#endif
  if (fabs(getJointAngle(2)-JOINT_2_STOW_ANGLE)>5.0) {
    cout << "in stow moving 1 to safe\n";
    setJointAngleAbs(1,JOINT_1_SAFE_ANGLE);
    cout << "in stow moving 2 to stow\n";
    setJointAngleAbs(2,JOINT_2_STOW_ANGLE);
  }
  if (fabs(getJointAngle(1)-JOINT_1_STOW_ANGLE)>5.0) {
    cout << "in stow moving 1 to stow\n";
    setJointAngleAbs(1,JOINT_1_STOW_ANGLE);
  }
}


int MeteoriteArmController::putArmAt(double x, double y) {
  r3Vector r;
  r[0]=x;
  r[1]=y;
  /* check first to see if we can reach the point */
  if (inverseKinematics(r) == ARM_OK) {
    unstow();
    return moveToPosition(r);
  }
}

void MeteoriteArmController::setArmHeight(double z) {
  double t;
  t = determineThetaForHeight(z);
  setJointAngleAbs(3,t);
}

//#define DEBUG_DEPLOY_ARM
int MeteoriteArmController::deployArm(sensorRequest *request) {
  /*first, map request to arm space */
  r3Line A, B;
  r3Vector rockGuess;
  r3Vector estimate1, estimate2, position1, position2;
  char tmp;

  transformWorldToArmCoord(request);
  return deployPractice(rockPositionFromGPS[0],rockPositionFromGPS[1]);
}


//#define DEBUG_DEPLOY_PRACTICE
int MeteoriteArmController::deployPractice(double x, double y) {
  /*first, map request to arm space */
  r3Line A, B;
  r3Vector rockGuess;
  char c;
  rockPositionFromGPS[0] = x;
  rockPositionFromGPS[1]= y;
  r3Vector estimate1, estimate2, position1, position2;
  r3Vector position3, position4, estimate3, estimate4;


  unstow();
  int result;
  if ((result = moveToPosition(rockPositionFromGPS)) != ARM_OK) {
    cerr << "unable to deploy arm to position " << rockPositionFromGPS << " .\n";
    cerr << "in MeteoriteArmController.cpp - deployArm\n";
    stow();
    return result;
  } 
  
  try {
    estimate1 = estimateRockPosition();
  } catch (...) {
    cerr << "couldn't see rock in the image\n";
    cerr << "in MeteoriteArmController.cpp - deployArm\n";
    stow();
    return ARM_NO_TARGETS_FOUND_ERROR;
  }
  position1 = getCurrentPosition();  
#ifdef DEBUG_DEPLOY_PRACTICE
  cerr << "press c[enter] to continue \n";
  cin >>c; 
#endif


  if ((result = moveToPosition(estimate1)) != ARM_OK) {
    cerr << "unable to deploy arm to position " << estimate1 << " .\n";
    cerr << "in MeteoriteArmController.cpp - deployArm\n";
    stow();
    return result;
  } 
  try {
    estimate2 = estimateRockPosition();
  } catch (...) {
    cerr << "couldn't see rock in the image\n";
    cerr << "in MeteoriteArmController.cpp - deployArm\n";
    stow();
    return ARM_NO_TARGETS_FOUND_ERROR;
  }
  position2 = getCurrentPosition();  
  
  A.from2Pts(estimate1,position1);
  B.from2Pts(estimate2,position2);
  rockGuess = A.centerTo(B);

  //#define OLD_DOWN_CODE
#define NEW_DOWN_CODE
#ifdef NEW_DOWN_CODE
  rockGuess[0] +=0.03;
  rockGuess[0] +=0.03;
  if ((result = moveToPosition(rockGuess)) != ARM_OK) {
    cerr << "unable to deploy arm to position " << rockGuess << " .\n";
    cerr << "in MeteoriteArmController.cpp - deployArm\n";
    stow();
    return result;
  } 
  setArmHeight(DEFAULT_CAMERA_HEIGHT/2.0);
  try {
    estimate3 = estimateRockPosition();
  } catch (...) {
    cerr << "couldn't see rock in the image\n";
    cerr << "in MeteoriteArmController.cpp - deployArm\n";
    stow();
    return ARM_NO_TARGETS_FOUND_ERROR;
  }
  position3 = getCurrentPosition();



  rockGuess[0] -=0.06;
  rockGuess[0] -=0.06;
  if ((result = moveToPosition(rockGuess)) != ARM_OK) {
    cerr << "unable to deploy arm to position " << rockGuess << " .\n";
    cerr << "in MeteoriteArmController.cpp - deployArm\n";
    stow();
    return result; 
  } 
  setArmHeight(DEFAULT_CAMERA_HEIGHT/2.0);
  try {
    estimate4 = estimateRockPosition();
  } catch (...) {
    cerr << "couldn't see rock in the image\n";
    cerr << "in MeteoriteArmController.cpp - deployArm\n";
    stow();
    return ARM_NO_TARGETS_FOUND_ERROR;
  }
  position4 = getCurrentPosition();
  
  A.from2Pts(estimate3,position3);
  B.from2Pts(estimate4,position4);
  rockGuess = A.centerTo(B);
  if ((result = moveToPosition(rockGuess)) != ARM_OK) {
    cerr << "unable to deploy arm to position " << rockGuess << " .\n";
    cerr << "in MeteoriteArmController.cpp - deployArm\n";
    stow();
    return result;
  } 
  //  setArmHeight(0.22); NOW DONE IN deployDownFully




#endif

#ifdef OLD_DOWN_CODE
  if ((result = moveToPosition(rockGuess)) != ARM_OK) {
    cerr << "unable to deploy arm to position " << estimate2 << " .\n";
    cerr << "in MeteoriteArmController.cpp - deployArm\n";
    stow();
    return result;
  } 
  /*descend half way down*/
  setJointAngleAbs(3,determineThetaForHeight(DEFAULT_CAMERA_HEIGHT/2.0));

  try {
    estimate1 = estimateRockPosition();
  } catch (...) {
    cerr << "couldn't see rock in the image\n";
    cerr << "in MeteoriteArmController.cpp - deployArm\n";
    stow();
    return ARM_NO_TARGETS_FOUND_ERROR;
  }
  position1 = getCurrentPosition();  

  /*now make a second estimate of where the rock is and use that to fine tune*/
  A.from2Pts(estimate1,position1);
  rockGuess=A.centerTo(B);  
  
  if ((result = moveToPosition(rockGuess)) != ARM_OK) {  
    cerr << "unable to deploy arm to position " << rockPositionFromGPS << " .\n";  
    cerr << "in MeteoriteArmController.cpp - deployArm\n";  
    stow();  
    return result;  
    }
  /*now ram the thing through the ground because the contact sensor is attached to the limit switch
    for the controller which will stop the motor.
    */
  cerr << "Heading to ground 0 @" << 0.22 << "\n";
  setJointAngleAbs(3,determineThetaForHeight(0.22));
#endif  
  /*we are now deployed so all is good */

  return(ARM_OK);

}


void MeteoriteArmController::deployDownFully() {
  setArmHeight(0.22);
}


int MeteoriteArmController::isWithinWorkspace(sensorRequest *request) {
  /*first, map request to arm space */
  r3Line A, B;
  r3Vector rockGuess;
  transformWorldToArmCoord(request);
  return(inverseKinematics(rockPositionFromGPS) == ARM_OK);
}


int MeteoriteArmController::calibrate(void) {
  return(0); // TBD: IMPLEMENT THIS!!!
}
