#ifndef __ELECTRONICS_INTERFACE
#define __ELECTRONICS_INTERFACE
#include <math.h>
#include <iostream.h>

class ElectronicsInterface;
#include "PotLookUpTable.h"
/*This code was written by Chris Urmson                                     *
 *please contact me with any problems, quesetions, random insults...        *
 *curmson@ri.cmu.edu                                                        *
 *or at 268-8919.                                                           *
 *                                                                          */

/*This file provides an interface to the DMC-2000 (made by GALIL) through
  the a ostream.  The user has to provide as an argument to the constructor
  a io stream (this is so that the code could be used with either the USB or 
  serial port version

  Axes are zero indexed.

  CRITICAL POINT OF INFORMATION-
  The calling program needs to have run the approrpriate IOCTL commands to set up the serial
  port apropriately.  This means that the calling program should call something like the 
  following
  
        portstream_fd serialPortFD= openserial("/dev/ttyS1"); //substitute apropriate port here

  and then in the clean up at shutdown, the calling program must call

        closeserial(serialPortFD);

  These calls make calls to ioctl to set up the point and are absolutely critical to the port
  working as desired...  
  */



#define VELOCITY  16000 /*encoder ticks per second*/
#define ACCELERATION 300 /*encoder ticks per (second)^2 */
#define DECELERATION ACCELERATION /*set decel and accel the same*/
#define SMOOTHING_PARAM 0.1 /* between 0 and 1 see page 97 
			       of DMC 2000 Manual for details*/

#define ERROR_LIMIT (VELOCITY/100)  /* a hundredth of a second of error*/



#define MAXIMUM_AXES 8



class ElectronicsInterface {
private: 
  ostream * output;
  istream * input;
  //portstream_fd serialStream;
  /*this array holds the conversion factor for tranforming between degrees 
    and encoder counts */
  double countsPerDegree[MAXIMUM_AXES];

/*these paramaters specify the analog zero values for the axes*/
  double  axesZeroPositions[MAXIMUM_AXES];

  //  double potCountsPerDegree[MAXIMUM_AXES];
  //#ifndef __POTLOOKUPTABLE  // i'm sorry about this hack.. but it's the only way
  // to get it to compile.
  PotLookUpTable *converters[4];
  //#endif
  /* sendCommand-
     this will send a command through the stream to the controller and 
     wait for confirmation that it was successful
     returns:
     nonzero if the command was accepted
     0 if the command was not understood.
     */
  //  int sendCommand(char * command); 

public:  
  /* convertAngleToCounts-
     convertCountsToAngle-
     these functions translate angles in degrees into encoder counts and vice
     versa.
  */
  double queryPotValue(int axis);
  long queryCounts(int axis);
  long convertAngleToCounts(int axis,double angle);
  double convertCountsToAngle(int axis, long counts);
  double convertAnalogToAngle(int axis, double position);

  /* ElectronicsInterface-
     this constructor will create a new ElectronicsInterface object that
     will talk to the controller through the provided streams.
     zeros specifies the count value that is equivalent to a zero degree
     rotation of the axis, convs is an array that indicates how many
     counts are equivalent to a degree
     */
  ElectronicsInterface(ostream * out, istream * in, 
		       double *zeros, double * convs, double *analogConvs);
  
  /*sendCommand-
    this sends command that doesn't have any return expected.
    returns:
    0 if unsuccessful
    1 otherwise.
  */
  int sendCommand(char * command) {sendCommand(command,0);};
  int sendCommand(char * command, int SHOW_CONTROLLER);  

  /*queryEncPosition-
    this transforms the current encoder count position of the given axis to
    degrees and returns it.
    returns:
    the current position in degrees of the given axis
    */
  double queryEncPosition(int axis);

  /*queryPotPosition-
    this returns the position value from the potentiometers
    returns the current angle of the given axis in degrees
    */
  double queryPotPosition(int axis); 

  /*queryPosition-
    this returns the absolute position of the axis based on the Pot radings
  */
  double queryPosition(int axis) {return queryPotPosition(axis);}

  /*queryDigitalInput-
    this checks the state of a digital input bit.
    bits are base one index- ie 1 is the LSB of TI0 or input 1 on the 
    controller.
    returns:
    1 if the bit is set
    0 if the bit is not set
  */
  int queryDigitalInput(int bit);

  /*setDigitalOutput-
    clearDigitalOutput-
    these set and clear the given bit.  Bits are 1 indexed
  */
  void setDigitalOutput(int bit);
  void clearDigitalOutput(int bit);


  /*moveToPosition-
    this tells the motor controller to move the given axis to the desired 
    angle.  It returns immediately
    */     
  void moveToPosition(int axis, double angle);

#define ARM_GOT_THERE 0
#define ARM_MOVE_ERROR 1

  /* waitTillAtPosition-
     this holds in a non-busy wait until the axis reaches the given angle
     */
  int waitTillAtPosition(int axis, double angle);


  /*moveToAbsolutePosition-
    this uses the pots to figure out where the arm is absolutely and then 
    moves to the required angle
    */
  int moveToAbsolutePosition(int axis, double angle);

  int moveToAndWait(int axis, double angle){
    moveToPosition(axis,angle);
    waitTillAtPosition(axis,angle);
  }
  /*initialize-
    this talks to the controller,analog feedback mode for the appropriate 
    axes... probably there will be some kind of homing routine here.
  */
  void initialize(void);
  void burn(void);
};


class InvalidCommandException {
};


#endif
