/* FILE:    smManipulatorDriver.cpp
   AUTHOR:  Michael Wagner
   CREATED: Aug 30, 1999

   DESCRIPTION: This file implements the manipulator
     arm sensor manager device driver class, 
     smManipulatorDriver. This class is inhereted from the
     virtual class smDriver.h
*/

#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "ndds/NDDS.h"
#include "smManipulatorDriver.h"
#include "smDriverDef.h"
#include "database.h"
#include "dbDef.h"
#include "sensorDef.h"
#include "sensorMsgs.h"
#include "armSensorDef.h"

// Public functions

smManipulatorDriver::smManipulatorDriver(DGPS *currPosition, pose *currPose) : smDriver(currPosition, currPose) {
  sensorRequestMsg = (sensorRequest *)calloc(1, sizeof(sensorRequest));
  sensorRequestMsg->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
#ifdef NO_ARM
  cerr << "[smManipulatorDriver] Starting up driver without the arm -- SENS_MAN_SUCCESSFUL will always be returned" << endl;
#else
  arm = new armSensor(ARM_SENSOR_ID, ARM_DAQ_SAVE_FILE_NAME_BASE);
  arm->startup(sensorRequestMsg);
#endif
  strcpy(this->driverID, MANIPULATOR_DRIVER_ID);
  this->currPosition = currPosition;
  this->currPose = currPose;
}

smManipulatorDriver::~smManipulatorDriver() {
}

SENS_MAN_STATUS smManipulatorDriver::execute(sensManRequest *sensManRequestMsg, int driverNum, dbRecord *workingData, dbRecordBitMask *bitMask) {

#ifdef NO_ARM
  NddsUtilitySleep(3.0); // So it doesn't just immediately return.
  cerr << "[smManipulatorDriver] smManipulatorDriver::execute() called -- returning success even though arm is not present!" << endl;
  return(SENS_MAN_SUCCESSFUL);
#endif

  // TBD: Take care of calibration!!!

  if(sensManRequestMsg == NULL || workingData == NULL || bitMask == NULL) {
    return(SENS_MAN_FAILED);
  } else {

    SENS_MAN_STATUS retStatus = SENS_MAN_SUCCESSFUL;

    // *** DEPLOY ***
    
    strcpy(sensorRequestMsg->sensorID, ARM_SENSOR_ID);
    sensorRequestMsg->function = SENSOR_DEPLOY_FUNCTION;
    memcpy(&sensorRequestMsg->robotPosition, currPosition, sizeof(DGPS));
    memcpy(&sensorRequestMsg->robotPose, currPose, sizeof(pose));
    memcpy(&sensorRequestMsg->data, workingData, sizeof(dbRecord));
    sensorRequestMsg->sensorNum = driverNum;

    sensorReplyMsg = arm->deployment(sensorRequestMsg);
    if(sensorReplyMsg->status != SENSOR_OK) {
      // Failure condition.
      // TBD: real diagnosis?
      cerr << "[" << this->driverID << "] ERROR: \"" << ARM_SENSOR_ID << "\" returned " << sensorReplyMsg->status << ", extended status = " << sensorReplyMsg->extendedStatus << endl;
      retStatus = SENS_MAN_FAILED;
    }
    // If deploy fails, don't do anything else.
    if(retStatus != SENS_MAN_SUCCESSFUL) {
      return(retStatus);
    }

    // *** DAQ *** 

    strcpy(sensorRequestMsg->sensorID, ARM_SENSOR_ID);
    sensorRequestMsg->function = SENSOR_ACQUIRE_DATA_FUNCTION;
    memcpy(&sensorRequestMsg->robotPosition, currPosition, sizeof(DGPS));
    memcpy(&sensorRequestMsg->robotPose, currPose, sizeof(pose));
    memcpy(&sensorRequestMsg->data, workingData, sizeof(dbRecord));
    sensorRequestMsg->sensorNum = driverNum;

    sensorReplyMsg = arm->dataAcquisition(sensorRequestMsg);
    if(sensorReplyMsg->status != SENSOR_OK) {
      // Failure condition.
      // TBD: real diagnosis?
      cerr << "[" << this->driverID << "] ERROR: \"" << ARM_SENSOR_ID << "\" returned " << sensorReplyMsg->status << endl;
      if(sensorReplyMsg->extendedStatus == ARM_SENSOR_LOST_TARGET) {
	retStatus = SENS_MAN_TRY_AGAIN;
      } else {
	retStatus = SENS_MAN_FAILED;
      }
    }

    memcpy(workingData, &sensorReplyMsg->data, sizeof(dbRecord));
    memcpy(bitMask, &sensorReplyMsg->changedFields, sizeof(dbRecordBitMask));

    // *** UNDEPLOY ***

    strcpy(sensorRequestMsg->sensorID, ARM_SENSOR_ID);
    sensorRequestMsg->function = SENSOR_UNDEPLOY_FUNCTION;
    memcpy(&sensorRequestMsg->robotPosition, currPosition, sizeof(DGPS));
    memcpy(&sensorRequestMsg->robotPose, currPose, sizeof(pose));
    memcpy(&sensorRequestMsg->data, workingData, sizeof(dbRecord));
    sensorRequestMsg->sensorNum = driverNum;

    sensorReplyMsg = arm->unDeployment(sensorRequestMsg);
    if(sensorReplyMsg->status != SENSOR_OK) {
      // Failure condition.
      // TBD: real diagnosis?
      cerr << "[" << this->driverID << "] ERROR: \"" << ARM_SENSOR_ID << "\" returned " << sensorReplyMsg->status << endl;
      retStatus = SENS_MAN_FAILED;
    }

    // If it got here, all is well.

    return(retStatus);
  }
}

sensorCost *smManipulatorDriver::getCost(sensManRequest *sensManRequestMsg, int driverNum, dbRecord *workingData) {
  // TBD: Actually calculate these numbers!
  sensorCost *retVal = new sensorCost();
  retVal->timeCost = 10.0; 
  retVal->energyCost = 10.0; 
  return(retVal);
}

sensorWorkspace *smManipulatorDriver::getWorkspace() {
  // TBD: Actually find numbers that aren't arbitrary!
  sensorWorkspace *retVal = new sensorWorkspace();
  retVal->minDistanceTolerance = 3.0; 
  retVal->maxDistanceTolerance = 4.0; 
  retVal->angularTolerance = 0.785398; /* half angle of PI/4 */
  return(retVal);
}

SENS_MAN_STATUS smManipulatorDriver::getStatus() {
  // TBD: ACTUALLY IMPLEMENT THIS!
  return(SENS_MAN_SUCCESSFUL);
}

