/* FILE:    smHiResDriver.cpp
   AUTHOR:  Michael Wagner
   CREATED: Apr 6, 1999

   DESCRIPTION: This file implements the high-resolution
     camera sensor manager device driver class, 
     smHiResDriver. This class is inhereted from the
     virtual class smDriver.h
*/

#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "ndds/NDDS.h"
#include "smHiResDriver.h"
#include "smDriverDef.h"
#include "ppm.h"
#include "database.h"
#include "dbDef.h"
#include "sensorDef.h"
#include "sensorMsgs.h"
#include "hiResSensorDef.h"

// Public functions

smHiResDriver::smHiResDriver(DGPS *currPosition, pose *currPose) : smDriver(currPosition, currPose) {
  strcpy(this->driverID, HI_RES_DRIVER_ID);
  this->currPosition = currPosition;
  this->currPose = currPose;
  sensorClientSetup();
}

smHiResDriver::~smHiResDriver() {
}

SENS_MAN_STATUS smHiResDriver::execute(sensManRequest *sensManRequestMsg, int driverNum, dbRecord *workingData, dbRecordBitMask *bitMask) {
  SENS_MAN_STATUS retStatus = SENS_MAN_SUCCESSFUL;

  if(sensManRequestMsg == NULL) {
    return(SENS_MAN_FAILED);
  } else {

    // *** Check for sensor server(s) ***
    if(NddsClientServerWait(client, 2.0, 5, 1) == RTI_FALSE) { 
      cerr << "[" << this->driverID << "] ERROR: \"" << HI_RES_SENSOR_ID << "\" server missing!" << endl;
      return(SENS_MAN_FAILED);
    } 

    // *** DEPLOY ***

    strcpy(sensorRequestMsg->sensorID, HI_RES_SENSOR_ID);
    sensorRequestMsg->function = SENSOR_DEPLOY_FUNCTION;
    memcpy(&sensorRequestMsg->robotPosition, currPosition, sizeof(DGPS));
    memcpy(&sensorRequestMsg->robotPose, currPose, sizeof(pose));
    memcpy(&sensorRequestMsg->data, workingData, sizeof(dbRecord));
    sensorRequestMsg->sensorNum = driverNum;

    clientStatus = NddsClientCallAndWait(client, sensorReplyMsg, sensorRequestMsg,
					 SENSOR_MIN_WAIT_TIME, SENSOR_MAX_WAIT_TIME);
    if(clientStatus == NDDS_CLIENT_RECV_REPLY) {
      if(sensorReplyMsg->status != SENSOR_OK) {
	// Failure condition.
	// TBD: real diagnosis?
	cerr << "[" << this->driverID << "] ERROR: \"" << HI_RES_SENSOR_ID << "\" returned " << sensorReplyMsg->status << ", extended status = " << sensorReplyMsg->extendedStatus << endl;
	retStatus = SENS_MAN_FAILED;
      }
    } else {
      // Sensor did not respond to NDDS message
      cerr << "[" << this->driverID << "] ERROR: \"" << HI_RES_SENSOR_ID << "\" did not respond to request!" << endl;
      retStatus = SENS_MAN_FAILED;
    }

    // If deploy fails, don't do anything else.
    if(retStatus != SENS_MAN_SUCCESSFUL) {
      return(retStatus);
    }

    // *** DAQ *** 

    strcpy(sensorRequestMsg->sensorID, HI_RES_SENSOR_ID);
    sensorRequestMsg->function = SENSOR_ACQUIRE_DATA_FUNCTION;
    memcpy(&sensorRequestMsg->robotPosition, currPosition, sizeof(DGPS));
    memcpy(&sensorRequestMsg->robotPose, currPose, sizeof(pose));
    memcpy(&sensorRequestMsg->data, workingData, sizeof(dbRecord));
    sensorRequestMsg->sensorNum = driverNum;

    clientStatus = NddsClientCallAndWait(client, sensorReplyMsg, sensorRequestMsg,
					 SENSOR_MIN_WAIT_TIME, SENSOR_MAX_WAIT_TIME);
    if(clientStatus == NDDS_CLIENT_RECV_REPLY) {
      if(sensorReplyMsg->status != SENSOR_OK) {
	// Failure condition.
	// TBD: real diagnosis?
	cerr << "[" << this->driverID << "] ERROR: \"" << HI_RES_SENSOR_ID << "\" returned " << sensorReplyMsg->status << endl;
	if(sensorReplyMsg->extendedStatus == HI_RES_SENSOR_LOST_TARGET) {
	  retStatus = SENS_MAN_TRY_AGAIN;
	} else {
	  retStatus = SENS_MAN_FAILED;
	}
      }
    } else {
      // Sensor did not respond to NDDS message
      cerr << "[" << this->driverID << "] ERROR: \"" << HI_RES_SENSOR_ID << "\" did not respond to request!" << endl;
      retStatus = SENS_MAN_FAILED;
    }

    memcpy(workingData, &sensorReplyMsg->data, sizeof(dbRecord));
    memcpy(bitMask, &sensorReplyMsg->changedFields, sizeof(dbRecordBitMask));

    // *** UNDEPLOY ***

    strcpy(sensorRequestMsg->sensorID, HI_RES_SENSOR_ID);
    sensorRequestMsg->function = SENSOR_UNDEPLOY_FUNCTION;
    memcpy(&sensorRequestMsg->robotPosition, currPosition, sizeof(DGPS));
    memcpy(&sensorRequestMsg->robotPose, currPose, sizeof(pose));
    memcpy(&sensorRequestMsg->data, workingData, sizeof(dbRecord));
    sensorRequestMsg->sensorNum = driverNum;

    clientStatus = NddsClientCallAndWait(client, sensorReplyMsg, sensorRequestMsg,
					 SENSOR_MIN_WAIT_TIME, SENSOR_MAX_WAIT_TIME);
    if(clientStatus == NDDS_CLIENT_RECV_REPLY) {
      if(sensorReplyMsg->status != SENSOR_OK) {
	// Failure condition.
	// TBD: real diagnosis?
	cerr << "[" << this->driverID << "] ERROR: \"" << HI_RES_SENSOR_ID << "\" returned " << sensorReplyMsg->status << endl;
	retStatus = SENS_MAN_FAILED;
      }
    } else {
      // Sensor did not respond to NDDS message
      cerr << "[" << this->driverID << "] ERROR: \"" << HI_RES_SENSOR_ID << "\" did not respond to request!" << endl;
      retStatus = SENS_MAN_FAILED;
    }

    // If it got here, all is well.

    return(retStatus);
  }
}

sensorCost *smHiResDriver::getCost(sensManRequest *sensManRequestMsg, int driverNum, dbRecord *workingData) {
  // TBD: Actually calculate these numbers!
  sensorCost *retVal = new sensorCost();
  retVal->timeCost = 1.0; 
  retVal->energyCost = 1.0; 
  return(retVal);
}

sensorWorkspace *smHiResDriver::getWorkspace() {
  // TBD: Actually find numbers that aren't arbitrary!
  sensorWorkspace *retVal = new sensorWorkspace();
  retVal->minDistanceTolerance = 4.0; 
  retVal->maxDistanceTolerance = 7.0; 
  retVal->angularTolerance = 0.785398; /* half angle of PI/4 */
  return(retVal);
}

SENS_MAN_STATUS smHiResDriver::getStatus() {
  // TBD: ACTUALLY IMPLEMENT THIS!
  return(SENS_MAN_SUCCESSFUL);
}

void smHiResDriver::sensorClientSetup() {
  sensorRequestNddsRegister();
  sensorReplyNddsRegister();
  sensorRequestMsg = (sensorRequest *)calloc(1, sizeof(sensorRequest));
  sensorRequestMsg->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  sensorReplyMsg = (sensorReply *)calloc(1, sizeof(sensorReply));
  sensorReplyMsg->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));
  client = NddsClientCreate(HI_RES_SENSOR_ID, sensorReplyPublicationType, sensorRequestPublicationType);
  cerr << "[" << this->driverID << "] Created sensor \"" << HI_RES_SENSOR_ID << "\" client!" << endl;
}  


