/* FILE: sensManMain.cpp
   AUTHOR: Michael Wagner
   CREATED: Apr 19, 1999
   DESCRIPTION: This is code that uses a sensMan object. It serves as a link
     between this object and NDDS.
*/

#include <iostream.h>
#include "ndds/NDDS.h"
#include "sensMan.h"
#include "sensManDef.h"
#include "smDriverDef.h"
#include "sensManMsgs.h"
#include "dbRecord.h"
#include "dbMsgs.h"
#include "dbDef.h"
#include "database.h"
#include "SAS_ServiceNames.h"
#include "sensManMain.h"
#include "posDerivedState.h"
#include "nddsMsgNames.h"
#include "SAS_Config.h"

#ifndef PI
#define PI 3.14159265359
#endif

// Includes for sensor manager drivers. 
// *** Add new includes when you want to add a new sensor manager driver.
#include "smDriver.h"
#include "smHiResDriver.h"
#include "smManipulatorDriver.h"

// Service routines
// These return RTI_TRUE when a reply should be sent,
// RTI_FALSE if no reply should be sent.
RTIBool sensManServiceRtn(NDDSObjectInstance reply,
			  NDDSObjectInstance request,
			  void *userData) 
{
  sensManRequest *serverRequest   = (sensManRequest *)request;
  sensManReply   *serverReply     = (sensManReply *)reply;
  sensManServiceParam *param      = (sensManServiceParam *)userData;
  sensMan *sensorManager          = param->sm;
  sensManQueue *reqQ              = param->q;
  smDriver *driver = sensorManager->getDriver(serverRequest->driverNum);

  // Respond to mission planner commands
  switch(serverRequest->mode) {
  case SENS_MAN_COST_REQUEST:
    // Only calculate costs.
    if(driver == NULL) {
      // This driver number is not valid!
      serverReply->status = SENS_MAN_NO_SUCH_DRIVER;
    } else {
      // The driver is valid...put the request in the queue to be serviced FIFO.
      sensManRequest *tmpReq = (sensManRequest *)calloc(1, sizeof(sensManRequest));
      memcpy(tmpReq, serverRequest, sizeof(sensManRequest));
      serverReply->reqNum = reqQ->add(tmpReq);
      serverReply->status = SENS_MAN_SUBMITTED_REQ;
    }
    return(RTI_TRUE);
    break;
  case SENS_MAN_WORKSPACE_REQUEST:
    // Return workspace. 
    if(driver == NULL) {
      // This driver number is not valid!
      serverReply->status = SENS_MAN_NO_SUCH_DRIVER;
    } else {
      // The driver is valid...put the request in the queue to be serviced FIFO.
      sensManRequest *tmpReq = (sensManRequest *)calloc(1, sizeof(sensManRequest));
      memcpy(tmpReq, serverRequest, sizeof(sensManRequest));
      serverReply->reqNum = reqQ->add(tmpReq);
      serverReply->status = SENS_MAN_SUBMITTED_REQ;
    }
    return(RTI_TRUE);
    break;
  case SENS_MAN_ACTION_REQUEST:
    // Check to make sure the driver number is ok.
    if(driver == NULL) {
      // This driver number is not valid!
      serverReply->status = SENS_MAN_NO_SUCH_DRIVER;
    } else {
      // The driver is valid...put the request in the queue to be serviced FIFO.
      sensManRequest *tmpReq = (sensManRequest *)calloc(1, sizeof(sensManRequest));
      memcpy(tmpReq, serverRequest, sizeof(sensManRequest));
      serverReply->reqNum = reqQ->add(tmpReq);
      serverReply->status = SENS_MAN_SUBMITTED_REQ;
    }
    return(RTI_TRUE);
    break;
  case SENS_MAN_STATUS_REQUEST:
    // Check what the current status of the sensor manager is.
    // Ignore serverReply->reqNum.
    serverReply->status = SENS_MAN_SUCCESSFUL; // TBD: Implement this!
    return(RTI_TRUE);
    break;
  case SENS_MAN_NUM_DRIVER_REQUEST:
    // Return how many sensor manager drivers exist in serverReply->status.
    // Ignore serverReply->reqNum.
    serverReply->status = sensorManager->getNumDrivers();
    return(RTI_TRUE);
    break;
  default:
    return(RTI_FALSE);
    break;
  }
}

// NDDS server and client setup messages.
void dbClientSetup(NDDSClient *client) {
  *client = NddsClientCreate(DB_SERVICE,
			     dbReplyPublicationType,
			     dbRequestPublicationType);
}

void sensManServerSetup(sensMan *sensorManager, sensManQueue *reqQ) {
  sensManRequest *sensManRequestMsg = (sensManRequest *)calloc(1, sizeof(sensManRequest));
  sensManReply   *sensManReplyMsg = (sensManReply *)calloc(1, sizeof(sensManReply));
  sensManServiceParam *param = (sensManServiceParam *)calloc(1, sizeof(sensManServiceParam));

  param->sm = sensorManager;
  param->q  = reqQ;

  NddsServerCreate(SENS_MAN_SERVICE, NDDS_SERVER_IMMEDIATE,
		   SENS_MAN_SERVER_STRENGTH,
		   sensManReplyMsg, sensManRequestMsg,
		   sensManReplyPublicationType,
		   sensManRequestPublicationType,
		   sensManServiceRtn, param);
}


// NDDS publish/subscribe messages.
RTIBool sensorRegCallback(NDDSRecvInfo *issue) {
  sensorReg *item = (sensorReg *)issue->instance;
  sensMan *sm = (sensMan *)issue->callBackRtnParam;

  if (issue->status == NDDS_FRESH_DATA) {
    int sensorNum;
    if(sm->addSensor(item->sensorID, (SENSOR_STATUS)item->status, &sensorNum) == SENS_MAN_SUCCESSFUL) {
      cerr << "[sensManMain] Registered sensor \"" << item->sensorID << "\"" << endl;
    }
  }

  return(RTI_TRUE);
}
  
RTIBool telemetryCallback(NDDSRecvInfo *issue) {
  posDerivedState *item = (posDerivedState *)issue->instance;
  telemetryParam *param = (telemetryParam *)issue->callBackRtnParam;

  if (issue->status == NDDS_FRESH_DATA) {
    param->currPosition->x = item->posWest / -100.0;
    param->currPosition->y = item->posNorth / 100.0;
    param->currPosition->z = item->posUp / 100.0;
    param->currPose->pitch = item->rotX / 100.0;
    param->currPose->roll  = item->rotY / 100.0;
    param->currPose->yaw   = 2 * PI - item->rotZ / 100.0;

  }

  return(RTI_TRUE);
}

void publishComplete(NDDSPublication pub, sensManComplete *msg, SENS_MAN_STATUS status, int reqNum) {
  msg->status = status;
  msg->reqNum = reqNum;

  NddsPublicationSend(pub);
}

void publishComplete(NDDSPublication pub, sensManComplete *msg, sensorCost *c, int reqNum) {
  msg->status = SENS_MAN_SUCCESSFUL;
  msg->timeCost = c->timeCost;
  msg->energyCost = c->energyCost;
  msg->reqNum = reqNum;

  NddsPublicationSend(pub);
}

void publishComplete(NDDSPublication pub, sensManComplete *msg, sensorWorkspace *w, int reqNum) {
  msg->status = SENS_MAN_SUCCESSFUL;
  msg->maxDistanceTolerance = w->maxDistanceTolerance;
  msg->minDistanceTolerance = w->minDistanceTolerance;
  msg->angularTolerance = w->angularTolerance;
  msg->reqNum = reqNum;
  NddsPublicationSend(pub);
}

int main(int argc, char *argv[]) {

  int nddsDomain;
  if(!getSAS_Config(NDDS_DOMAIN, nddsDomain)) {
    cerr << "[sensManMain] ERROR: Cannot read " << NDDS_DOMAIN << " from config file!" << endl;
    return(-1);
  }

  NddsInit(nddsDomain, NULL);
  cerr << "[sensManMain] Initialized NDDS" << endl;

  // Sensor registration stuff
  NDDSSubscription sensorRegSub;
  sensorReg *sensorRegMsg = (sensorReg *)calloc(1, sizeof(sensorReg));
  sensorRegMsg->sensorID = (char *)calloc(SENSOR_MAX_ID_LENGTH, sizeof(char));

  // Telemetry stuff
  NDDSSubscription posDerivedStateSub;
  posDerivedState *posDerivedStateMsg = NULL;
  DGPS *currPosition = new DGPS();
  pose *currPose = new pose();
  telemetryParam *tParam = new telemetryParam();
  tParam->currPosition = currPosition;
  tParam->currPose = currPose;

  // Database stuff
  NDDSClient            dbClient;
  NDDSClientReplyStatus dbClientStatus;
  dbRequest             *dbRequestMsg = (dbRequest *)calloc(1, sizeof(dbRequest));
  dbReply               *dbReplyMsg   = (dbReply *)calloc(1, sizeof(dbReply));

  // NDDS client/server messages to/from the Sensor Manager, and publications that will be
  // made when a Sensor Manager request is completed, along with their publications.
  sensManRequest  *sensManRequestMsg  = (sensManRequest *)calloc(1, sizeof(sensManRequest));
  sensManComplete *sensManCompleteMsg = (sensManComplete *)calloc(1, sizeof(sensManComplete));
  NDDSPublication    sensManPublication;

  // Other objects
  database *db = new database();
  sensMan *sensorManager = new sensMan();
  sensManQueue *reqQ = new sensManQueue();

  // Register all NDDS types
  sensorRegNddsRegister();
  sensManReplyNddsRegister();
  sensManRequestNddsRegister();
  sensManCompleteNddsRegister();
  dbReplyNddsRegister();
  dbRequestNddsRegister();
  posDerivedStateNddsRegister();

  // Next set up the sensor manager subscriber for registration of sensors.
  // When a sensor driver registers itself, it will set up the appropriate
  // sensorClient in the array.
  cerr << "[sensManMain] Setting up sensor registration subscriber...";
  sensorRegSub = NddsSubscriptionCreate(NDDS_SUBSCRIPTION_IMMEDIATE,
					SENSOR_REG_MSG_NAME, sensorRegPublicationType,
					sensorRegMsg, SENSOR_REG_DEADLINE,
					SENSOR_REG_MIN_SEPARATION, sensorRegCallback, sensorManager);
  cerr << "done! Awaiting sensor registration messages." << endl;

  // Now we start reading in telemetry
  cerr << "[targetAcqMain] Setting up telemetry subscriber...";
  posDerivedStateMsg = (posDerivedState *)calloc(1, sizeof(posDerivedState));
  posDerivedStateSub = NddsSubscriptionCreate(NDDS_SUBSCRIPTION_IMMEDIATE,
					      POS_DERIVED_STATE_MSG_NAME, posDerivedStatePublicationType,
					      posDerivedStateMsg, TELEMETRY_DEADLINE,
					      TELEMETRY_MIN_SEPARATION, telemetryCallback, tParam);
  cerr << "done! Awaiting sensor registration messages." << endl;

  // Before the mission planner can command the sensor manager, a database client
  // must be set up.
  cerr << "[sensManMain] Setting up the database client...";
  dbClientSetup(&dbClient);
  if(NddsClientServerWait(dbClient, 2.0, 5, 1) == RTI_FALSE) { 
    cerr << "[sensManMain] ERROR: \"" << DB_SERVICE << "\" server missing!" << endl;
    return(-1);
  } 
  cerr << "done!" << endl;

  // We have to initialize all sensor manager drivers here.
  // ****** This is the part of the code you'll need to modify to add new drivers ******

  // *** numDesiredDrivers should be equal to the number of drivers you want in the system.
  const int numDesiredDrivers = 2;
  int numDrivers;

  // *** Use initDriver to add new drivers here.
  sensorManager->initDriver((smDriver *)(new smHiResDriver(currPosition, currPose)), &numDrivers);
  sensorManager->initDriver((smDriver *)(new smManipulatorDriver(currPosition, currPose)), &numDrivers);

  if(sensorManager->getNumDrivers() != numDesiredDrivers) {
    cerr << "[sensManMain] ERROR: Failed to register sensor manager drivers!" << endl;
    return(-1);
  }

  // *** Use a dbRequest to register drivers with the database
  dbRequestMsg->function = DB_SM_DRIVER_REG_FUNCTION;
  strcpy(dbRequestMsg->name, HI_RES_DRIVER_ID);
  dbClientStatus = NddsClientCallAndWait(dbClient,
					 dbReplyMsg, dbRequestMsg,
					 DB_MIN_WAIT_TIME,
					 DB_MAX_WAIT_TIME);
  if(dbClientStatus == NDDS_CLIENT_RECV_REPLY) {
    if(dbReplyMsg->status == DB_SM_DRIVER_REG_DUPLICATE_ERROR) {
      cerr << "[sensManMain] Sensor manager driver \"" << dbRequestMsg->name << "\" already registered with database" << endl;
    } else if(dbReplyMsg->status != DB_OK) {
      // Database returned fail condition. Set status and return.
      // TBD: Do a real diagnosis?
      cerr << "[sensManMain] ERROR: Registration of sensor manager driver \"" << dbRequestMsg->name << "\" returned " << dbReplyMsg->status << endl;
    }
  } else {
    // Database did not respond. 
    cerr << "[sensManMain] ERROR: Database did not respond to smDriver register request!" << endl;
  }

  dbRequestMsg->function = DB_SM_DRIVER_REG_FUNCTION;
  strcpy(dbRequestMsg->name, MANIPULATOR_DRIVER_ID);
  dbClientStatus = NddsClientCallAndWait(dbClient,
					 dbReplyMsg, dbRequestMsg,
					 DB_MIN_WAIT_TIME,
					 DB_MAX_WAIT_TIME);
  if(dbClientStatus == NDDS_CLIENT_RECV_REPLY) {
    if(dbReplyMsg->status == DB_SM_DRIVER_REG_DUPLICATE_ERROR) {
      cerr << "[sensManMain] Sensor manager driver \"" << dbRequestMsg->name << "\" already registered with database" << endl;
    } else if(dbReplyMsg->status != DB_OK) {
      // Database returned fail condition. Set status and return.
      // TBD: Do a real diagnosis?
      cerr << "[sensManMain] ERROR: Registration of sensor manager driver \"" << dbRequestMsg->name << "\" returned " << dbReplyMsg->status << endl;
    }
  } else {
    // Database did not respond. 
    cerr << "[sensManMain] ERROR: Database did not respond to smDriver register request!" << endl;
  }

  // Now we can set up the sensor manager server that accepts command request from the
  // mission planner.
  cerr << "[sensManMain] Setting up Sensor Manager server...";
  sensManServerSetup(sensorManager, reqQ);
  cerr << "done!" << endl;

  // Now we set up the publishers, which will create messages when a request in
  // the queue has been serviced.
  cerr << "[sensManMain] Setting up publishers...";
  sensManPublication = NddsPublicationCreate(SENS_MAN_COMPLETE_MSG_NAME, sensManCompletePublicationType,
						sensManCompleteMsg, SENS_MAN_COMPLETE_PERSISTANCE,
						SENS_MAN_COMPLETE_STRENGTH);
  cerr << "done!" << endl;  

  // Set stuff up for the queue checks
  requestStruct *req;
  smDriver *currentDriver;
  SENS_MAN_STATUS sensorRetStatus;
  sensorCost *sensorRetCost;
  sensorWorkspace *sensorRetWorkspace;

  while(1) {
    // In here we check for mission planner requests in the reqQ.
    if(!reqQ->isEmpty()) {
      req = reqQ->remove();
      if(req != NULL) {
	// There is a valid request in the queue.

	sensManRequestMsg = (sensManRequest *)req->req;

	if(sensorManager->isValidDriver(sensManRequestMsg->driverNum)) {
	  currentDriver = sensorManager->getDriver(sensManRequestMsg->driverNum);

	  switch(sensManRequestMsg->mode) {
	  case SENS_MAN_ACTION_REQUEST:
	    {
	      // First get working data from the database.
	      dbRequestMsg->function = DB_GET_FUNCTION;
	      dbRequestMsg->driverNum = sensManRequestMsg->driverNum;
	      dbRequestMsg->record.targetID = sensManRequestMsg->targetID;
	      
	      dbClientStatus = NddsClientCallAndWait(dbClient,
						     dbReplyMsg, dbRequestMsg,
						     DB_MIN_WAIT_TIME,
						     DB_MAX_WAIT_TIME);
	      if(dbClientStatus == NDDS_CLIENT_RECV_REPLY) {
		if(dbReplyMsg->status != DB_OK) {
		  // Database returned fail condition. Set status and return.
		  // TBD: Do a real diagnosis?
		  cerr << "[sensManMain] ERROR: Database get record returned " << dbReplyMsg->status << endl;
		  publishComplete(sensManPublication, sensManCompleteMsg, SENS_MAN_FAILED, req->reqNum);
		  continue;
		}
	      } else {
		// Database did not respond. 
		cerr << "[sensManMain] ERROR: Database did not respond to get record request" << endl;
		publishComplete(sensManPublication, sensManCompleteMsg, SENS_MAN_FAILED, req->reqNum);
		continue;
	      }
	      
	      db->createUnalteredBitMask(dbRequestMsg->recordBitMask);
	      
	      cerr << "******" << endl;
	      cerr << "[sensManMain] Taking measurement of target " << sensManRequestMsg->targetID 
		   << " with driver " << sensManRequestMsg->driverNum << "..." << endl;
	      sensorRetStatus = currentDriver->execute(sensManRequestMsg, sensManRequestMsg->driverNum, &dbReplyMsg->record, &dbRequestMsg->recordBitMask);
	      cerr << "[sensManMain] Done! Driver returned " << sensorRetStatus << endl;
	      
	      if(sensorRetStatus == SENS_MAN_SUCCESSFUL) {
		cerr << "[sensManMain] Updating database...";
		
		dbRequestMsg->function = DB_PUT_FUNCTION;
		dbRequestMsg->driverNum = sensManRequestMsg->driverNum;
		memcpy(&dbRequestMsg->record, &dbReplyMsg->record, sizeof(dbRecord));
		
		// dbRequestMsg->recordBitMask already edited by currentDriver->execute();
		
		dbClientStatus = NddsClientCallAndWait(dbClient,
						       dbReplyMsg, dbRequestMsg,
						       DB_MIN_WAIT_TIME,
						       DB_MAX_WAIT_TIME);
		if(dbClientStatus == NDDS_CLIENT_RECV_REPLY) {
		  if(dbReplyMsg->status != DB_OK) {
		    // Database returned fail condition. Set status and return.
		    // TBD: Do a real diagnosis?
		    cerr << "[sensManMain] ERROR: Database put record returned " << dbReplyMsg->status << endl;
		    publishComplete(sensManPublication, sensManCompleteMsg, SENS_MAN_FAILED, req->reqNum);
		    continue;
		  }
		} else {
		  // Database did not respond. 
		  cerr << "[sensManMain] ERROR: Database did not respond to put record request" << endl;
		  publishComplete(sensManPublication, sensManCompleteMsg, SENS_MAN_FAILED, req->reqNum);
		  continue;
		}
		
		// Success!
		cerr << "done!" << endl;
	      }

	      publishComplete(sensManPublication, sensManCompleteMsg, sensorRetStatus, req->reqNum);
	      break;
	    }
	  case SENS_MAN_COST_REQUEST:
	    {
	      // Request cost from the sensor manager driver

	      // First get working data from the database.
	      dbRequestMsg->function = DB_GET_FUNCTION;
	      dbRequestMsg->driverNum = sensManRequestMsg->driverNum;
	      dbRequestMsg->record.targetID = sensManRequestMsg->targetID;
	      
	      dbClientStatus = NddsClientCallAndWait(dbClient,
						     dbReplyMsg, dbRequestMsg,
						     DB_MIN_WAIT_TIME,
						     DB_MAX_WAIT_TIME);
	      if(dbClientStatus == NDDS_CLIENT_RECV_REPLY) {
		if(dbReplyMsg->status != DB_OK) {
		  // Database returned fail condition. Set status and return.
		  // TBD: Do a real diagnosis?
		  cerr << "[sensManMain] ERROR: Database get record returned " << dbReplyMsg->status << endl;
		  publishComplete(sensManPublication, sensManCompleteMsg, SENS_MAN_FAILED, req->reqNum);
		  continue;
		}
	      } else {
		// Database did not respond. 
		cerr << "[sensManMain] ERROR: Database did not respond to get record request" << endl;
		publishComplete(sensManPublication, sensManCompleteMsg, SENS_MAN_FAILED, req->reqNum);
		continue;
	      }
	      
	      cerr << "******" << endl;
	      cerr << "[sensManMain] Requesting costs from driver #" << sensManRequestMsg->driverNum << "...";
	      sensorRetCost = currentDriver->getCost(sensManRequestMsg, sensManRequestMsg->driverNum, &dbReplyMsg->record);
	      cerr << "done!" << endl;
	      
	      // Success!
	      publishComplete(sensManPublication, sensManCompleteMsg, sensorRetCost, req->reqNum);
	      break;
	    }
	  case SENS_MAN_WORKSPACE_REQUEST:
	    {
	      // Request workspace from the sensor manager driver
	      cerr << "******" << endl;
	      cerr << "[sensManMain] Requesting workspace from driver #" << sensManRequestMsg->driverNum << "...";
	      sensorRetWorkspace = currentDriver->getWorkspace();
	      cerr << "done!" << endl;
	      
	      // Success!
	      publishComplete(sensManPublication, sensManCompleteMsg, sensorRetWorkspace, req->reqNum);
	      break;
	    }
	  }
	}

	delete(req);
      }
    } 

    NddsUtilitySleep(0.1);
  }

  cerr << "[sensManMain] ERROR: main() should not return!" << endl;
  return(-1);

}
