#include <stdio.h>
#include <math.h>
#include "navplan.h"
#include "straight.h"

/**************************************************************************/
/* This helper function takes the current farming state structure and     */
/* finds the two endpoints of one of the coverage area's edges.  The      */
/* desired edge number is stored in the topedge field of the farming      */
/* state structure.  The two endpoints are "one" and "two".               */
/* "one" is before "two" in a counter-clockwise direction.                */
/**************************************************************************/
void get_topedge(straight_state_type* state, MP_POSE *one, MP_POSE *two) {
  polygon *temp;
  int num;

  temp = state->area;
  num = 1;
  while (num < state->topedge) {
    temp = temp->next;
    num++;
  }
  one->x = temp->vertex.x;
  one->y = temp->vertex.y;
  one->z = temp->vertex.z;
  one->yaw = temp->vertex.yaw;
  if (temp->next == NULL) {
    two->x = state->area->vertex.x;
    two->y = state->area->vertex.y;
    two->z = state->area->vertex.z;
    two->yaw = state->area->vertex.yaw;
  }
  else {
    two->x = temp->next->vertex.x;
    two->y = temp->next->vertex.y;
    two->z = temp->next->vertex.z;
    two->yaw = temp->next->vertex.yaw;
  }
}


/**************************************************************************/
/* This helper function takes the current farming state structure and     */
/* finds the two endpoints of one of the coverage area's edges.  The      */
/* desired edge number is stored in the bottomedge field of the farming   */
/* state structure.  The two endpoints are "one" and "two".               */
/* "two" is before "one" in a counter-clockwise direction.                */
/**************************************************************************/
void get_bottomedge(straight_state_type *state, MP_POSE *one, MP_POSE *two) {
  polygon *temp;
  int num;

  temp = state->area;
  num = 1;
  while (num < state->bottomedge) {
    temp = temp->next;
    num++;
  }
  two->x = temp->vertex.x;
  two->y = temp->vertex.y;
  two->z = temp->vertex.z;
  two->yaw = temp->vertex.yaw;
  if (temp->next == NULL) {
    one->x = state->area->vertex.x;
    one->y = state->area->vertex.y;
    one->z = state->area->vertex.z;
    one->yaw = state->area->vertex.yaw;
  }
  else {
    one->x = temp->next->vertex.x;
    one->y = temp->next->vertex.y;
    one->z = temp->next->vertex.z;
    one->yaw = temp->next->vertex.yaw;
  }
}


/**************************************************************************/
/* This function initializes the straight rows pattern.  It calculates    */
/* the straight value and the first 2 path segments, including the        */
/* direction of the pattern turns and top and bottom area edges.          */
/* (These top and bottom edges are used in finding the endpoints of path  */
/*  segments.)                                                            */
/**************************************************************************/
void init_straight(straight_state_type* state, plan_command* command) {
  int num = 1;
  polygon* temp;
  MP_POSE one, two, point;

  state->curr = state->area->vertex;
  state->currseg.start = state->curr;
  state->currseg.end = state->area->next->vertex;

  /* angle is atan2(-deltax, deltay), to convert to nav yaw */
  state->straight = atan2(state->curr.x - state->currseg.end.x,
			  state->currseg.end.y - state->curr.y);
  if (state->straight < 0) state->straight = 2.0 * PI + state->straight;
  state->curr.yaw = state->straight;
  state->currseg.start.yaw = state->straight;
  state->currseg.end.yaw = state->straight;

  state->currseg.curvature = 1000.0;
  state->currseg.center.x = 0.0; state->currseg.center.y = 0.0;
  if (state->hand == 1)
    state->currseg.dir = 2;  /* (counterclockwise) */
  else
    state->currseg.dir = 1;  /* (clockwise) */
  state->currseg.lookahead = STR_LOOKAHEAD;

  state->topedge = 2;
  temp = state->area;
  while (temp->next != NULL) {
    temp = temp->next;
    num = num + 1; }
  state->bottomedge = num;

  state->currseg.type = PathSeg_StraightRow;

  /* set up next path segment - the first row end turn */
  state->currseg.next = (path_seg*)malloc(sizeof(path_seg));
  state->currseg.next->dir = state->currseg.dir;
  state->currseg.next->lookahead = CIR_LOOKAHEAD;
  state->currseg.next->next = NULL;
  state->currseg.next->type = PathSeg_StraightTurn;
  state->currseg.next->start = state->currseg.end;

  /* find topedge */
  get_topedge(state, &one, &two);
  /* check to see if we need to move to next edge next time */
  if (GPS_diff(two, state->currseg.end) < state->rowWidth)
    state->topedge = state->topedge + 1;

  /* find point rowWidth away from start */
  /* Note: should be rowWidth away normal to "straight" */
  if (state->currseg.next->dir == 1) point.x = state->rowWidth;
  else point.x = -1.0 * state->rowWidth;
  point.y = 0.0;
  point.yaw = PI;
  state->currseg.next->end = RelToGlo(point, state->currseg.next->start);

  /* find point half a rowWidth away from start (circle's center) */
  if (state->rowWidth/2.0 > state->minTurnRad)
    state->currseg.next->curvature = state->rowWidth/2.0;
  else
    state->currseg.next->curvature = state->minTurnRad;

  if (state->currseg.dir == 1) point.x = state->rowWidth/2.0;
  else point.x = -1.0 * state->rowWidth/2.0;
  /* This will make center on the start-end line, if the row width
     isn't too narrow.  Otherwise, it will make the center above the
     start-end line, making an endrow which "bubbles" out */
  point.y = sqrt(sqr(state->currseg.next->curvature) -
		 sqr(state->rowWidth/2.0));
  point.yaw = 0.0;
  state->currseg.next->center = RelToGlo(point, state->currseg.next->start);
}


/**************************************************************************/
/* This function selects the new path segment.  The path segment found is */
/* the next one after the current path segment about to be traversed.     */
/* There are two types of path segments -- the straight row and the       */
/* row-end half circle, and there are two directions -- up and down,      */
/* where the first row was considered "up."  When finding a row-end       */
/* segment, the lookahead is set to CIR_LOOKAHEAD.  When finding a row    */
/* segment, the lookahead is set to STR_LOOKAHEAD.  When finding a row    */
/* segment, we check to see if the endpoint is past the current area edge */
/* and if so, we move to the next edge of the area.  When the topedge is  */
/* the same as the bottomedge, the pattern is done.  The direction is     */
/* switched when a new row segment is defined.                            */
/**************************************************************************/
int straight_next_seg(straight_state_type *state) {
  int complete = 0;
  MP_POSE one, two, new, point;
  float slope, m;

  /* copy currseg.next into currseg */
  copy_segment(&(state->currseg), *(state->currseg.next));

  /* find new segment, and put into currseg.next */
  if (state->currseg.type == PathSeg_StraightRow) {
    /* we're about to travel row - select next circular path segment to turn
       end of row */
    if ((state->hand == 1 && state->currseg.dir == 2) ||
	(state->hand == 2 && state->currseg.dir == 1)) {
      /* left pattern, and next segment is counterclockwise, or
	 right pattern, and next segment is clockwise => going up row */
      /* find topedge */
      get_topedge(state, &one, &two);
      /* check to see if we need to move to next edge next time */
      if (GPS_diff(two, state->currseg.end) < state->rowWidth)
	state->topedge = state->topedge + 1;
    }
    else {  /* going down row */
      /* find bottomedge */
      get_bottomedge(state, &one, &two);
      /* check to see if we need to move to next edge next time */
      if (GPS_diff(two, state->currseg.end) < state->rowWidth)
	state->bottomedge = state->bottomedge - 1;
    }
    /* find the point a rowWidth away from the last waypoint 
       (should be a rowWidth away perpendicular to "straight") */
    state->currseg.next->start = state->currseg.end;
    state->currseg.next->dir = state->currseg.dir;
    if (state->currseg.dir == 1)
      point.x = state->rowWidth;
    else
      point.x = -1.0 * state->rowWidth;
    point.y = 0.0;
    point.yaw = PI;
    state->currseg.next->end = RelToGlo(point, state->currseg.next->start);

    if (state->rowWidth/2.0 > state->minTurnRad)
      state->currseg.next->curvature = state->rowWidth/2.0;
    else
      state->currseg.next->curvature = state->minTurnRad;

    if (state->currseg.dir == 1) point.x = state->rowWidth/2.0;
    else point.x = -1.0 * state->rowWidth/2.0;
    /* This will make center on the start-end line, if the row width
       isn't too narrow.  Otherwise, it will make the center above the
       start-end line, making an endrow which "bubbles" out */
    point.y = sqrt(sqr(state->currseg.next->curvature) -
		   sqr(state->rowWidth/2.0));
    point.yaw = 0.0;

    state->currseg.next->center = RelToGlo(point, state->currseg.next->start);
    state->currseg.next->type = PathSeg_StraightTurn;
    state->currseg.next->lookahead = CIR_LOOKAHEAD;
  }


  else if (state->currseg.type == PathSeg_StraightTurn) {
    /* select straight path segment for next row */
    /* intersect opposite edge with "straight" line */
    state->currseg.next->lookahead = STR_LOOKAHEAD;

    /* First, we need to find the coverage area edge to intersect with */
    if ((state->hand == 1 && state->currseg.dir == 2) ||
	(state->hand == 2 && state->currseg.dir == 1))
      /* find bottomedge -- going up the row */
      get_bottomedge(state, &one, &two);
    else /* find topedge -- going down the row */
      get_topedge(state, &one, &two);

    /* find intersection of this edge with a line from current endpt with
       a slope equal to "straight" */
    if (fabs(state->straight) < 0.0005) {
      if (fabs(two.x - one.x) < 0.005) {
	/* we should move to next edge in this case */
	new.x = state->currseg.end.x;
	new.y = one.y;
      }
      else {
	m = (two.y - one.y) / (two.x - one.x);
	new.x = state->currseg.end.x;
	new.y = one.y + m * (new.x - one.x);
      }
    }
    else {
      slope = -cos(state->straight) / sin(state->straight);
      if (fabs(two.x - one.x) < 0.005) {
	new.x = one.x;
	new.y = state->currseg.end.y + slope * (new.x - state->currseg.end.x);
      }
      else {
	m = (two.y - one.y) / (two.x - one.x);
	new.x = ((state->currseg.end.y - one.y) -
		 (slope * state->currseg.end.x) + (m * one.x)) /
	  (m - slope);
	new.y = state->currseg.end.y + slope * (new.x - state->currseg.end.x);
      }
    }

    /* check to see if the new point is outside the edge segment */
    /* may need to fix this to take care of near equalities */
    if (((two.x > one.x) && (new.x > two.x || new.x < one.x)) ||
	((two.x < one.x) && (new.x < two.x || new.x > one.x)) ||
	((two.x == one.x) && (two.y > one.y) &&
	 (new.y > two.y || new.y < one.y)) ||
	((two.x == one.x) && (two.y < one.y) &&
	 (new.y < two.y || new.y > one.y)) ||
	(two.x == one.x && state->straight == 0.0)) {
      /* yes it is, move to the next edge */
      if ((state->hand == 1 && state->currseg.dir == 2) ||
	  (state->hand == 2 && state->currseg.dir == 1)) {
	/* going up the row */
	state->bottomedge = state->bottomedge - 1;
	if (state->bottomedge <= state->topedge)
	  complete = 1;
	get_bottomedge(state, &one, &two);
      }
      else {  /* going down the row */
	state->topedge = state->topedge + 1;
	if (state->bottomedge <= state->topedge)
	  complete = 1;
	get_topedge(state, &one, &two);
      }
      /* Now redo what we did earlier, but with the new edge we've found: */
      /* find intersection of this next edge with a line from current endpt
	 with a slope equal to "straight" */
      if (fabs(state->straight) < 0.0005) {
	if (fabs(two.x - one.x) < 0.005) {
	  /* this may be a problem, if it ever occurs - it means the area
	     to cover was entered funny, with 2 edges in a row, both in
	     line with each other and parallel to "straight" */
	  new.x = state->currseg.end.x;
	  new.y = one.y;
	}
	else {
	  m = (two.y - one.y) / (two.x - one.x);
	  new.x = state->currseg.end.x;
	  new.y = one.y + m * (new.x - one.x);
	}
      }
      else {
	slope = -cos(state->straight) / sin(state->straight);
	if (fabs(two.x - one.x) < 0.005) {
	  new.x = one.x;
	  new.y = state->currseg.end.y + slope*(new.x - state->currseg.end.x);
	}
	else {
	  m = (two.y - one.y) / (two.x - one.x);
	  new.x = ((state->currseg.end.y - one.y) -
		   (slope * state->currseg.end.x) + (m * one.x)) /
	    (m - slope);
	  new.y = state->currseg.end.y + slope*(new.x - state->currseg.end.x);
	}
      }
    }

    state->currseg.next->start = state->currseg.end;
    state->currseg.next->end.x = new.x;
    state->currseg.next->end.y = new.y;
    state->currseg.next->end.yaw = state->currseg.next->start.yaw;

    state->currseg.next->curvature = 1000.0;
    state->currseg.next->dir = 3 - state->currseg.dir;
    /* if clockwise(1) => counterclockwise(2), else
       if counterclockwise(2) => clockwise(1) */
    state->currseg.next->center.x = 0.0;  /* not needed */
    state->currseg.next->center.y = 0.0;

    state->currseg.next->type = PathSeg_StraightRow;
  }

  return(complete);
}


/**************************************************************************/
/* This function finds the pure pursuit goal point based on the robot's   */
/* current position, checks to see if a new path segment needs to be      */
/* calculated, checks to see if the pattern has been completed, and if    */
/* not, calls follow_path to calculate the desired turn to command.       */
/* Stores the turn and speed command in "command".                        */
/* Returns a 1 if the pattern is done, 0 otherwise.                       */
/**************************************************************************/
int straightrows(straight_state_type *state, plan_command *command, int verbose) {
  int complete = 0;
  int newseg = 1;
  int obst = 0;
  MP_POSE closest;

  while (newseg && !complete) {
    closest = find_closest(state->currseg, state->curr);
    if (verbose)
      printf("endpoint = %.2f, %.2f (dist from closest to endpt = %.2f)\n",
	     state->currseg.end.x, state->currseg.end.y,
	     sqrt(sqr(closest.y - state->currseg.end.y) +
		  sqr(closest.x - state->currseg.end.x)));
    newseg = find_goal(&(state->currseg), state->curr, closest);
    if (newseg) {
      complete = straight_next_seg(state);
      if (verbose)
	printf("Reached endpoint...new endpoint is %.2f, %.2f\n",
	       state->currseg.end.x, state->currseg.end.y);
    }
  }

  if (!complete) {
    if (!obst) {  /* no obstacle in front - normal mode */
      command->speed = state->speed;
      command->turn = follow_path(state->currseg, state->minTurnRad,
				  state->curr, closest, verbose);
      /* check for obstacles ahead here, when implemented */
    }
    /* else go into obstacle mode */
  }
  else
    free(state->currseg.next);

  return(complete);
}

