#include <stdio.h>
#include <math.h>
#include "navplan.h"
#include "spiral.h"

/**************************************************************************/
/* This function initializes a spiral pattern, setting the center points, */
/* starting curvature, lookahead distance, and the first two path         */
/* segments.                                                              */
/**************************************************************************/
void init_spiral(spiral_state_type* state) {
  MP_POSE delta;

  state->currseg.curvature = (state->rowWidth)/2.0;

  if (state->currseg.dir == 1)
    delta.x = state->currseg.curvature;  /* clockwise */
  else delta.x = -1.0 * state->currseg.curvature;  /* counterclockwise */
  delta.y = 0.0;
  delta.yaw = 0.0;
  state->top_center = RelToGlo(delta, state->curr);
  state->bot_center = state->curr;
  /* find end of circular path segment - a diameter away */
  delta.x = 2.0 * delta.x;
  state->currseg.end = RelToGlo(delta, state->curr);

  state->half = 1;
  state->currseg.start = state->curr;
  state->currseg.center.x = state->top_center.x;
  state->currseg.center.y = state->top_center.y;
  state->currseg.lookahead = CIR_LOOKAHEAD;

  state->currseg.next = (path_seg*)malloc(sizeof(path_seg));
  state->currseg.next->curvature = state->currseg.curvature +
    (state->rowWidth)/2.0;
  state->currseg.next->center = state->bot_center;
  state->currseg.next->dir = state->currseg.dir;
  state->currseg.next->lookahead = state->currseg.lookahead + INCR_LOOKAHEAD;
  state->currseg.next->next = NULL;

  state->currseg.next->start = state->currseg.end;
  state->currseg.next->start.yaw = state->currseg.start.yaw + PI;
  if (state->currseg.next->start.yaw > 2.0 * PI)
    state->currseg.next->start.yaw =
      state->currseg.next->start.yaw - 2.0 * PI;

  /* find end of circular path segment - a diameter away */
  if (state->currseg.dir == 1)
    delta.x = 2.0 * state->currseg.next->curvature;
  else delta.x = -2.0 * state->currseg.next->curvature;
  delta.y = 0.0;
  delta.yaw = 0.0;
  state->currseg.next->end = RelToGlo(delta, state->currseg.next->start);

  /* unless row width is narrower than turning radius... */
  if ((state->rowWidth)/2.0 < state->minTurnRad) {
    if (state->currseg.dir == 1) delta.x = state->minTurnRad;
    else delta.x = -1.0 * state->minTurnRad;
    state->currseg.center = RelToGlo(delta, state->curr);
    delta.x = 2.0 * delta.x;
    state->currseg.end = RelToGlo(delta, state->curr);
    state->currseg.curvature = state->minTurnRad;
  }

}


/**************************************************************************/
/* This function establishes the next path segment for the spiral         */
/* pattern, expanding the radius by half the row width, and increasing    */
/* the lookahead distance by INCR_LOOKAHEAD as the spiral gets larger.    */
/* Returns a 1 if the new spiral radius is larger than the desired max    */
/* radius, 0 otherwise.                                                   */
/**************************************************************************/
int spiral_next_seg(spiral_state_type* state, int verbose) {
  int complete = 0;
  MP_POSE delta;

  /* copy currseg.next into currseg */
  copy_segment(&(state->currseg), *(state->currseg.next));

  /* find new segment, and put into currseg.next */
  state->currseg.next->curvature = state->currseg.next->curvature +
    (state->rowWidth)/2.0;

  if (state->currseg.curvature > state->maxrad)
    complete = 1;
  else {
    state->half = !(state->half);
    /* if currently on top half, then next will be bottom half... */
    if (state->half) state->currseg.next->center = state->bot_center;
    else state->currseg.next->center = state->top_center;
    if (verbose)
      printf("Expanding radius to %.2f\n", state->currseg.curvature);

    state->currseg.next->start = state->currseg.next->end;
    state->currseg.next->start.yaw = state->currseg.next->start.yaw + PI;
    if (state->currseg.next->start.yaw > 2.0 * PI)
      state->currseg.next->start.yaw =
	state->currseg.next->start.yaw - 2.0 * PI;

    /* find end of circular path segment - a diameter away */
    if (state->currseg.next->dir == 1)
      delta.x = 2.0 * state->currseg.next->curvature;
    else delta.x = -2.0 * state->currseg.next->curvature;
    delta.y = 0.0;
    delta.yaw = 0.0;
    state->currseg.next->end = RelToGlo(delta, state->currseg.next->start);

    state->currseg.next->lookahead = state->currseg.next->lookahead +
      INCR_LOOKAHEAD;
    /* just a hack, to increase the lookahead distance when our circle
       gets "straighter" */
  }

  return(complete);
}


/**************************************************************************/
/* This function finds the pure pursuit goal point based on the robot's   */
/* current position, checks to see if a new path segment needs to be      */
/* calculated, checks to see if the pattern has been completed, and if    */
/* not, calls follow_path to calculate the desired turn to command.       */
/* Stores the turn and speed command in "command".                        */
/* Returns a 1 if the pattern is done, 0 otherwise.                       */
/**************************************************************************/
int spiral(spiral_state_type* state, plan_command* command, int verbose) {
  int complete = 0;
  int newseg = 1;
  MP_POSE closest;

  while (newseg && !complete) {
    closest = find_closest(state->currseg, state->curr);
    if (verbose)
      printf("endpoint = %.2f, %.2f (dist from closest to endpt = %.2f)\n",
	     state->currseg.end.x, state->currseg.end.y,
	     sqrt(sqr(closest.y - state->currseg.end.y) +
		  sqr(closest.x - state->currseg.end.x)));
    newseg = find_goal(&(state->currseg), state->curr, closest);
    if (newseg)
      /* find the next path segment */
      complete = spiral_next_seg(state, verbose);
  }

  if (!complete) {
    command->turn = follow_path(state->currseg, state->minTurnRad,
				state->curr, closest, verbose);
    command->speed = state->speed;
  }
  else
    free(state->currseg.next);

  return(complete);
}
