/*****************************************************************************
 *
 * FILE: pplanner.c
 *
 * ABSTRACT: Waypoint path finder, derived from dstarStub.c
 * 
 *****************************************************************************/
#include <stdio.h>
#include <math.h>
#include "navplan.h"
#include "pplanner.h"


/************************************************************************/
/* This function returns 1 if the current pose is within a tolerance    */
/* amount of the goal, 0 otherwise.  The tolerance desired is part of   */
/* the input Goal struct.                                               */
/************************************************************************/
int goalReached(MP_POSE pose, MP_POSE pointGoal, float tolerance)
{
  float distSq;

  distSq = SQR(pose.x - pointGoal.x) + SQR(pose.y - pointGoal.y);

  return (distSq <= SQR(tolerance));
}


/************************************************************************/
/* This is the primary function, which calculates the position of the   */
/* goal relative to the robot's current position (localX and localY).   */
/* The direction to turn is then calculated and returned.  This         */
/* function is almost entirely taken from dstarStub.c.                  */
/************************************************************************/
double pointGoalArcAngle(pplanner_state_type* state, int verbose)
{
  float localX, localY, radius;
  POINT3D_TYPE globalPoint, rcPoint;
  double epsilon = 0.1, arcAngle = STRAIGHT_ARC;
  NOMAD_POSE_TYPE pose;
  MP_POSE pointGoal;

  pose.x = state->curr.x;
  pose.y = state->curr.y;
  pose.z = state->curr.z;
  pose.yaw = state->curr.yaw;
  pose.roll = 0.0;
  pose.leftPitch = pose.rightPitch = 0.0;
  pointGoal = state->goal;

  globalPoint.x = pointGoal.x;  globalPoint.y = pointGoal.y;
  globalPoint.z = 0.0;
  rcPoint = Global_to_RC(rRotMat(pose), globalPoint);
  localX = rcPoint.x + BOGIE_OFFSET_X;  localY = rcPoint.y + BOGIE_OFFSET_Y;

  if (verbose)
    printf("Local goal: (%.2f, %.2f), Turn: ", localX, localY);
  /* Local coordinates: Y points forward, X points to the right */
  if (localY < 0) {
    /* Need to turn around; really should favor *both* ends of the arc set */
    if (verbose) printf("Turning around\n");
    if (localX < 0)
      arcAngle = PI - atan(state->minTurnRad);
    else
      arcAngle = atan(state->minTurnRad);
  }
  else if (ABS(localX) < epsilon) {
    if (verbose) printf("Straight\n");
    arcAngle = STRAIGHT_ARC;
  }
  else {
    radius = (SQR(localX) + SQR(localY))/(2.0*localX);
    if (verbose)
      printf("%s (r=%.2f) ", (radius < 0 ? "Left" : "Right"), radius);

    /* Result is angle starting at zero = positive??? X */
    arcAngle = atan2(localY, localX);

    if (fabs(tan(arcAngle)) < state->minTurnRad) {
      if (radius < 0)
	arcAngle = atan(-1.0 * state->minTurnRad);
      else
	arcAngle = atan(state->minTurnRad);
    }
    if (arcAngle < 0) arcAngle += PI;
  }

  return arcAngle;
}


/************************************************************************/
/* This function takes the waypoint plan state, and checks to see if    */
/* the robot has reached the goal yet.  If so, it returns a 1, else it  */
/* calls pointGoalArcAngle to calculate the command to send the robot   */
/* and returns a 0.                                                     */
/************************************************************************/
int pplanner(pplanner_state_type *state, plan_command *command, int verbose)
{
  double arcAngle=0.0;
  int complete = 0;

  if (goalReached(state->curr, state->goal, state->tolerance)) {
    printf("Goal Reached!! Desired: (%.2f, %.2f) Actual: (%.2f, %.2f)\n",
	   state->goal.x, state->goal.y, state->curr.x, state->curr.y);
    complete = 1;
  }
  else
    arcAngle = pointGoalArcAngle(state, verbose);

  command->turn = arcAngle;
  command->speed = state->speed;

  return(complete);
}
