/**************************************************************************/
/* This file contains functions for following a path segment using the    */
/* pure pursuit algorithm.  It is generalized for any path segment struct */
/* no matter what pattern type is selected.                               */
/**************************************************************************/
#include <stdio.h>
#include <math.h>
#include "navplan.h"


/**************************************************************************/
/* This function copies the old path segment struct data into the new     */
/* path segment struct.                                                   */
/**************************************************************************/
void copy_segment(path_seg* new, path_seg old) {
  new->curvature = old.curvature;
  new->start.x = old.start.x; new->start.y = old.start.y;
  new->start.yaw = old.start.yaw;
  new->end.x = old.end.x; new->end.y = old.end.y;
  new->end.yaw = old.end.yaw;
  new->center.x = old.center.x; new->center.y = old.center.y;
  new->center.yaw = old.center.yaw;
  new->goal.x = old.goal.x; new->goal.y = new->goal.y;
  new->goal.yaw = old.goal.yaw;
  new->dir = old.dir;
  new->lookahead = old.lookahead;
  new->type = old.type;
}


/**************************************************************************/
/* This function finds the closest point of a path segment to the current */
/* location.  The inputs are the path segment and the current pose,       */
/* and the output is a MP_POSE point.                             */
/**************************************************************************/
MP_POSE find_closest(path_seg segment, MP_POSE curr) {
  MP_POSE closest;
  float angle_closest, angle_start, angle_end;
  float slope;
  float radius;

  if (segment.curvature > 500.0) {
    /* straight path segment */
    /* drop line from curr, perpendicular to line, find intersection */
    if (segment.end.x == segment.start.x) {
      /* infinite slope */
      closest.x = segment.end.x;
      closest.y = curr.y;
    }
    else {
      slope = (segment.end.y - segment.start.y) /
	      (segment.end.x - segment.start.x);
      if (slope == 0.0) {
	/* zero slope */
	closest.x = curr.x;
	closest.y = segment.end.y; }
      else {
	closest.x = (curr.y + curr.x / slope - segment.start.y +
		     segment.start.x * slope) / (slope + 1.0 / slope);
	closest.y = curr.y - (closest.x - curr.x) / slope; }
    }

    /* check to see if intersection point is within segment */
    if (segment.end.x - segment.start.x > SMALL_DIST) {
      if (closest.x - segment.end.x > -SMALL_DIST)
	/* we're within SMALL_DIST or greater past end point */
	closest = segment.end;
      else if (closest.x - segment.start.x < SMALL_DIST)
	/* we're within SMALL_DIST or greater before start point */
	closest = segment.start;
    }
    else if (segment.end.x - segment.start.x < -SMALL_DIST) {
      if (closest.x - segment.start.x > -SMALL_DIST) closest = segment.start;
      else if (closest.x - segment.end.x < SMALL_DIST) closest = segment.end;
    }
    else if (segment.end.y > segment.start.y) {
      if (closest.y - segment.end.y > -SMALL_DIST) closest = segment.end;
      else if (closest.y - segment.start.y < SMALL_DIST)
	closest = segment.start;
    }
    else if (segment.end.y < segment.start.y) {
      if (closest.y - segment.start.y > -SMALL_DIST) closest = segment.start;
      else if (closest.y - segment.end.y < SMALL_DIST) closest = segment.end;
    }
  }

  else {
    /* circular path segment */
    radius = sqrt(sqr(segment.start.x - segment.center.x) +
		  sqr(segment.start.y - segment.center.y));

    /* Special case:  in line with circle center in x direction */
    if (segment.center.x == curr.x) {
      closest.x = segment.center.x;
      if (curr.y > segment.center.y) closest.y = segment.center.y + radius;
      else closest.y = segment.center.y - radius; }
    /* Normal case: */
    else {
      slope = (segment.center.y - curr.y) / (segment.center.x - curr.x);
      /* find closest.y */
      if (curr.y > segment.center.y) closest.y = segment.center.y +
	     radius * fabs(slope) / sqrt(1.0 + sqr(slope));
      else closest.y = segment.center.y -
	     radius * fabs(slope) / sqrt(1.0 + sqr(slope));
      /* find closest.x */
      /* Special case:  in line with circle center in y direction */
      if (slope == 0.0) {
	if (curr.x > segment.center.x) closest.x = segment.center.x + radius;
	else closest.x = segment.center.x - radius; }
      /* back to normal case */
      else
	closest.x = segment.center.x + (closest.y - segment.center.y) / slope;
    }

    /* check to see if within path segment */
    angle_start = atan2(segment.start.y - segment.center.y,
			segment.start.x - segment.center.x);
    angle_closest = atan2(closest.y - segment.center.y,
			  closest.x - segment.center.x);
    angle_end = atan2(segment.end.y - segment.center.y,
		      segment.end.x - segment.center.x);

    if (fabs(angle_closest - angle_start) < SMALL_ANGLE)
      /* within half a degree of start */
      closest = segment.start;
    if (fabs(angle_closest - angle_end) < SMALL_ANGLE)
      /* within half a degree of end */
      closest = segment.end;

    else if (segment.dir == 1) {  /* clockwise */
      if (angle_end > angle_start)
	angle_end = angle_end - 2.0 * PI;
      if (angle_closest > angle_start)
	  angle_closest = angle_closest - 2.0 * PI;
      if (angle_closest < angle_end) {
	/* we're not within segment - find which endpoint we're closer to */
	if (fabs(angle_diff(angle_closest, angle_end)) <
	    fabs(angle_diff(angle_closest, angle_start)))
	  /* we're closer to end */
	  closest = segment.end;
	else
	  /* we're closer to start */
	  closest = segment.start;
      }
    }
    else {  /* counterclockwise */
      if (angle_end < angle_start)
	angle_end = angle_end + 2.0 * PI;
      if (angle_closest < angle_start)
	angle_closest = angle_closest + 2.0 * PI;
      if (angle_closest > angle_end) {
	/* we're not within segment - find which endpoint we're closer to */
	if (fabs(angle_diff(angle_closest, angle_end)) <
	    fabs(angle_diff(angle_closest, angle_start)))
	  /* we're closer to end */
	  closest = segment.end;
	else
	  /* we're closer to start */
	  closest = segment.start;
      }
    }
  }
  return(closest);
}


/**************************************************************************/
/* This function takes the current point and the closest path segment     */
/* point (from find_closest), and finds a path segment point which is a   */
/* lookahead distance away from the current point.  If the closest point  */
/* is more than a lookahead distance away, the closest point is chosen.   */
/* If the current point is close enough to the path segment endpoint, the */
/* end of the path segment is assumed to be reached, and a value of 1 is  */
/* returned (otherwise, 0).  This "close enough" distance is SMALL_DIST,  */
/* for a straight path segment, or SMALL_ANGLE, for a circular path       */
/* segment, and is used in find_closest.                                  */
/* The chosen path segment point is stored in the path segment struct in  */
/* the goal field, and is used for the pure pursuit algorithm.            */
/*                                                                        */
/* The lookahead distance is stored in the path segment struct, but the   */
/* next segment's lookahead distance is also used -- if robot is closer   */
/* to the current path segment's endpoint than the next segment's         */
/* lookahead distance, we switch to using the next path segment (and its  */
/* lookahead distance) to find the goal point.                            */
/**************************************************************************/
int find_goal(path_seg* segment, MP_POSE curr, MP_POSE closest)
{
  int value = 0;
  float dist, slope;
  float radius, theta, yaw;
  float deltax, deltay, x;
  float theta2, deltax2, deltay2;
  MP_POSE point, cl, testpt;
  float angle_start, angle_end, angle_testpt;
  int next = 0;

  /* if closest point is farther away from current point than lookahead 
     distance, set goal to be the closest point */
  dist = sqrt(sqr(curr.y - closest.y) + sqr(curr.x - closest.x));
  if (dist > segment->lookahead) segment->goal = closest;

  /* else, if closest point is the end point then we're done, set value to 1 */
  /* note: closest should be set to exactly segment->end in find_closest
     function already, if we were close enough */
  else if (sqrt(sqr(closest.y - segment->end.y) +
		sqr(closest.x - segment->end.x)) < 0.001)
    value = 1;

  /* else, move up from closest point until we find a point the lookahead
     distance away, along the path segment.  If the end of the path segment
     is close enough, find a point on the NEXT path segment that is a
     lookahead distance away (the lookahead distance may be different
     in this case). */
  else {
    /* We're on a straight path segment */
    if (segment->curvature > 500.0) {
      /* Special case:  vertical slope */
      if (segment->end.x - segment->start.x == 0.0) {
	segment->goal.x = segment->start.x;
	if (segment->end.y > closest.y) {
	  segment->goal.y =
	    closest.y + sqrt(sqr(segment->lookahead) - sqr(dist));
          if (segment->next != NULL &&
	      fabs(segment->end.y - closest.y) < segment->next->lookahead)
	    /* move goal to being on NEXT path segment */
	    next = 1;
	}
	else {
	  segment->goal.y =
	    closest.y - sqrt(sqr(segment->lookahead) - sqr(dist));
          if (segment->next != NULL &&
	      fabs(segment->end.y - closest.y) < segment->next->lookahead)
	    /* move goal to being on NEXT path segment */
	    next = 1;
	}
      }
      else { /* Normal case:  slope is not vertical */
	slope = (segment->end.y - segment->start.y) /
	        (segment->end.x - segment->start.x);
	/* find goal.y */
	if (segment->end.y > closest.y)
	  segment->goal.y = closest.y + fabs(slope) / sqrt(1.0 + sqr(slope)) *
	    sqrt(sqr(segment->lookahead) - sqr(dist));
	else
	  segment->goal.y = closest.y - fabs(slope) / sqrt(1.0 + sqr(slope)) *
	    sqrt(sqr(segment->lookahead) - sqr(dist));
	/* find goal.x */
	if (slope == 0.0) { /* Special case:  horizontal slope */
	  if (segment->end.x > closest.x) segment->goal.x = closest.x +
			  sqrt(sqr(segment->lookahead) - sqr(dist));
	  else segment->goal.x = closest.x -
		 sqrt(sqr(segment->lookahead) - sqr(dist));
	}
	else /* (back to normal case) */
	  segment->goal.x = closest.x + (segment->goal.y - closest.y) / slope;

        if (segment->next != NULL &&
	    sqrt(sqr(segment->end.x - closest.x) +
		 sqr(segment->end.y - closest.y)) < segment->next->lookahead)
	  /* move goal to being on NEXT path segment */
          next = 1;
      }

      if (next) {
	/* move to NEXT path segment */
	closest = segment->next->start;
	find_goal(segment->next, curr, closest);
	/* set new goal */
	segment->goal = segment->next->goal; }
    }

    /* We're on a circular path segment */
    else {
      /* we will need to use direction of tangent to circle at closest
	 point, or perpendicular to closest-center line */
      /* Special case: */
      if (sqrt(sqr(closest.x - segment->start.x) +
	       sqr(closest.y - segment->start.y)) < 0.001) {
	/* closest point on segment is the start point - geometry may
	   not be the same, so find a new closest point to use for
	   calculation - in line from current position to circle center */
	/* (the following code is basically the same as in find_closest) */
	radius = sqrt(sqr(segment->start.x - segment->center.x) +
		      sqr(segment->start.y - segment->center.y));
	if (fabs(segment->center.x - curr.x) < 0.005) {
	  cl.x = segment->center.x;
	  if (curr.y > segment->center.y) cl.y = segment->center.y + radius;
	  else cl.y = segment->center.y - radius; }
	else {
	  slope = (segment->center.y - curr.y) / (segment->center.x - curr.x);
	  if (curr.y > segment->center.y) cl.y = segment->center.y +
		 radius * fabs(slope) / sqrt(1.0 + sqr(slope));
	  else cl.y = segment->center.y -
		 radius * fabs(slope) / sqrt(1.0 + sqr(slope));
	  if (slope == 0.0) {
	    if (curr.x > segment->center.x) cl.x = segment->center.x + radius;
	    else cl.x = segment->center.x - radius; }
	  else cl.x = segment->center.x + (cl.y - segment->center.y) / slope;
	}
      }
      /* Normal case: */
      else { /* original closest point was good */
	cl.x = closest.x;
	cl.y = closest.y; }

      /* find the relative coordinates point a lookahead away from closest */
      /* (consider 2 lookaheads - current and next) */
      radius = sqrt(sqr(segment->start.x - segment->center.x) +
		    sqr(segment->start.y - segment->center.y));
      x = sqrt(sqr(curr.x - cl.x) + sqr(curr.y - cl.y));
      if (sqrt(sqr(curr.x - segment->center.x) +
	       sqr(curr.y - segment->center.y)) < radius)
	x = -1.0 * x;
      theta = acos((2.0 * sqr(radius) + sqr(x) + 2.0 * x * radius -
		    sqr(segment->lookahead)) / (2.0 * radius * (radius + x)));
      deltax = radius * (1.0 - fabs(cos(theta)));
      if (segment->dir == 2) deltax = -1.0 * deltax; /* counterclockwise */
      deltay = radius * fabs(sin(theta));

      if (segment->next != NULL) {
	theta2 = acos((2.0 * sqr(radius) + sqr(x) + 2.0 * x * radius -
		       sqr(segment->next->lookahead)) /
		      (2.0 * radius * (radius + x)));
	deltax2 = radius * (1.0 - fabs(cos(theta2)));
	if (segment->dir == 2) deltax2 = -1.0 * deltax2; /* counterclockwise */
	deltay2 = radius * fabs(sin(theta2)); }

      /* converting to global coordinates... */
      yaw = atan2(cl.x - segment->center.x, segment->center.y - cl.y);
      if (segment->dir == 1) cl.yaw = yaw + PI/2.0; /* clockwise */
      else cl.yaw = yaw - PI/2.0; /* counterclockwise */

      point.x = deltax;
      point.y = deltay;
      point.yaw = 0.0;
      segment->goal = RelToGlo(point, cl);

      /* check this point too, to see if we want to switch to next segment */
      if (segment->next != NULL) {
	point.x = deltax2;
	point.y = deltay2;
	testpt = RelToGlo(point, cl);

	/* check to see if testpt is past segment->end */
	/* (this code is also taken from find_closest) */
	angle_start = atan2(segment->start.y - segment->center.y,
			    segment->start.x - segment->center.x);
	angle_testpt = atan2(testpt.y - segment->center.y,
			     testpt.x - segment->center.x);
	angle_end = atan2(segment->end.y - segment->center.y,
			  segment->end.x - segment->center.x);

	if (segment->dir == 1) {  /* clockwise */
	  if (angle_end > angle_start) angle_end = angle_end - 2.0 * PI;
	  if (angle_testpt > angle_start)
	    angle_testpt = angle_testpt - 2.0 * PI;
	  if (angle_testpt < angle_end) next = 1; /* not within segment */ }
	else {
	  if (angle_end < angle_start) angle_end = angle_end + 2.0 * PI;
	  if (angle_testpt < angle_start)
	    angle_testpt = angle_testpt + 2.0 * PI;
	  if (angle_testpt > angle_end) next = 1; /* not within segment */ }

	if (next) {
	  /* move to next path segment */
	  closest = segment->next->start;
	  find_goal(segment->next, curr, closest);
	  /* set new goal */
	  segment->goal = segment->next->goal; }
      }
    }
  }
  return(value);
}


/**************************************************************************/
/* This functions inputs are:                                             */
/*     current robot position                                             */
/*     closest point on path from find_closest                            */
/*     minimum turning radius of the robot                                */
/*     path segment struct (including goal point from find_goal)          */
/* It calculates the turning angle the robot must use to reach that goal  */
/* and returns it.                                                        */
/* The function uses SMALL_ANGLE to tell if robot is too far off from a   */
/* straight path segment, and SMALL_DIST to tell if robot is too far off  */
/* from a circular path segment.  The turning radius is calculated as     */
/* CORR_GAIN divided by the absolute value of the difference in yaw       */
/* between the current robot yaw and the yaw direction from the current   */
/* robot position to the goal.                                            */
/**************************************************************************/
float follow_path(path_seg currseg, float minTurnRad, MP_POSE curr,
		  MP_POSE closest, int verbose) {
  float yaw, yawdiff, yawsign;
  float turn, radius = 1000.0;
  float dist, circ_radius;

  /* yaw direction from current position to goal point */
  yaw = atan2(curr.x - currseg.goal.x, currseg.goal.y - curr.y);
  yawdiff = angle_diff(yaw, curr.yaw);
  yawsign = yawdiff / fabs(yawdiff);

  if (currseg.curvature > 500.0) {  /* traveling straight path */
    if (fabs(yawdiff) > SMALL_ANGLE) {  /* too far off: turn first */
      if (verbose)
	printf("Too far off straight (%.2f deg, %.2f m)\n",
	       yawdiff * 180.0 / PI, GPS_diff(curr, closest));
      radius = CORR_GAIN / fabs(yawdiff);
      /* we can't turn sharper than minTurnRad */
      if (radius < minTurnRad) radius = minTurnRad;
      if (yawsign > 0) turn = PI - atan(radius);  /* left turn */
      else turn = atan(radius);  /* right turn */ }
    else {  /* just go straight */
      if (verbose) printf("Going straight forwards\n");
      turn = PI/2.0; }
  }

  else {   /* traveling a circular path segment */
    /* check how far off we are from path */
    dist = GPS_diff(curr, currseg.center);
    circ_radius = GPS_diff(currseg.center, currseg.start);
    if (verbose) printf("segment radius = %.2f\n", circ_radius);
    if (fabs(dist - circ_radius) > SMALL_DIST) {
      /* correct path, using goal */
      if (verbose)
	printf("Too far off circle (%.2f m)\n", dist - circ_radius);
      radius = CORR_GAIN / fabs(yawdiff);
      if (radius < minTurnRad) radius = minTurnRad;
      if (yawsign > 0) turn = PI - atan(radius);  /* left turn */
      else turn = atan(radius);  /* right turn */
    }
    else {
      /* continue turning at constant curvature around end */
      if (verbose) printf("Turning at constant rate\n");
      radius = currseg.curvature;
      if (radius < minTurnRad) radius = minTurnRad;
      if (currseg.dir == 1) turn = atan(radius);  /* turn right, clockwise */
      else turn = PI - atan(radius);  /* turn left, counterclockwise */
    }
  }

  if (verbose) {
    if (fabs(fabs(turn) - PI/2.0) < 0.001) printf("turning radius = 1000.0\n");
    else printf("turning radius = %.2f\n", tan(turn)); }

  return(turn);
}


/**************************************************************************/
/* This function simulates the movement of the robot, given the turning   */
/* command which is otherwise given to arbiter.  It is for planning paths */
/* without actually moving the robot.  It converts a turning angle arc to */
/* the robot's new position, based on current position, turning arc,      */
/* desired speed, and amount of time to move.                             */
/**************************************************************************/
void arc_to_pos(MP_POSE *curr, float arc, float sp, float dt) {
  float deltax, deltay;
  float newarc;

  if (fabs(arc-PI) < 0.0001 || arc == 0.0) {
    /* point turn */
    deltax = 0.0;
    deltay = 0.0;
    newarc = arc;
  }
  else if (fabs(fabs(arc)-PI/2.0) < 0.001) {
    deltax = 0.0;
    if (arc < 0.0)
      deltay = -sp * dt;
    else
      deltay = sp * dt;
    newarc = arc;
  }
  else {
    /* arc is not really true turning angle, but atan(radius) */
    newarc = sp * dt / tan(arc);

    deltax = tan(arc) * (1.0 - cos(fabs(newarc)));
    deltay = fabs(tan(arc)) * sin(fabs(newarc));
    if (arc > PI)
      deltay = -1 * deltay;
  }

  curr->x = curr->x + cos(curr->yaw) * deltax - sin(curr->yaw) * deltay;
  curr->y = curr->y + sin(curr->yaw) * deltax + cos(curr->yaw) * deltay;

  if (fabs(arc-PI) < 0.0001)  /* left point turn */
    curr->yaw = curr->yaw + sp * dt;
  else if (arc == 0.0)  /* right point turn */
    curr->yaw = curr->yaw - sp * dt;
  else if (fabs(fabs(arc)-PI/2.0) < 0.001)  /* going straight */
    curr->yaw = curr->yaw;
  /* what if it takes longer to turn to new heading, and we don't get
     there this time? should be more realistic */
  else if (arc < PI) /* moving forward */
    curr->yaw = curr->yaw - newarc;
  else  /* moving backward */
    curr->yaw = curr->yaw + newarc;

  while (curr->yaw > 2.0 * PI)
    curr->yaw = curr->yaw - 2.0 * PI;
  while (curr->yaw < 0.0)
    curr->yaw = curr->yaw + 2.0 * PI;

  return;
}
