#include "commonTypes.h"
#include "nomadDimensions.h"
#include "nomadTransforms.h"
#include "libc.h"
#include "DstarMsgStructs.h"
#include "PlanParams.h"

/* Path following parameters */
#define SMALL_ANGLE 0.01  /* angular difference we can be off */
#define SMALL_DIST 0.10    /* distance we can be off */
#define CORR_GAIN 0.8      /* correction gain for regaining path */
#define STR_LOOKAHEAD 5.0  /* lookahead distance to use for straight rows */
#define CIR_LOOKAHEAD 2.0  /* lookahead distance to use for row ends */
#define INCR_LOOKAHEAD 1.0 /* amount to increment lookahead for spiral */

#define PP_PIXEL_SIZE 0.5   /* 50 centimeters */
#define PP_BLOCK_SIZE 100

/* Types of tasks */
#define STRAIGHT_TYPE  1
#define SPIRAL_TYPE    2
#define SUNFOLLOW_TYPE 3
#define WAYPOINT_TYPE  4
#define MANEUVER_TYPE  5
#define WORKSPACE_TYPE 6

/* These are adjacent pixels or grid blocks */
#define UP 0
#define UPPERRIGHT 1
#define RIGHT 2
#define LOWERRIGHT 3
#define DOWN 4
#define LOWERLEFT 5
#define LEFT 6
#define UPPERLEFT 7


/* Structures used in multiple files */
typedef struct polygon_t {
  MP_POSE vertex;
  struct polygon_t* next;
} polygon;   /* area to be covered */

typedef struct maneuver_step_t {
  MP_POSE desired;                   /* position robot should be at */
  float turn, speed;                 /* next commanded move */
  struct maneuver_step_t *next;      /* link to next step */
} maneuver_step;   /* step in maneuver plan, for every 5 seconds */

typedef struct pathseg {
  float curvature;            /* curvature of segment, in meters */
  MP_POSE start, end;         /* start and end points of segment */
  MP_POSE center;             /* center of circular path segment */
  MP_POSE goal;               /* pure pursuit semi-waypoint */
  int dir;                    /* 1 = clockwise, 2 = counterclockwise */
  float lookahead;            /* lookahead distance to use for this segment */
  int type;                   /* type of path segment */
  struct pathseg *next;       /* the next path_seg in the plan */
} path_seg;  /* path segment to follow */

typedef struct {
  int topedge, bottomedge; /* edge numbers from area polygon */
  int hand;                /* 1=left pattern, 2=right pattern */
  float straight;          /* yaw of the primary direction of travel */
  polygon* area;           /* area we want to cover */
  MP_POSE curr;            /* current position */
  path_seg currseg;        /* current path segment */
  float speed;             /* max speed desired */
  float sensWidth;         /* sensor width, in meters */
  float rowWidth;          /* width of row desired */
  float minTurnRad;        /* minimum turning radius */
} straight_state_type;  /* Farming plan structure */

typedef struct {
  MP_POSE top_center, bot_center;
  /* center of spiral, one for top half, one for bottom half */
  int half;              /* 1 = top half, 0 = bottom half */
  float maxrad;          /* maximum radius of pattern */
  MP_POSE curr;  /* current position */
  path_seg currseg;      /* current path segment */
  float speed;           /* max speed desired */
  float sensWidth;       /* sensor width */
  float rowWidth;        /* width of row desired */
  float minTurnRad;      /* minimum turning radius */
} spiral_state_type;     /* Spiral plan structure */
 
typedef struct {
  MP_POSE goal;
  float tolerance;
  int goal_type;         /* 1 = global; 2 = relative */
  MP_POSE curr;          /* current robot position */
  float speed;           /* max speed desired */
  float sensWidth;       /* sensor width */
  float minTurnRad;      /* minimum turning radius */
} pplanner_state_type;   /* Waypoint plan structure */

typedef struct {
  MP_POSE pos;
  float disttol;   /* tolerance in distance from goal */
  float angtol;    /* tolerance in angular distance from goal */
} mangoals;

typedef struct {
  mangoals Goal; /* goal pose */
  int man_type;    /* 0 = maneuver to given yaw, 1 = maneuver into workspace */
  maneuver_step* ManPlan;
  int step;                 /* number of seconds we've done current command */
  MP_POSE curr;
  float speed;
  float sensWidth;
  float minTurnRad;
} maneuver_state_type;  /* Maneuver plan structure */

typedef struct {
  float turn;         /* turning radius desired */
  float speed;        /* speed desired */
} plan_command;  /* Movement command to be sent to arbiter/simulator */

typedef struct _npstate{
  int type;  /* farming, spiral, sun-following, waypoint, maneuver */
  straight_state_type STRAIGHT_state;
  spiral_state_type SPIRAL_state;
  /* sun-following state - not implemented yet */
  pplanner_state_type PATH_state;
  maneuver_state_type MAN_state;
  struct _npstate *next;  /* for a series of tasks */
} NPState;  /* State structure for all plans */

typedef struct {
  int commanding;     /* are we sending commands to arbiter now? */
  NPState* curr;      /* currently active plan */
  NPState* previous;  /* previous plan we were doing (storage) */
} NPTasks;  /* Task list structure, global */


/* Structures in ppqueue.c */
typedef struct ppqueue_t {
  void *data;
  struct ppqueue_t *next;
} *ppqueue;


/* Structures in grid.c */
typedef struct {
  int i, j;              /* array elements for this pixel */
  float elevation;       /* average? elevation/z-coord of pixel */
  char contents;         /* contents: e.g. target object, crevasse, snow */
  char sensors;          /* sensor coverage: e.g. camera, radar */
  long int when;         /* time when last covered */
  struct grid_block_t* block;   /* address of "parent" grid block */
} map_pixel;

typedef struct grid_block_t {
  map_pixel grid[PP_BLOCK_SIZE][PP_BLOCK_SIZE];
  MP_POSE corner; /* coordinates of lower left corner of grid block */
  struct grid_block_t *connect[8];
   /* 1st element is up, increasing in clockwise order, NULL if not defined */
} grid_block;

typedef struct {
  grid_block* center;
} planner_map;

/* General functions in navplan.c */
float sqr(float number);
MP_POSE RelToGlo(MP_POSE point, MP_POSE curr);
float GPS_diff(MP_POSE one, MP_POSE two);
float angle_diff(float one, float two);
void free_polygon(polygon *area);

/* Functions in straight.c */
void init_straight(straight_state_type* state, plan_command* command);
int straightrows(straight_state_type *state, plan_command *command, int verbose);

/* Functions in spiral.c */
void init_spiral(spiral_state_type* state);
int spiral(spiral_state_type* state, plan_command* command, int verbose);

/* Functions in maneuver.c */
int man_planner(maneuver_state_type* state);
int maneuver(maneuver_state_type* state, plan_command *command, int verbose);

/* Functions in pplanner.c */
int pplanner(pplanner_state_type *state, plan_command *command, int verbose);

/* Functions in ppqueue.c */
ppqueue ppinit_queue(void); /* Initialize the queue - call first */
int ppempty_queue(ppqueue q);  /* Returns 1 if empty, 0 otherwise */
void ppenqueue(ppqueue *q, void *data);  /* Adds data to end of queue q */
void *ppdequeue(ppqueue *q); /* Removes & returns top queue element */
int ppin_queue(ppqueue *q, void *item);  /* Returns 1 if item is in queue */
void ppfree_queue(ppqueue *q); /* Empties queue q and frees all memory */

/* Functions in pathfollow.c */
void copy_segment(path_seg* new, path_seg old);
MP_POSE find_closest(path_seg segment, MP_POSE curr);
int find_goal(path_seg* segment, MP_POSE curr,
	      MP_POSE closest);
float follow_path(path_seg currseg, float minTurnRad, MP_POSE curr,
		  MP_POSE closest, int verbose);
void arc_to_pos(MP_POSE *curr, float arc, float sp, float dt);

/* Functions in grid.c */
void init_map(planner_map* map, MP_POSE curr);
void save_planner_map(planner_map map, char name[255]);
void save_command_map(planner_map map, char name[255]);
void mark_planning_map(planner_map* map, NPState state, int time, float* area);
void mark_map(planner_map* map, NPState state, float* area);
map_pixel* find_pixel_addr(planner_map map, MP_POSE curr, int* status);
MP_POSE find_pixel_GPS(map_pixel* curr);
void GPS_grid_block(planner_map* map, MP_POSE curr);

