#include <stdio.h>
#include <math.h>
#include "navplan.h"

/************************************************************************/
/* This function returns the (float) square of a (float) number.        */
/************************************************************************/
float sqr(float number) {
  return(number * number);
}

/************************************************************************/
/* This function converts relative-to-robot coordinates to global       */
/* coordinates.  (a yaw of 0 implies pointing in the +y direction)      */
/* Input values: point which we are converting in robot coordinates     */
/*               current pose of the robot in global coordinates        */
/* Output value: converted point in global coordinates                  */
/************************************************************************/
MP_POSE RelToGlo(MP_POSE point, MP_POSE curr) {
  MP_POSE newpoint;
 
  newpoint.x = curr.x + cos(curr.yaw) * point.x - sin(curr.yaw) * point.y;
  newpoint.y = curr.y + sin(curr.yaw) * point.x + cos(curr.yaw) * point.y;
  newpoint.yaw = curr.yaw + point.yaw;
  newpoint.z = curr.z;
  newpoint.roll = curr.roll;
  newpoint.pitch = curr.pitch;
  while (newpoint.yaw > 2.0 * PI)
    newpoint.yaw = newpoint.yaw - 2.0 * PI;
  while (newpoint.yaw < 0)
    newpoint.yaw = newpoint.yaw + 2.0 * PI;
 
  return(newpoint);
}
 
/************************************************************************/
/* This function returns the absolute distance between two GPS coords.  */
/* (only 2-dimensional)                                                 */
/************************************************************************/
float GPS_diff(MP_POSE one, MP_POSE two) {
  float diff;

  diff = sqrt(sqr(one.x - two.x) + sqr(one.y - two.y));

  return(diff);
}

/************************************************************************/
/* This function returns the smallest angular difference between two    */
/* angles, taking care of the 0/2PI duality.  The number returned is    */
/* between -PI and PI.  If traveling from one to two through the        */
/* smallest angle is clockwise, the difference returned is positive.    */
/* If the direction is counterclockwise, the difference is negative.    */
/************************************************************************/
float angle_diff(float one, float two) {
  float diff;
 
  /* make sure angles are between 0 and 2PI */
  while (one > 2.0 * PI) one = one - 2.0 * PI;
  while (one < 0) one = one + 2.0 * PI;
  while (two > 2.0 * PI) two = two - 2.0 * PI;
  while (two < 0) two = two + 2.0 * PI;
 
  diff = one - two;

  /* give smallest angle, between -PI and PI */
  if (diff > PI)
    diff = diff - (2.0 * PI);
  else if (diff < -PI)
    diff = diff + (2.0 * PI);
 
  return(diff);
}

/************************************************************************/
/* This function frees the memory in a polygon data structure.          */
/************************************************************************/
void free_polygon(polygon *area) {
  polygon *temp;
 
  temp = area;
  if (temp->next != NULL)
    free_polygon(temp->next);
 
  free(temp);
}

