#include <stdio.h>
#include "navplan.h"
#include "maneuver.h"
#include "PlanParams.h"
#include "navManMsgs.h"
#include "mPlannerMsgs.h"

int goal(MP_POSE curr, mangoals Goal, int type) {
  int value = 0;  /* 1 if goal, 0 if not */
  float angle;

  /* see if curr is close enough to Goals[i] */
  if (sqrt(sqr(curr.x-Goal.pos.x) + sqr(curr.y-Goal.pos.y)) < Goal.disttol) {
    if (type == 0) { /* maneuver to given yaw */
      if (minAngle(curr.yaw, Goal.pos.yaw) < Goal.angtol)
	value = 1;
    }
    else {
      angle = atan2(Goal.pos.y - curr.y, Goal.pos.x - curr.x) - PI/2.0;
      if (minAngle(curr.yaw, angle) < Goal.angtol)
	value = 1;
    }
  }
  return(value);
}


float minAngle(float angle1, float angle2) {
  float diff;

  while (angle1 > 2.0 * PI) angle1 = angle1 - 2.0 * PI;
  while (angle1 < 0.0) angle1 = angle1 + 2.0 * PI;

  while (angle2 > 2.0 * PI) angle2 = angle2 - 2.0 * PI;
  while (angle2 < 0.0) angle2 = angle2 + 2.0 * PI;

  diff = fabs(angle1 - angle2);
  if (diff > PI) diff = 2.0 * PI - diff;

  return(diff);
}

void read_model(float model[84][7]) {
  FILE *info;
  int i, j;

  info = fopen("model.txt", "r");
  if (info == NULL)
    printf("Model.txt is not in this directory - cannot process\n");
  else {
    for (i=0; i<84; ++i) {
      for (j=0; j<7; ++j)
	fscanf(info, "%f", &(model[i][j])); }
  }

  fclose(info);
}


void find_stop(float model[84][7], float stopmodel[6][7]) {
  int i, j;

  for (i=42; i<48; ++i) {
    for (j=0; j<7; ++j)
      stopmodel[i-42][j] = model[i][j];
  }
}


tree_list* lowestPriority(tree_list* tree) {
  tree_list *temp, *best;
  float priority;

  if (tree == NULL)
    printf("Error: lowestPriority - tree is null\n");
  temp = tree;
  best = tree;
  priority = temp->priority;
  while (temp != NULL) {
    if (temp->priority < priority) {
      best = temp;
      priority = temp->priority;
    }
    temp = temp->next;
  }

  return(best);
}


float heuristic(MP_POSE pose, mangoals Goal, int type) {
  float dist = 0.0;
  float mindist = 1000.0;
  float yawdiff = 0.0;
  float angle;

  /* just consider linear distance, not angular, to get an optimistic
     estimate of cost - also subtract distance tolerance */
  dist = sqrt(sqr(pose.x - Goal.pos.x) + sqr(pose.y - Goal.pos.y))
    - Goal.disttol;
  if (type == 0)  /* maneuver to given yaw */
    yawdiff = minAngle(pose.yaw, Goal.pos.yaw) - Goal.angtol;
  else { /* maneuver into workspace */
    angle = atan2(Goal.pos.y - pose.y, Goal.pos.x - pose.x) - PI/2.0;
    yawdiff = minAngle(pose.yaw, angle) - Goal.angtol;
  }
  if (yawdiff < 0) yawdiff = 0;
  if (dist + yawdiff < mindist) mindist = dist + yawdiff;

  return(mindist);
}


void addItem(tree_list* list, tree_list item) {
  tree_list* temp;

  temp = list;
  if (temp == NULL)
    printf("Error: addItem - list is null\n");
  else {
    while (temp->next != NULL) temp = temp->next;
    temp->next = (tree_list *)malloc(sizeof(tree_list));
    temp->next->prev = temp;
    temp = temp->next;
    temp->curr_pose = item.curr_pose;
    temp->curr_angle = item.curr_angle;
    temp->curr_speed = item.curr_speed;
    temp->turn = item.turn;
    temp->speed = item.speed;
    temp->priority = item.priority;
    temp->parent = item.parent;
    temp->next = NULL;
  }

  return;
}

tree_list* findParent(tree_list *tree, tree_list *current) {
  tree_list *parent, *temp;

  temp = tree;
  while ((temp != NULL) && ((sqrt(sqr(temp->curr_pose.x - current->curr_pose.x)
		   + sqr(temp->curr_pose.y - current->curr_pose.y)) > 0.01) ||
	 (fabs(temp->curr_pose.yaw - current->curr_pose.yaw) > 0.01))) {
    /* we haven't found current in the list yet */
    temp = temp->next;
  }
  if (temp == NULL)
    printf("Error: findParent - Item was not in tree list\n");
  else
    parent = temp;

  return(parent);
}

void changeItem(tree_list* list, tree_list item) {
  tree_list* temp;

  temp = list;
  while ((temp != NULL) && ((sqrt(sqr(temp->curr_pose.x - item.curr_pose.x)
		   + sqr(temp->curr_pose.y - item.curr_pose.y)) > 0.01) ||
	 (fabs(temp->curr_pose.yaw - item.curr_pose.yaw) > 0.01))) {
    /* we haven't found item in the list yet */
    temp = temp->next;
  }
  if (temp == NULL)
    printf("Error: changeItem - not in the list\n");
  else {
    temp->priority = item.priority;
    temp->parent = item.parent;
  }
}

void removeLeaf(tree_list* list, tree_list* current) {
  tree_list *temp, *parent, *child;

  temp = list;
  while ((temp != NULL) && ((sqrt(sqr(temp->curr_pose.x - current->curr_pose.x)
		   + sqr(temp->curr_pose.y - current->curr_pose.y)) > 0.01) ||
	 (fabs(temp->curr_pose.yaw - current->curr_pose.yaw) > 0.01))) {
    /* we haven't found current in the list yet */
    temp = temp->next;
  }
  if (temp == NULL)
    printf("Error: removeLeaf - not in the list\n");
  else {
    if (temp->prev == NULL) {
      /* this is first item in the list - replace with next */
      /* replace 1st item with 2nd item */
      temp = list->next;
      list->curr_pose = temp->curr_pose;
      list->curr_angle = temp->curr_angle;
      list->curr_speed = temp->curr_speed;
      list->turn = temp->turn;
      list->speed = temp->speed;
      list->priority = temp->priority;
      list->next = temp->next;
      temp->next->prev = list;
      /* remove 2nd item */
      free(temp);
    }
    else {
      parent = temp->prev;
      child = temp->next;
      parent->next = child;
      child->prev = parent;
      free(temp);
    }
  }

  return;
}


/* this finds the plan to follow - run one time */
int man_planner(maneuver_state_type* state) {
  float model[84][7];
  float stopmodel[6][7];
  MP_POSE newpos;
  tree_list *tree, *leafs;
  tree_list Item;
  tree_list *temptree, *tempstep, *current, *parent;
  maneuver_step *temp;
  int value = -1;  /* = -1 if no plan found */
  int done, i, loop, addleaf;
  float hn, gn, hp;
  float yaw;
  float deltax, deltay;
  /* int foo; */

  read_model(model);
  find_stop(model, stopmodel);

  /* This is an A* search */
  /* initialize trees */
  leafs = (tree_list*)malloc(sizeof(tree_list));
  tree = (tree_list*)malloc(sizeof(tree_list));

  /* put starting leaf on lists */
  leafs->prev = NULL;
  leafs->parent = NULL;
  leafs->next = NULL;
  leafs->curr_pose = state->curr;
  leafs->curr_angle = FORWARD;  /* assumed? */
  leafs->curr_speed = STOPPED;  /* assumed */
  leafs->turn = 0.0;  /* meaningless for tree root */
  leafs->speed = 0.0; /* meaningless for tree root */
  leafs->priority = heuristic(state->curr, state->Goal, state->man_type);

  tree->prev = NULL;
  tree->parent = NULL;
  tree->next = NULL;
  tree->curr_pose = state->curr;
  tree->curr_angle = FORWARD;
  tree->curr_speed = STOPPED;
  tree->turn = 0.0;
  tree->speed = 0.0;
  tree->priority = leafs->priority;

  /* lower priority ==> lower cost ==> good leaf to try next */
  current = lowestPriority(leafs);
  done = 0;
  /* Start loop until no more leafs */
  while (current != NULL && !done) {
    hn = heuristic(current->curr_pose, state->Goal, state->man_type);
    gn = current->priority - hn;
    /* printf("%f, %f, %f: hn = %f, gn = %f\n", current->curr_pose.x, current->curr_pose.y, current->curr_pose.yaw, hn, gn); */
    /* calculate position if we stop now */
    if (current->curr_speed < 0.001)   /* already stopped */
      newpos = current->curr_pose;
    else {
      i = 0;
      /* find relevant row for current turning angle */
      while (fabs(current->curr_angle - stopmodel[i][1]) > 0.001)
	++i;
      yaw = current->curr_pose.yaw;
      deltax = cos(yaw) * stopmodel[i][4] - sin(yaw) * stopmodel[i][5];
      deltay = sin(yaw) * stopmodel[i][4] + cos(yaw) * stopmodel[i][5];
      newpos.x = current->curr_pose.x + deltax;
      newpos.y = current->curr_pose.y + deltay;
      newpos.yaw = yaw + stopmodel[i][6];
    }
    if (goal(newpos, state->Goal, state->man_type)) {
      done = 1;
      /* add a final step to stop the robot, if not already stopped */
      if (current->curr_speed > 0.001) {
	/* current item should be final step */
	parent = findParent(tree, current);
	current->curr_speed = STOPPED;
	current->speed = STOPPED;
	current->turn = current->curr_angle;
	current->parent = parent;
	/* current->prev will be set in addItem */
	current->next = NULL;
	current->curr_pose = newpos;
	addItem(leafs, *current);
	addItem(tree, *current);
      }
    }
    else {
      /* find address of current in tree list */
      parent = findParent(tree, current);
      /* find relevant entries in the model array */
      for (i = 0; i < 84; ++i) {
	if (fabs(model[i][0] - current->curr_speed) < 0.001 &&
	    fabs(model[i][1] - current->curr_angle) < 0.001 &&
	    model[i][4] != -99) {
	  /* this is a relevant entry */
	  Item.curr_speed = model[i][2];  /* we've reached this speed */
	  Item.curr_angle = model[i][3];  /* we've reached this angle */
	  Item.speed = model[i][2];  /* commanded speed */
	  Item.turn = model[i][3];   /* commanded turn  */
	  Item.parent = parent;      /* address in tree, not leafs */
	  /* Item.prev will be set in addItem */
	  Item.next = NULL;
	  yaw = current->curr_pose.yaw;
	  deltax = cos(yaw) * model[i][4] - sin(yaw) * model[i][5];
	  deltay = sin(yaw) * model[i][4] + cos(yaw) * model[i][5];
	  Item.curr_pose.x = current->curr_pose.x + deltax;
	  Item.curr_pose.y = current->curr_pose.y + deltay;
	  Item.curr_pose.yaw = yaw + model[i][6];

	  /* determine priority of leaf */
	  hp = heuristic(Item.curr_pose, state->Goal, state->man_type);
	  Item.priority = gn + 5.0*state->speed + hp;
	  /* printf("leaf %f, %f, %f: hp = %f, priority = %f\n", Item.curr_pose.x, Item.curr_pose.y, Item.curr_pose.yaw, hp, Item.priority); */
	  /* instead of 5.0, we could base this on commanded speed and turn */

	  /* need to see if it was previously on tree, or if so, if it had
	     a lower priority */
	  temptree = tree;
	  loop = 1;
	  while (temptree != NULL && loop) {
	    if (sqrt(sqr(temptree->curr_pose.x - Item.curr_pose.x) +
		     sqr(temptree->curr_pose.y - Item.curr_pose.y)) < 0.01 &&
		fabs(temptree->curr_pose.yaw - Item.curr_pose.yaw) < 0.01) {
	      loop = 0;  /* found a match - break out of loop */
	      if (temptree->priority > Item.priority)
		/* new Item priority is better for this leaf */
		addleaf = 2;  /* replace this leaf in list */
	      else
		/* Item is already in list - ignore Item */
		addleaf = 0;
	    }
	    else /* this list item doesn't match Item - keep checking */
	      temptree = temptree->next;
	  }
	  if (temptree == NULL) {
	    /* we've checked entire list, and none match Item */
	    addleaf = 1; }

	  if (addleaf == 1) {
	    addItem(leafs, Item);
	    addItem(tree, Item);
	  }
	  else if (addleaf == 2) {
	    changeItem(leafs, Item);
	    changeItem(tree, Item);
	  }
	}
      }
      /* remove state from list, find next lowest priority */
      removeLeaf(leafs, current);
      current = lowestPriority(leafs);
      /* printf("Press a key to continue...");
      scanf("%d", &foo); */
    }
  }

  if (done) {
    value = 1;
    /* now build the path - current is the goal */
    /* follow parent back through the stored tree */
    state->ManPlan = (maneuver_step *)malloc(sizeof(maneuver_step));
    state->ManPlan->next = NULL;
    state->ManPlan->desired = current->curr_pose;
    state->ManPlan->turn = current->turn;
    state->ManPlan->speed = current->speed;
    tempstep = current->parent;

    while (tempstep->parent != NULL) {
      temp = (maneuver_step *)malloc(sizeof(maneuver_step));
      temp->next = state->ManPlan;
      state->ManPlan = temp;
      state->ManPlan->desired = tempstep->curr_pose;
      state->ManPlan->turn = tempstep->turn;
      state->ManPlan->speed = tempstep->speed;
      tempstep = tempstep->parent;
    }
    /* when tempstep->parent is NULL, we're at tree root */
    /* tree root has meaningless turn=speed=0, not part of plan */

    /* now free the lists */
    temptree = leafs;
    while (leafs != NULL) {
      temptree = leafs->next;
      free(leafs);
      leafs = temptree;
    }
    temptree = tree;
    while (tree != NULL) {
      temptree = tree->next;
      free(tree);
      tree = temptree;
    }

    /* set step for starting maneuver plan */
    state->step = 1;
  }
  else {
    printf("Error: no plan found\n");
    value = 0;
    /* this should never happen */
  }

  return(value);
}

/*
void enterWorkspace(maneuver_state* state, workspace) {
   convert workspace into possible goal states

} */


int maneuver(maneuver_state_type* state, plan_command *command, int verbose) {
  /* this produces the plan, step by step - called continually 
     until plan is completely followed */
  int complete = 0;
  maneuver_step *temp;

  if (state->ManPlan == NULL) 
    /* we've reached the end of the plan */
    complete = 1;
  else {
    /* return the next value of the plan */
    command->turn = state->ManPlan->turn;
    command->speed = state->ManPlan->speed;
    if (verbose)
      printf("turning at %.3f, speed %.2f\n", command->turn,
	     command->speed);

    /* compare the current value with what we expected 
    if (OK) { */
    /* do this step for 5 seconds, then move to next */
    if (state->step > 4) {
      temp = state->ManPlan;
      state->ManPlan = temp->next;
      free(temp);
      state->step = 1;
    }
    else
      state->step = state->step + 1;
    /* } 
    else {
       stop the robot, and replan or fail 
    } */

  }

  return(complete);
}
